% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GIRFandGFEVD.R, R/plotMethods.R,
%   R/printMethods.R
\name{GIRF}
\alias{GIRF}
\alias{plot.girf}
\alias{print.girf}
\title{Estimate generalized impulse response function for
 structural STVAR models.}
\usage{
GIRF(
  stvar,
  which_shocks,
  shock_size = 1,
  N = 30,
  R1 = 250,
  R2 = 250,
  init_regime = 1,
  init_values = NULL,
  which_cumulative = numeric(0),
  scale = NULL,
  scale_type = c("instant", "peak"),
  scale_horizon = N,
  ci = c(0.95, 0.8),
  ncores = 2,
  burn_in = 1000,
  exo_weights = NULL,
  seeds = NULL,
  use_parallel = TRUE
)

\method{plot}{girf}(x, margs, ...)

\method{print}{girf}(x, ..., digits = 2, N_to_print)
}
\arguments{
\item{stvar}{an object of class \code{'stvar'}, created by, e.g., \code{fitSTVAR} or \code{fitSSTVAR}.}

\item{which_shocks}{a numeric vector of length at most \eqn{d}
(\code{=ncol(data)}) and elements in \eqn{1,...,d} specifying the
structural shocks for which the GIRF should be estimated.}

\item{shock_size}{a non-zero scalar value specifying the common size for all scalar
components of the structural shock. Note that the conditional covariance
matrix of the structural shock is normalized to an identity matrix and that the
(generalized) impulse responses may not be symmetric with respect to the sign
and size of the shock.}

\item{N}{a positive integer specifying the horizon how far ahead should the
generalized impulse responses be calculated.}

\item{R1}{the number of repetitions used to estimate GIRF for each initial
value.}

\item{R2}{the number of initial values to use, i.e., to draw from \code{init_regime}
if \code{init_values} are not specified. The confidence bounds
will be sample quantiles of the GIRFs based on different initial values.
Ignored if the argument \code{init_value} is specified.}

\item{init_regime}{an integer in \eqn{1,...,M} specifying the regime from which
the initial values should be generated from. The initial values will be generated
from the stationary distribution of the specific regime. Due to the lack of
knowledge of the stationary distribution, models with other than Gaussian conditional distribution
uses a simulation procedure with a burn-in period. See the details section.}

\item{init_values}{a size \code{[p, d, R2]} array specifying the initial values in each slice
for each Monte Carlo repetition, where d is the number of time series in the system and \code{R2}
is an argument of this function. In each slice, the \strong{last} row will be used as initial values
for the first lag, the second last row for second lag etc. If not specified, initial values will be
drawn from the regime specified in \code{init_regimes}.}

\item{which_cumulative}{a numeric vector with values in \eqn{1,...,d}
(\code{d=ncol(data)}) specifying which the variables for which the impulse
responses should be cumulative. Default is none.}

\item{scale}{should the GIRFs to some of the shocks be scaled so that they
correspond to a specific magnitude of instantaneous or peak response
of some specific variable (see the argument \code{scale_type})?
Provide a length three vector where the shock of interest
is given in the first element (an integer in \eqn{1,...,d}), the variable of
interest is given in the second element (an integer in \eqn{1,...,d}), and
the magnitude of its instantaneous or peak response in the third element
(a non-zero real number). If the GIRFs of multiple shocks should be scaled, provide
a matrix which has one column for each of the shocks with the columns being
the length three vectors described above.}

\item{scale_type}{If argument \code{scale} is specified, should the GIRFs be
scaled to match an instantaneous response (\code{"instant"}) or peak response
(\code{"peak"}). If \code{"peak"}, the scale is based on the largest magnitude
of peak response in absolute value. Ignored if \code{scale} is not specified.}

\item{scale_horizon}{If \code{scale_type == "peak"} what the maximum horizon up
to which peak response is expected? Scaling won't based on values after this.}

\item{ci}{a numeric vector with elements in \eqn{(0, 1)} specifying the
confidence levels of the confidence intervals.}

\item{ncores}{the number CPU cores to be used in parallel computing. Only
single core computing is supported if an initial value is specified (and
the GIRF won't thus be estimated multiple times).}

\item{burn_in}{Burn-in period for simulating initial values from a regime when \code{cond_dist!="Gaussian"}.
See the details section.}

\item{exo_weights}{if \code{weight_function="exogenous"}, provide a size
\eqn{(N+1 x M)} matrix of exogenous transition weights for the regimes: \code{[h, m]}
for the (after-the-impact) period \eqn{h-1} and regime \eqn{m} weight (\code{[1, m]}
is for the impact period). Ignored if \code{weight_function!="exogenous"}.}

\item{seeds}{a length \code{R2} vector containing the random number generator
seed for estimation of each GIRF. A single number of an initial value is
specified. or \code{NULL} for not initializing the seed. Exists for
creating reproducible results.}

\item{use_parallel}{employ parallel computing? If \code{FALSE}, does not print
anything.}

\item{x}{object of class \code{'girf'} generated by the function \code{GIRF}.}

\item{margs}{numeric vector of length four that adjusts the
\code{[bottom_marginal, left_marginal, top_marginal, right_marginal]}
as the relative sizes of the marginals to the figures of the responses.}

\item{...}{graphical parameters passed to \code{plot} method plotting the GIRFs}

\item{digits}{the number of decimals to print}

\item{N_to_print}{an integer specifying the horizon how far to print the estimates and
confidence intervals. The default is that all the values are printed.}
}
\value{
Returns a class \code{'girf'} list with the GIRFs in the first
  element (\code{$girf_res}) and the used arguments the rest. The first
  element containing the GIRFs is a list with the \eqn{m}th element
  containing the point estimates for the GIRF in \code{$point_est} (the first
  element) and confidence intervals in \code{$conf_ints} (the second
  element). The first row is for the GIRF at impact \eqn{(n=0)}, the second
  for \eqn{n=1}, the third for \eqn{n=2}, and so on.

  The element \code{$all_girfs} is a list containing results from all the individual GIRFs
  obtained from the MC repetitions. Each element is for one shock and results are in
  array of the form \code{[horizon, variables, MC-repetitions]}.
}
\description{
\code{GIRF} estimates generalized impulse response function for
 structural STVAR models.
}
\details{
The confidence bounds reflect uncertainty about the initial state (but
  not about the parameter estimates) if initial values are not
  specified. If initial values are specified, confidence intervals won't be
  estimated.

  Note that if the argument \code{scale} is used, the scaled responses of
  the transition weights might be more than one in absolute value.

  If \code{weight_function="exogenous"}, exogenous transition weights used in
  the Monte Carlo simulations for the future sample paths of the process must
  the given in the argument \code{exo_weights}. The same weights are used as
  the transition weights across the Monte Carlo repetitions.
}
\section{Functions}{
\itemize{
\item \code{plot(girf)}: plot method

\item \code{print(girf)}: print method

}}
\examples{
 \donttest{
 # These are long-running examples that use parallel computing.
 # It takes approximately 30 seconds to run all the below examples.
 # Note that larger R1 and R2 should be used for more reliable results;
 # small R1 and R2 are used here to shorten the estimation time.

 # Recursively identified logistic Student's t STVAR(p=3, M=2) model with the first
 # lag of the second variable as the switching variable:
 params32logt <- c(0.5959, 0.0447, 2.6279, 0.2897, 0.2837, 0.0504, -0.2188, 0.4008,
  0.3128, 0.0271, -0.1194, 0.1559, -0.0972, 0.0082, -0.1118, 0.2391, 0.164, -0.0363,
  -1.073, 0.6759, 3e-04, 0.0069, 0.4271, 0.0533, -0.0498, 0.0355, -0.4686, 0.0812,
   0.3368, 0.0035, 0.0325, 1.2289, -0.047, 0.1666, 1.2067, 7.2392, 11.6091)
 mod32logt <- STVAR(gdpdef, p=3, M=2, params=params32logt, weight_function="logistic",
  weightfun_pars=c(2, 1), cond_dist="Student", identification="recursive")

 # GIRF for one-standard-error positive structural shocks, N=30 steps ahead,
 # with the inital values drawn from the first regime.
 girf1 <- GIRF(mod32logt, which_shocks=1:2, shock_size=1, N=30, R1=50, R2=50,
  init_regime=2)
 print(girf1) # Print the results
 plot(girf1) # Plot the GIRFs

 # GIRF for one-standard-error positive structural shocks, N=30 steps ahead,
 # with the inital values drawn from the second regime. The responses of the
 # GDP and GDP deflator growth rates are accumulated.
 girf2 <- GIRF(mod32logt, which_shocks=1:2, which_cumulative=1:2, shock_size=1,
  N=30, R1=50, R2=50, init_regime=2)
 plot(girf2) # Plot the GIRFs

 # GIRF for two-standard-error negative structural shock - the first shock only.
 # N=50 steps ahead with the inital values drawn from the first regime. The responses
 # are scaled to correspond an instantanous increase of 0.5 of the first variable.
 girf3 <- GIRF(mod32logt, which_shocks=1, shock_size=-2, N=50, R1=50, R2=50,
  init_regime=1, scale_type="instant", scale=c(1, 1, 0.5))
 plot(girf3) # Plot the GIRFs
 }
}
\seealso{
\code{\link{GFEVD}}, \code{\link{linear_IRF}}, \code{\link{fitSSTVAR}}
 \itemize{
   \item Kilian L., Lütkepohl H. 20017. Structural Vector Autoregressive Analysis. 1st edition.
     \emph{Cambridge University Press}, Cambridge.
 }
}
