% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/algorithms.R
\name{ssr}
\alias{ssr}
\title{Fits a semi-supervised regression model}
\usage{
ssr(theFormula, L, U, regressors = list("lm", "rvmLinear"),
  regressors.params = NULL, pool.size = 10, gr = 1, maxits = 20,
  testdata = NULL, shuffle = TRUE, verbose = TRUE,
  plotmetrics = FALSE, U.y = NULL)
}
\arguments{
\item{theFormula}{a \code{\link[stats]{formula}} that specifies the response and the predictor variables.
Two formats are supported: \code{"Y ~ ."} and \code{"Y ~ var1 + var2 + ... + varn"}.}

\item{L}{a data frame that contains the initial labeled training set.}

\item{U}{a data frame that contains the unlabeled data.
If the provided data frame has the response variable as one of its columns, it will be discarded.}

\item{regressors}{a list of custom functions and/or strings naming the regression models to be used.
The strings must contain a valid name of a regression model from the 'caret' package.
The list of available regression models from the 'caret' package can be found \href{https://topepo.github.io/caret/available-models.html}{here}.
Functions must be named, e.g., \code{list(linearModel=lm)}. List names for models defined with strings are optional.
A list can contain both, strings and functions: \code{list("kknn", linearModel=lm)}.
Examples can be found in the vignettes.}

\item{regressors.params}{a list of lists that specifies the parameters for each custom function.
For 'caret' models specified as strings in \code{regressors}, parameters cannot be passed, use \code{NULL} instead.
The parameters are specified with a named list.
For example, if \code{regressors = list("lm", knn=knnreg)}, the number of nearest neighbors for knn can be set with \code{list(NULL, list(k = 7))}.}

\item{pool.size}{specifies the number of candidate elements to be sampled from the unlabeled set \code{U}.
The best candidate elements from the pool are labeled and added to the training set.
The \code{gr} parameter controls how many of the best candidates are used to augment the training set at each iteration.
This parameter has big influence in computational time since in each iteration, \code{pool.size * length(regressors)} models are trained and evaluated in order to find the best candidate data points.}

\item{gr}{an integer specifying the \emph{growth rate}, i.e., how many of the best elements from the pool are added to the training set for each base model at each iteration.}

\item{maxits}{an integer that specifies the maximum number of iterations.
The training phase will terminate either when \code{maxits} is reached or when \code{U} becomes empty.}

\item{testdata}{a data frame containing the test set to be evaluated within each iteration.
If \code{verbose = TRUE} and \code{plotmetrics = TRUE} the predictive performance of the model on the test set will be printed/plotted for each iteration.}

\item{shuffle}{a boolean specifying whether or not to shuffle the data frames rows before training the models. Defaults to \code{TRUE}.
Some models like neural networks are sensitive to row ordering. Often, you may want to shuffle before training.}

\item{verbose}{a boolean specifying whether or not to print diagnostic information to the console within each iteration.
If \code{testdata} is provided, the information includes performance on the test set such as RMSE and improvement percent with respect to the initial model before data from \code{U} was used.
Default is \code{TRUE}.}

\item{plotmetrics}{a boolean that specifies if performance metrics should be plotted for each iteration when \code{testdata} is provided. Default is \code{FALSE}.}

\item{U.y}{an optional numeric vector with the true values fo the response variable for the unlabeled set \code{U}.
If this parameter is \code{!= NULL} then, the true values will be used to determine the best candidates to augment the training set
and the true values will be kept when adding them to the training set.
\emph{This parameter should be used with caution} and is intended to be used to generate an upper bound model for comparison purposes only.
This is to simulate the case when the model can label the unlabeled data points used to augment the training set with 100\% accuracy.}
}
\value{
A list object of class "ssr" containing:

\bold{models} A list of the final trained models in the last iteration.

\bold{formula} The formula provided by the user in \code{theFormula}.

\bold{regressors} The list of initial \code{regressors} set by the user with formatted names.

\bold{regressors.names} The names of the regressors \code{names(regressors)}.

\bold{regressors.params} The initial list of parameters provided by the user.

\bold{pool.size} The initial \code{pool.size} specified by the user.

\bold{gr} The initial \code{gr} specified by the user.

\bold{testdata} A boolean indicating if test data was provided by the user: \code{!is.null(testdata)}.

\bold{U.y} A boolean indicating if \code{U.y} was provided by the user: \code{!is.null(U.y)}.

\bold{numits} The total number of iterations performed by the algorithm.

\bold{shuffle} The initial \code{shuffle} value specified by the user.

\bold{valuesRMSE} A numeric vector with the Root Mean Squared error on the \code{testdata} for each iteration.
The length is the number of iterations + 1.
The first position \code{valuesRMSE[1]} stores the initial RMSE before using any data from \code{U}.

\bold{valuesRMSE.all} A numeric matrix with \emph{n} rows and \emph{m} columns.
Stores Root Mean Squared Errors of the individual regression models.
The number of rows is equal to the number of iterations + 1 and the number of columns is equal to the number of regressors.
A column represents a regressor in the same order as they were provided in \code{regressors}.
Each row stores the RMSE for each iteration and for each regression model.
The first row stores the initial RMSE before using any data from \code{U}.

\bold{valuesMAE} Stores Mean Absolute Error information. Equivalent to \bold{valuesRMSE}.

\bold{valuesMAE.all} Stores Mean Absolute Errors of the individual regression models. Equivalent to \bold{valuesRMSE.all}

\bold{valuesCOR} Stores Pearson Correlation information. Equivalent to \bold{valuesRMSE}.

\bold{valuesCOR.all} Stores the Pearson Correlation of the individual regression models. Equivalent to \bold{valuesRMSE.all}
}
\description{
This function implements the \emph{co-training by committee} and \emph{self-learning} semi-supervised regression algorithms with a set of \emph{n} base regressor(s) specified by the user.
When only one model is present in the list of regressors, self-learning is performed.
}
\details{
The co-training by committee implementation is based on Hady et al. (2009). It consists of a set of \emph{n} base models (the committee), each, initially trained with independent bootstrap samples from the labeled training set \code{L}. The Out-of-Bag (OOB) elements are used for validation. The training set for each base model \emph{b} is augmented by selecting the most relevant elements from the unlabeled data set \code{U}. To determine the most relevant elements for each base model \emph{b}, the other models (excluding \emph{b}) label a set of data \code{pool.size} points sampled from \code{U} by taking the average of their predictions. For each newly labeled data point, the base model \emph{b} is trained with its current labeled training data plus the new data point and the error on its OOB validation data is computed. The top \code{gr} points that reduce the error the most are kept and used to augment the labeled training set of \emph{b} and removed from \code{U}.

When the \code{regressors} list contains a single model, \emph{self-learning} is performed. In this case, the base model labels its own data points as opposed to co-training by committee in which the data points for a given model are labeled by the other models.

In the original paper, Hady et al. (2009) use the same type of regressor for the base models but with different parameters to introduce diversity. The \code{ssr} function allows the user to specify any type of regressors as the base models. The regressors can be models from the 'caret' package, other packages, or custom functions. Models from other packages or custom functions need to comply with certain structure. First, the model's function used for training must have a formula as its first parameter and a parameter named \code{data} that accepts a data frame as the training set. Secondly, the \code{predict()} function must have the trained model as its first parameter. Most of the models from other libraries follow this pattern. If they do not follow this pattern, you can still use them by writing a wrapper function. To see examples of all those cases, please check the vignettes.
}
\examples{
dataset <- friedman1 # Load friedman1 dataset.

set.seed(1234)

# Split the dataset into 70\% for training and 30\% for testing.
split1 <- split_train_test(dataset, pctTrain = 70)

# Choose 5\% of the train set as the labeled set L and the remaining will be the unlabeled set U.
split2 <- split_train_test(split1$trainset, pctTrain = 5)

L <- split2$trainset

U <- split2$testset[, -11] # Remove the labels.

testset <- split1$testset

# Define list of regressors. Here, only one regressor (KNN). This trains a self-learning model.
# For co-training by committee, add more regressors to the list. See the vignettes for examples.
regressors <- list(knn = caret::knnreg)

# Fit the model.
model <- ssr("Ytrue ~ .", L, U, regressors = regressors, testdata = testset)

# Plot RMSE.
plot(model)

# Get the predictions on the testset.
predictions <- predict(model, testset)

# Calculate RMSE on the test set.
sqrt(mean((predictions - testset$Ytrue)^2))
}
\references{
Hady, M. F. A., Schwenker, F., & Palm, G. (2009). Semi-supervised Learning for Regression with Co-training by Committee. In International Conference on Artificial Neural Networks (pp. 121-130). Springer, Berlin, Heidelberg.
}
