% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spsurgs3sls.R
\name{spsurgs3sls}
\alias{spsurgs3sls}
\title{General Spatial 3SLS for systems of spatial equations.}
\usage{
spsurgs3sls(formula = NULL, data = NULL, na.action,
                  listw = NULL, zero.policy = NULL, 
                  type = "slm", Durbin = FALSE,
                  endog = NULL, instruments = NULL,
                  lag.instr = FALSE, initial.value = 0.2, 
                  het = FALSE, trace = TRUE)
}
\arguments{
\item{formula}{An object type \code{\link[Formula]{Formula}} 
similar to objects created with the package \pkg{Formula} 
describing the equations to be estimated in the model. 
This model may contain several responses (explained 
variables) and a varying number of regressors in each equation.}

\item{data}{An object of class data.frame or a matrix.}

\item{na.action}{A function (default \code{options("na.action")}),
can also be \code{na.omit} or \code{na.exclude} with consequences 
 for residuals and fitted values. It may be necessary to set 
 \code{zero.policy} to \code{TRUE} because this subsetting may 
 create no-neighbour observations.}

\item{listw}{A \code{listw} object created for example by 
\code{\link[spdep]{nb2listw}} from \pkg{spatialreg} package; if 
\code{\link[spdep]{nb2listw}} not given, set to 
the same spatial weights as the \code{listw} argument. It can
also be a spatial weighting matrix of order \emph{(NxN)} instead of
a \code{listw} object. Default = \code{NULL}.}

\item{zero.policy}{Similar to the corresponding parameter of 
  \code{\link[spatialreg]{lagsarlm}} function in \pkg{spatialreg} package. 
(NxN) instead of a listw object. Default = NULL.}

\item{type}{Type of spatial model specification: "sim", "iv",
"slm", "sem", "sdm" or "sarar" . Default = "slm".}

\item{Durbin}{If a formula object and model is type "sdm"
the subset of explanatory variables to lag for each equation.}

\item{endog}{Additional endogenous variables. Default \emph{NULL}. 
If not \emph{NULL} should be specified as a 
\code{\link[Formula]{Formula}} with no dependent variable. 
Examples: ~ x1 | x2 (x1 endogeous regressor for the first 
equation and x2 endogeneous regressor for the second equation) 
or ~ x1 | . (x1 endogenous regressor for the first equation and 
none endogenous regressors for the second equation)}

\item{instruments}{external instruments. Default \emph{NULL}. If not 
\emph{NULL} should be specified as a formula with no 
dependent variable in the same way than 
previous \emph{endog} argument.}

\item{lag.instr}{should the external instruments be spatially lagged?}

\item{initial.value}{he initial value for rho. It can be either numeric
(default is 0.2) or set to 'SAR', in which case the optimization 
will start from the estimated coefficient of a regression of 
the 2SLS residuals over their spatial lag (i.e. a spatial 
AR model)}

\item{het}{default {FALSE}: if {TRUE} uses the methods 
developed for heteroskedasticity for each equation. 
Wrapper using \code{\link[sphet]{spreg}} function.}

\item{trace}{A logical value to show intermediate results during 
the estimation process. Default = \code{TRUE}.}
}
\value{
Object of \code{spsur} class with the output of the  three-stages 
  least-squares estimation of the specified spatial model.
  A list with:
  \tabular{ll}{
    \code{call} \tab Matched call. \cr
    \code{type} \tab  Type of model specified. \cr
    \code{Durbin} \tab Value of \code{Durbin} argument. \cr
    \code{coefficients} \tab Estimated coefficients for the regressors. \cr
    \code{deltas} \tab Estimated spatial coefficients. \cr
    \code{rest.se} \tab Estimated standard errors for the estimates of 
      \emph{\eqn{\beta}} coefficients. \cr
    \code{deltas.se} \tab Estimated standard errors for the estimates of 
      the spatial coefficients. \cr
    \code{resvar} \tab Estimated covariance matrix for the estimates of 
      \emph{beta's} and spatial coefficients.\cr
    \code{R2} \tab Coefficient of determination for each equation, 
      obtained as the squared of the correlation coefficient between 
      the corresponding explained variable and its estimates. 
      \emph{spsur3sls} also shows a \emph{global} coefficient of
       determination obtained, in the same manner, for the set of 
       \emph{G} equations. \cr
    \code{Sigma} \tab Estimated covariance matrix for the residuals of the 
      \emph{G} equations. \cr
    \code{residuals} \tab Residuals of the model. \cr
    \code{df.residuals} \tab Degrees of freedom for the residuals. \cr
    \code{fitted.values} \tab Estimated values for the dependent variables. \cr
    \code{G} \tab Number of equations. \cr
    \code{N} \tab Number of cross-sections or spatial units. \cr
    \code{Tm} \tab Number of time periods. \cr
    \code{p} \tab Number of regressors by equation (including intercepts). \cr
    \code{Y} \tab If \emph{data} is \emph{NULL}, vector \emph{Y} of the explained variables of the SUR model. \cr
    \code{X} \tab If \emph{data} is \emph{NULL}, matrix \emph{X} of the regressors of the SUR model. \cr
    \code{W} \tab Spatial weighting matrix. \cr
    \code{zero.policy} \tab Logical value of \code{zero.policy} . \cr
    \code{listw_style} \tab	Style of neighborhood matrix \code{W}. \cr
 }
}
\description{
The function estimates spatial SUR models using 
general spatial three stages least squares. This is a 
system instrumental variable procedure which also include 
GMM estimation when there is spatial correlations in the errors.
The procedure allows for additional endogenous regressors in
addition to spatial lags of the dependent variable. It could 
be applied to "slm", "sdm", "sem" and "sarar" spatial models. 
Furthermore, for non-spatial models including endogenous 
regressors ("iv"), it could be used to estimate using 
instrumental variables and Feasible Generalized Least Squares.
}
\details{
\emph{spsurg3sls} generalize the \code{\link[sphet]{spreg}} function
 to multiequational spatial SUR models. The methodology to estimate
 spatial SUR models by Generalized 3SLS follows the steps outlined in
 Kelejian and Piras (pp. 304-305). The summary of the algorithm is
 the next one:
  \itemize{ 
   \item Estimate each equation by 2SLS and obtain the estimated 
    residuals \eqn{\hat{u}_j} for each equation. 
   \item If the model includes a spatial lag for the errors. 
    (that is, it is a SEM/SARAR model), apply GMM to obtain 
    the spatial parameters \eqn{\hat{\lambda}_j} for the residuals
    in each equation. In this case the \code{\link[sphet]{spreg}} 
    function is used as a wrapper for the GMM estimation. If the
    model does not include a spatial lag for the errors (that is, 
    it is a "sim", "iv", "slm" or "sdm" model), then \eqn{\hat{\lambda}_j = 0}
    \item Compute 
     \deqn{\hat{v}_j = \hat{u}_j-\hat{\lambda}_j W \hat{u}_j} 
     and the covariances 
     \deqn{\hat{\sigma}_{i,j} = N^{-1}\hat{v}_i\hat{v}_j}. 
     Build \eqn{\hat{Sigma}=\lbrace \hat{\sigma_{i,j}} \rbrace}
   \item Compute 
     \deqn{y_j^* = y_j - \hat{\lambda}_j W y_j}
    and
     \deqn{X_j^* = X_j- \hat{\lambda}_j W X_j}
    Compute 
     \deqn{ \hat{X}_j^* = H_j(H_j^T H_j)^{-1} H_j^T X_j^*}      
    where \eqn{H_j} is the matrix including all the instruments and
    the exogenous regressors for each equation. That is,
    \eqn{\hat{X}_j^*} 
    is the projection of \eqn{X_j^*} using
    the instruments matrix \eqn{H_j}. 
   \item Compute, in a multiequational way, 
    the Feasible Generalized Least Squares estimation using the 
    new variables \eqn{\hat{y}_j^*}, \eqn{\hat{X}_j^*} and 
    \eqn{\hat{Sigma}}. This is the 3sls step.
  }
}
\examples{

#### Example 1: Spatial Phillips-Curve. Anselin (1988, p. 203)
rm(list = ls()) # Clean memory
data(spc)
lwspc <- spdep::mat2listw(Wspc, style = "W")
## No endogenous regressors  
Tformula <- WAGE83 | WAGE81 ~ UN83 + NMR83 + SMSA | UN80 + NMR80 + SMSA
## Endogenous regressors and Instruments
Tformula2 <- WAGE83 | WAGE81 ~  NMR83 | NMR80 
## Endogenous regressors: UN83 , UN80
## Instrumental variable: SMSA

## A IV model with endogenous regressors only in first equation
spciv <- spsurgs3sls(formula = Tformula2, data = spc,
                      type = "iv", listw = lwspc,
                      endog = ~ UN83 | ., 
                      instruments = ~ SMSA | .)
summary(spciv)
print(spciv)
#########################################################################
## A SLM model with endogenous regressors 
spcslm <- spsurgs3sls(formula = Tformula2, data = spc,
                              endog = ~ UN83 | ., 
                              instruments = ~ SMSA |.,
                              type = "slm", 
                              listw = lwspc)
summary(spcslm)
print(spcslm)                           
impacts_spcslm <- impactspsur(spcslm, listw = lwspc, R = 1000)
summary(impacts_spcslm[[1]], zstats = TRUE, short = TRUE)
summary(impacts_spcslm[[2]], zstats = TRUE, short = TRUE)
#########################################################################
## A SDM model with endogenous regressors 
spcsdm <- spsurgs3sls(formula = Tformula2, data = spc,
                   endog = ~ UN83 | UN80, 
                   instruments = ~ SMSA | SMSA,
                   type = "sdm", listw = lwspc,
                   Durbin =  ~  NMR83 | NMR80)
summary(spcsdm)
## Durbin only in one equation
spcsdm2 <- spsurgs3sls(formula = Tformula2, data = spc,
                    endog = ~ UN83 | UN80, 
                    instruments = ~ SMSA | SMSA,
                    type = "sdm", listw = lwspc,
                    Durbin =  ~  NMR83 | .)
summary(spcsdm2)
#########################################################################
## A SEM model with endogenous regressors 
spcsem <- spsurgs3sls(formula = Tformula2, data = spc,
                      endog = ~ UN83 | UN80, 
                      instruments = ~ SMSA | SMSA,
                      type = "sem", listw = lwspc)
summary(spcsem)
print(spcsem)                           
#########################################################################
## A SARAR model with endogenous regressors 
spcsarar <- spsurgs3sls(formula = Tformula2, data = spc,
                        endog = ~ UN83 | UN80, 
                        instruments = ~ SMSA | SMSA,
                        type = "sarar", listw = lwspc)
summary(spcsarar)
print(spcsarar)                           
impacts_spcsarar <- impactspsur(spcsarar, listw = lwspc, R = 1000)
summary(impacts_spcsarar[[1]], zstats = TRUE, short = TRUE)
summary(impacts_spcsarar[[2]], zstats = TRUE, short = TRUE)
}
\references{
\itemize{
   \item Kelejian, H. H. and Piras, G. (2017). 
     \emph{ Spatial Econometrics}. Academic Press.
   \item Kelejian, H.H. and Prucha, I.R. (2010).
    Specification and Estimation of Spatial 
    Autoregressive Models with Autoregressive 
    and Heteroskedastic Disturbances. 
    \emph{Journal of Econometrics}, 157, 
    pp. 53-67.
    \item Kelejian, H.H. and Prucha, I.R. (1999). 
      A Generalized Moments Estimator for the 
      Autoregressive Parameter in a Spatial 
      Model. \emph{International Economic Review},
      40, pp. 509-533.
    \item Kelejian, H.H. and Prucha, I.R. (1998).
      A Generalized Spatial Two Stage Least 
      Square Procedure for Estimating a Spatial 
      Autoregressive Model with Autoregressive 
      Disturbances. \emph{Journal of Real 
      Estate Finance and Economics}, 17, 
      pp. 99–121.
    \item Piras, G. (2010). sphet: Spatial
      Models with Heteroskedastic Innovations 
      in R. \emph{Journal of Statistical 
      Software}, 35(1), pp. 1-21. 
      \url{https://www.jstatsoft.org/v35/i01/}. -          
  }
}
\seealso{
\code{\link[sphet]{spreg}},
\code{\link{spsur3sls}},
\code{\link[spatialreg]{stsls}},
\code{\link{spsurml}}
}
\author{
\tabular{ll}{
  Fernando López  \tab \email{fernando.lopez@upct.es} \cr
  Román Mínguez  \tab \email{roman.minguez@uclm.es} \cr
  Jesús Mur  \tab \email{jmur@unizar.es} \cr
  }
}
