% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spsur3sls.R
\name{spsur3sls}
\alias{spsur3sls}
\title{Three Stages Least Squares estimation,3sls, of spatial SUR models.}
\usage{
spsur3sls (formula = NULL, data = NULL, na.action,
                  R = NULL, b = NULL, listw = NULL, 
                  zero.policy = NULL, X= NULL, Y = NULL, G = NULL, 
                  N = NULL, Tm = NULL, p = NULL,  
                  type = "slm", Durbin = NULL, maxlagW = NULL,
                  trace = TRUE)
}
\arguments{
\item{formula}{An object type \code{\link[Formula]{Formula}} 
similar to objects created with the package \pkg{Formula} 
describing the equations to be estimated in the model. 
This model may contain several responses (explained 
variables) and a varying number of regressors in each equation.}

\item{data}{An object of class data.frame or a matrix.}

\item{na.action}{A function (default \code{options("na.action")}),
can also be \code{na.omit} or \code{na.exclude} with consequences 
 for residuals and fitted values. It may be necessary to set 
 \code{zero.policy} to \code{TRUE} because this subsetting may 
 create no-neighbour observations.}

\item{R}{A row vector of order \emph{(1xpr)} with  the set of 
\emph{r} linear constraints on the \emph{beta} parameters. The 
\emph{first} restriction appears in the first \emph{p} terms,
the second restriction in the next \emph{p} terms and so on. 
Default = \code{NULL}.}

\item{b}{A column vector of order \emph{(rx1)} with the values of 
the linear restrictions on the \emph{beta} parameters. 
Default = \code{NULL}.}

\item{listw}{A \code{listw} object created for example by 
\code{\link[spdep]{nb2listw}} from \pkg{spatialreg} package; if 
\code{\link[spdep]{nb2listw}} not given, set to 
the same spatial weights as the \code{listw} argument. It can
also be a spatial weighting matrix of order \emph{(NxN)} instead of
a \code{listw} object. Default = \code{NULL}.}

\item{zero.policy}{Similar to the corresponding parameter of 
\code{\link[spatialreg]{lagsarlm}} function in \pkg{spatialreg} package. 
If \code{TRUE} assign zero to the lagged value of zones without 
neighbours, if \code{FALSE} assign \code{NA} - causing 
\code{spsurml()} to terminate with an error. Default = \code{NULL}.}

\item{X}{A data matrix of order \emph{(NTmGxp)} with the observations
of the regressors. The number of covariates in the SUR model is 
\emph{p} = \eqn{sum(p_{g})} where \emph{\eqn{p_{g}}} is the number 
of regressors (including the intercept) in the g-th equation, 
\emph{g = 1,...,G}). The specification of "X" is only 
necessary if not available a \code{\link[Formula]{Formula}} and a 
data frame. Default = \code{NULL}.}

\item{Y}{A column vector of order \emph{(NTmGx1)}, with the 
observations of the explained variables. The ordering of the data 
must be (first) equation, (second) time dimension and (third) 
cross-sectional/spatial units. The specification of \emph{Y} is 
only necessary if not available a \code{\link[Formula]{Formula}}
and a data frame. Default = \code{NULL}.}

\item{G}{Number of equations.}

\item{N}{Number of cross-section or spatial units}

\item{Tm}{Number of time periods.}

\item{p}{Number of regressors by equation, including the intercept. 
\emph{p} can be a row vector of order \emph{(1xG)}, if the number 
 of regressors is not the same for all the equations, or a scalar, 
 if the \emph{G} equations have the same number of regressors. The 
specification of \emph{p} is only necessary if not available a 
\code{\link[Formula]{Formula}} and a data frame.}

\item{type}{Type of spatial model, restricted to cases where lags of the 
explained variable appear in the rigth hand side of the equations. There 
are two possibilities: "slm" or "sdm". Default = "slm".}

\item{Durbin}{If a formula object and model is type "sdm" the subset 
of explanatory variables to lag for each equation.}

\item{maxlagW}{Maximum spatial lag order of the regressors employed to 
produce spatial instruments for the spatial lags of the explained variables. 
Default = 2. Note that in case of \code{type}="sdm", the default value for 
\code{maxlagW} is set to 3 because the first lag of the regressors, 
\eqn{WX_{tg}}, can not be used as spatial instruments.}

\item{trace}{A logical value to show intermediate results during 
the estimation process. Default = \code{TRUE}.}
}
\value{
Object of \code{spsur} class with the output of the  three-stages 
  least-squares estimation of the specified spatial model.
  A list with:
  \tabular{ll}{
    \code{call} \tab Matched call. \cr
    \code{type} \tab  Type of model specified. \cr
    \code{Durbin} \tab Value of \code{Durbin} argument. \cr
    \code{coefficients} \tab Estimated coefficients for the regressors. \cr
    \code{deltas} \tab Estimated spatial coefficients. \cr
    \code{rest.se} \tab Estimated standard errors for the estimates of 
      \emph{\eqn{\beta}} coefficients. \cr
    \code{deltas.se} \tab Estimated standard errors for the estimates of 
      the spatial coefficients. \cr
    \code{resvar} \tab Estimated covariance matrix for the estimates of 
      \emph{beta's} and spatial coefficients.\cr
    \code{R2} \tab Coefficient of determination for each equation, 
      obtained as the squared of the correlation coefficient between 
      the corresponding explained variable and its estimates. 
      \emph{spsur3sls} also shows a \emph{global} coefficient of
       determination obtained, in the same manner, for the set of 
       \emph{G} equations. \cr
    \code{Sigma} \tab Estimated covariance matrix for the residuals of the 
      \emph{G} equations. \cr
    \code{residuals} \tab Residuals of the model. \cr
    \code{df.residuals} \tab Degrees of freedom for the residuals. \cr
    \code{fitted.values} \tab Estimated values for the dependent variables. \cr
    \code{G} \tab Number of equations. \cr
    \code{N} \tab Number of cross-sections or spatial units. \cr
    \code{Tm} \tab Number of time periods. \cr
    \code{p} \tab Number of regressors by equation (including intercepts). \cr
    \code{Y} \tab If \emph{data} is \emph{NULL}, vector \emph{Y} of the explained variables of the SUR model. \cr
    \code{X} \tab If \emph{data} is \emph{NULL}, matrix \emph{X} of the regressors of the SUR model. \cr
    \code{W} \tab Spatial weighting matrix. \cr
    \code{zero.policy} \tab Logical value of \code{zero.policy} . \cr
    \code{listw_style} \tab	Style of neighborhood matrix \code{W}. \cr
 }
}
\description{
The function estimates spatial SUR models using three stages
 least squares, where the instruments are obtained from the spatial lags
 of the \emph{X} variables, assumed to be exogenous. The number of 
 equations, time periods and spatial units is not restricted. The user can 
 choose between a Spatial Durbin Model or a Spatial Lag Model, as described 
 below. The estimation procedure allows for the introduction of linear 
 restrictions on the \eqn{\beta} parameters associated to the regressors.
}
\details{
\emph{spsur3sls} can be used to estimate two groups of spatial models:
  \itemize{
    \item "slm": SUR model with spatial lags of the endogenous in 
    the right hand side of the equations
      \deqn{y_{tg} = \rho_{g} Wy_{tg} + X_{tg} \beta_{g} + \epsilon_{tg} }
    \item "sdm": SUR model of the Spatial Durbin type
      \deqn{ y_{tg} = \rho_{g} Wy_{tg} + X_{tg} \beta_{g} +
             WX_{tg} \theta_{g} + \epsilon_{tg} }
             }

  where \eqn{y_{tg}} and \eqn{\epsilon_{tg}} are \emph{(Nx1)} vectors,
  corresponding to the g-th equation and time period t; \eqn{X_{tg}} is 
  the matrix of regressors, of order \emph{(Nxp_{g})}. Moreover, 
  \eqn{\rho_{g}} is a spatial coefficient and \emph{W} is a 
  \emph{(NxN)} spatial weighting matrix.

 By default, the input of this function is an object created with 
 \code{\link[Formula]{Formula}} and a data frame. However, 
 \emph{spsur3sls} also allows for the direct specification of vector
 \emph{Y} and matrix \emph{X}, with the explained variables and  regressors 
 respectively, as inputs (these terms may be the result, for example, 
 of \code{\link{dgp_spsur}}). \cr


 \code{spsur3sls} is a Least-Squares procedure in three-stages designed 
 to circumvent the endogeneity problems due to the presence of spatial 
 lags of the explained variable in the right hand side of the equations 
 do the SUR. The instruments are produced internally by \code{spsur3sls} 
 using a sequence of spatial lags of the \emph{X} variables, which are 
 assumed to be exogenous. The user must define the number of (spatial) 
 instruments to be used in the procedure, through the argument 
 \code{maxlagW} (i.e. maxlagW = 3). Then, the collection of instruments 
 generated is \eqn{[WX_{tg}; W*WX_{tg}; W*W*WX_{tg}]}. In the case of 
 a \emph{SDM}, the first lag of the \emph{X} matrix already is in the 
 equation and cannot be used as instrument. In the example above, the 
 list of instruments for a \emph{SDM} model would be 
 \eqn{[W^{2}X_{tg}; W^{3}X_{tg}]}.

  The \emph{first} stage of the procedure consists in the least squares 
  of the \emph{Y} variables on the set of instruments. From this 
  estimation, the procedure retains the estimates of \emph{Y}
  in the so-called \emph{Yls} variables. In the \emph{second} stage, 
  the \emph{Y} variables that appear in the right hand side of the equation 
  are substituted by \emph{Yls} and the SUR model
  is estimated by Least Squares. The \emph{third} stage improves the 
  estimates of the second stage through a Feasible Generalized Least Squares 
  estimation of the parameters of the model,
  using the residuals of the \emph{second} stage to estimate the 
  \emph{Sigma} matrix.


 The arguments \emph{R} and \emph{b} allows to introduce linear 
 restrictions on the \emph{beta} coefficients of the \emph{G} equations. 
 \code{\link{spsur3sls}}, first, introduces the linear restrictions in 
 the SUR model and builds, internally, the corresponding constrained
 SUR model. Then, the function estimates the restricted model which is 
 shown in the output. The function does not compute the unconstrained 
 model nor test for the linear restrictions. The user may ask for the 
 unconstrained estimation using another \code{\link{spsurml}}
 estimation. Moreover, the function \code{\link{wald_betas}} obtains 
 the Wald test of a set of linear restrictions for an object created 
 previously by \code{\link{spsurml}} or \code{\link{spsur3sls}}.
}
\examples{

#################################################
######## CLASSIC PANEL DATA (G=1; Tm>1)  ########
#################################################

#### Example 1: Spatial Phillips-Curve. Anselin (1988, p. 203)
## A SUR model without spatial effects
rm(list = ls()) # Clean memory
data(spc)
lwspc <- spdep::mat2listw(Wspc, style = "W")
Tformula <- WAGE83 | WAGE81 ~ UN83 + NMR83 + SMSA | UN80 + NMR80 + SMSA

## A SUR-SLM model (3SLS Estimation)
spcsur_slm_3sls <-spsur3sls(formula = Tformula, data = spc,
                            type = "slm", listw = lwspc)
summary(spcsur_slm_3sls)
print(spcsur_slm_3sls)

if (require(gridExtra)) {
  pl <- plot(spcsur_slm_3sls, viewplot = FALSE) 
  grid.arrange(pl$lplbetas[[1]], pl$lplbetas[[2]], 
               pl$pldeltas, nrow = 3)
}

## VIP: The output of the whole set of the examples can be examined 
## by executing demo(demo_spsur3sls, package="spsur")

\donttest{
## A SUR-SDM model (3SLS Estimation)
spcsur_sdm_3sls <- spsur3sls(formula = Tformula, data = spc,
                             type = "sdm", listw = lwspc)
summary(spcsur_sdm_3sls)
if (require(gridExtra)) {
  pl <- plot(spcsur_sdm_3sls, viewplot = FALSE) 
  grid.arrange(pl$lplbetas[[1]], pl$lplbetas[[2]], 
               pl$pldeltas, nrow = 3)
}
rm(spcsur_sdm_3sls)

## A SUR-SDM model with different spatial lags in each equation
 TformulaD <-  ~ UN83 + NMR83 + SMSA | UN80 + NMR80  
 spcsur_sdm2_3sls <-spsur3sls(formula = Tformula, data = spc,
                             type = "sdm", listw = lwspc,
                             Durbin = TformulaD)
 summary(spcsur_sdm2_3sls)
if (require(gridExtra)) {
  pl <- plot(spcsur_sdm2_3sls, viewplot = FALSE) 
  grid.arrange(pl$lplbetas[[1]], pl$lplbetas[[2]], 
               pl$pldeltas, nrow = 3)
}
rm(spcsur_sdm2_3sls)
}

#################################################
###  MULTI-DIMENSIONAL PANEL DATA (G>1; Tm>1) ###
#################################################

#### Example 3: Homicides + Socio-Economics (1960-90)
# Homicides and selected socio-economic characteristics for continental
# U.S. counties.
# Data for four decennial census years: 1960, 1970, 1980 and 1990.
# https://geodacenter.github.io/data-and-lab/ncovr/

\donttest{
rm(list = ls()) # Clean memory
data(NCOVR, package = "spsur")
nbncovr <- spdep::poly2nb(NCOVR.sf, queen = TRUE)
## Some regions with no links...
lwncovr <- spdep::nb2listw(nbncovr, style = "W", zero.policy = TRUE)
Tformula <- HR80  | HR90 ~ PS80 + UE80 | PS90 + UE90
## A SUR-SLM model
NCOVRSUR_slm_3sls <- spsur3sls(formula = Tformula, data = NCOVR.sf, 
                               type = "slm", zero.policy = TRUE,
                               listw = lwncovr, trace = FALSE)
summary(NCOVRSUR_slm_3sls)
if (require(gridExtra)) {
  pl <- plot(NCOVRSUR_slm_3sls, viewplot = FALSE) 
  grid.arrange(pl$lplbetas[[1]], pl$lplbetas[[2]], 
               pl$pldeltas, nrow = 3)
}
rm(NCOVRSUR_slm_3sls)
}
}
\references{
\itemize{
      \item Lopez, F. A., Minguez, R., Mur, J. (2020). ML versus IV estimates 
      of spatial SUR models: evidence from the case of Airbnb in Madrid urban 
      area. \emph{The Annals of Regional Science}, 64(2), 313-347.
      <doi:10.1007/s00168-019-00914-1>
      \item Anselin, L. (2016) Estimation and Testing in the Spatial Seemingly 
      Unrelated Regression (SUR). \emph{Geoda Center for Geospatial Analysis 
      and Computation, Arizona State University}. Working Paper 2016-01.
      <doi:10.13140/RG.2.2.15925.40163>
      \item, Anselin, L. (1988). \emph{Spatial Econometrics: Methods and Models}. 
      Kluwer Academic Publishers, Dordrecht, The Netherlands (p. 146).
      \item Anselin, L., Le Gallo, J., Hubert J. (2008) Spatial Panel Econometrics. 
      In \emph{The econometrics of panel data. 
      Fundamentals and recent developments in theory and practice}. (Chap 19, p. 653)
  }
}
\seealso{
\code{\link{spsurml}}, \code{\link[spatialreg]{stsls}}, 
\code{\link{wald_betas}}
}
\author{
\tabular{ll}{
  Fernando Lopez  \tab \email{fernando.lopez@upct.es} \cr
  Roman Minguez  \tab \email{roman.minguez@uclm.es} \cr
  Jesus Mur  \tab \email{jmur@unizar.es} \cr
  }
}
