#' Group Lines
#'
#' \code{group_lines} groups rows into spatial groups by creating trajectories
#' and grouping based on spatial overlap. The function accepts a
#' \code{data.table} with relocation data, individual identifiers and a
#' \code{threshold}. The relocation data is transformed into \code{SpatialLines}
#' and overlapping \code{SpatialLines} are grouped. The \code{threshold}
#' argument is used to specify the criteria for distance between lines.
#' Relocation data should be in two columns representing the X and Y
#' coordinates.
#'
#' The \code{DT} must be a \code{data.table}. If your data is a
#' \code{data.frame}, you can convert it by reference using
#' \code{\link[data.table:setDT]{data.table::setDT}}.
#'
#' The \code{id}, \code{coords}, \code{sortBy} (and optional \code{timegroup}
#' and \code{splitBy}) arguments expect the names of respective columns in
#' \code{DT} which correspond to the individual identifier, X and Y coordinates,
#' sorting, timegroup (generated by \code{group_times}) and additional grouping
#' columns.
#'
#' The \code{projection} argument expects a character string defining
#' the EPSG code. For example, for UTM zone 36N (EPSG 32736), the projection
#' argument is "EPSG:32736". See \url{https://spatialreference.org}
#' for a list of EPSG codes. Please note, R spatial has followed updates
#' to GDAL and PROJ for handling projections, see more at
#' \url{https://www.r-spatial.org/r/2020/03/17/wkt.html}. It is likely
#' that \code{build_polys} will return "Warning in proj4string(xy) :
#' CRS object has comment, which is lost in output" due to these changes.
#'
#' The \code{sortBy} is used to order the input \code{data.table} when creating
#' \code{SpatialLines}. It must a \code{POSIXct} to ensure the rows are sorted
#' by date time.
#'
#' The \code{threshold} must be provided in the units of the coordinates. The
#' \code{threshold} can be equal to 0 if strict overlap is required, else it
#' needs to be greater than 0. The coordinates must be planar coordinates (e.g.:
#' UTM). In the case of UTM, a \code{threshold} = 50 would indicate a 50m
#' distance threshold.
#'
#' The \code{timegroup} argument is optional, but recommended to pair with
#' \code{\link{group_times}}. The intended framework is to group rows temporally
#' with \code{\link{group_times}} then spatially with \code{group_lines} (or
#' \code{\link{group_pts}}, \code{\link{group_polys}}). With \code{group_lines},
#' pick a relevant \code{group_times} \code{threshold} such as \code{'1 day'} or
#' \code{'7 days'} which is informed by your study species and system.
#'
#' The \code{splitBy} argument offers further control over grouping. If within
#' your \code{DT}, you have multiple populations, subgroups or other distinct
#' parts, you can provide the name of the column which identifies them to
#' \code{splitBy}. The grouping performed by \code{group_lines} will only
#' consider rows within each \code{splitBy} subgroup.
#'
#' @return \code{group_lines} returns the input \code{DT} appended with a
#'   \code{group} column.
#'
#'   This column represents the spatial (and if \code{timegroup} was provided -
#'   spatiotemporal) group calculated by overlapping lines. As with the other
#'   grouping functions,  the actual value of group is arbitrary and represents
#'   the identity of a given group where 1 or more individuals are assigned to a
#'   group. If the data was reordered, the group may change, but the contents of
#'   each group would not.
#'
#'   A message is returned when a column named \code{group} already exists in
#'   the input \code{DT}, because it will be overwritten.
#'
#'
#' @inheritParams group_pts
#' @inheritParams build_polys
#' @param threshold The width of the buffer around the lines in the units of the
#'   projection. Supply 0 to compare intersection without buffering.
#' @param spLines Alternatively to providing a DT, provide a SpatialLines object
#'   created with the sp package. If a spLines object is provided, groups cannot
#'   be calculated by a    timegroup or splitBy.
#' @param sortBy Character string of date time column(s) to sort rows by. Must
#'   be a POSIXct.
#'
#' @export
#'
#' @family Spatial grouping
#' @seealso \code{\link{build_lines}} \code{\link{group_times}}
#'
#' @examples
#' # Load data.table
#' library(data.table)
#'
#' # Read example data
#' DT <- fread(system.file("extdata", "DT.csv", package = "spatsoc"))
#'
#' # Subset only individuals A, B, and C
#' DT <- DT[ID %in% c('A', 'B', 'C')]
#'
#' # Cast the character column to POSIXct
#' DT[, datetime := as.POSIXct(datetime, tz = 'UTC')]
#'
#' # EPSG code for example data
#' utm <- 'EPSG:32736'
#'
#' \donttest{group_lines(DT, threshold = 50, projection = utm, sortBy = 'datetime',
#'             id = 'ID', coords = c('X', 'Y'))}
#'
#' ## Daily movement tracks
#' # Temporal grouping
#' group_times(DT, datetime = 'datetime', threshold = '1 day')
#'
#' # Subset only first 50 days
#' DT <- DT[timegroup < 25]
#'
#' # Spatial grouping
#' group_lines(DT, threshold = 50, projection = utm,
#'             id = 'ID', coords = c('X', 'Y'),
#'             timegroup = 'timegroup', sortBy = 'datetime')
#'
#' ## Daily movement tracks by population
#' group_lines(DT, threshold = 50, projection = utm,
#'             id = 'ID', coords = c('X', 'Y'),
#'             timegroup = 'timegroup', sortBy = 'datetime',
#'             splitBy = 'population')

group_lines <-
  function(DT = NULL,
           threshold = NULL,
           projection = NULL,
           id = NULL,
           coords = NULL,
           timegroup = NULL,
           sortBy = NULL,
           splitBy = NULL,
           spLines = NULL) {

    # due to NSE notes in R CMD check
    group <- ..coords <- ..id <- ..sortBy <- withinGroup <- NULL

    if (is.null(threshold)) {
      message('threshold missing, using 0 by default')
      threshold <- 0
    } else if (!is.numeric(threshold)) {
      stop('threshold must be numeric')
    } else if (threshold < 0) {
      stop('cannot provide a negative threshold')
    }

    if (!is.null(spLines) && !is.null(DT)) {
      stop('cannot provide both DT and spLines')
    } else if (is.null(spLines) && is.null(DT)) {
      stop('must provide either DT or spLines')
    } else if (is.null(spLines) && !is.null(DT)) {
      if (is.null(projection)) {
        stop('projection must be provided when DT is')
      }

      if (is.null(coords)) {
        stop('coords must be provided')
      }

      if (is.null(id)) {
        stop('id must be provided')
      }

      if (is.null(sortBy)) {
        stop('sortBy must be provided')
      }

      if (any(!(c(id, coords, timegroup, sortBy) %in% colnames(DT)))) {
        stop(paste0(
          as.character(paste(setdiff(
            c(id, coords), colnames(DT)
          ),
          collapse = ', ')),
          ' field(s) provided are not present in input DT'
        ))
      }

      if ('group' %in% colnames(DT)) {
        message('group column will be overwritten by this function')
        set(DT, j = 'group', value = NULL)
      }
    } else if (!is.null(spLines) && is.null(DT)) {
      if (!('SpatialLines' %in% class(spLines) && isS4(spLines))) {
        stop('spLines provided must be a SpatialLines object')
      }

      if (threshold == 0) {
        inter <- rgeos::gIntersects(spLines, spLines, byid = TRUE)
      } else {
        buffered <- rgeos::gBuffer(spLines, width = threshold,
                                   byid = TRUE)
        inter <- rgeos::gIntersects(spLines, buffered, byid = TRUE)
      }
      g <- igraph::graph_from_adjacency_matrix(inter)
      ovr <- igraph::clusters(g)$membership
      out <- data.table::data.table(names(ovr),
                                      unlist(ovr))
      data.table::setnames(out, c('ID', 'group'))
      return(out[])
    }

    if (is.null(timegroup)) {
      withCallingHandlers({
        spLines <- build_lines(
          DT = DT,
          projection = projection,
          coords = coords,
          id = id,
          sortBy = sortBy
        )},
        warning = function(w){
          if (startsWith(conditionMessage(w), 'some rows dropped')) {
            invokeRestart('muffleWarning')
          }
        }
      )
      if (!is.null(spLines)) {
        if (threshold == 0) {
          inter <- rgeos::gIntersects(spLines, spLines, byid = TRUE)
        } else {
          buffered <- rgeos::gBuffer(spLines, width = threshold, byid = TRUE)
          inter <- rgeos::gIntersects(spLines, buffered, byid = TRUE)
        }
        g <- igraph::graph_from_adjacency_matrix(inter)
        ovr <- igraph::clusters(g)$membership
        ovrDT <- data.table::data.table(ID = names(ovr),
                                        group = unlist(ovr))
      } else {
        ovrDT <- data.table::data.table(ID = DT[[id]], group = as.integer(NA))
      }

      data.table::setnames(ovrDT, c(id, 'group'))
      DT[ovrDT, group := group, on = id]
      if (DT[is.na(group), .N] > 0) {
        warning(
          strwrap(
            prefix = " ",
            initial = "",
            x = 'some rows were dropped,
            cannot build a line with < 2 points.
            in this case, group set to NA.'
          )
        )
      }
      return(DT[])
    } else {

      if (is.null(splitBy)) {
        splitBy <- timegroup
      }
      else {
        splitBy <- c(splitBy, timegroup)
      }
      ovrDT <-
        DT[, {
          withCallingHandlers({
            spLines <- build_lines(
              DT = .SD,
              projection = projection,
              coords = ..coords,
              id = ..id,
              sortBy = ..sortBy
            )},
            warning = function(w){
              if (startsWith(conditionMessage(w), 'some rows dropped')) {
                invokeRestart('muffleWarning')
              }
            }
          )
          if (!is.null(spLines)) {
            if (threshold == 0) {
              inter <- rgeos::gIntersects(spLines, spLines, byid = TRUE)
            } else {
              buffered <- rgeos::gBuffer(spLines, width = threshold,
                                         byid = TRUE)
              inter <- rgeos::gIntersects(spLines, buffered, byid = TRUE)

            }
            g <- igraph::graph_from_adjacency_matrix(inter)
            ovr <- igraph::clusters(g)$membership
            out <- data.table::data.table(names(ovr),
                                          unlist(ovr))
            data.table::setnames(out, c(..id, 'withinGroup'))
          } else {
            out <- data.table(get(..id), withinGroup = as.double(NA))
            data.table::setnames(out, c(..id, 'withinGroup'))

          }
        }, by = splitBy, .SDcols = c(coords, id, sortBy)]

      DT[ovrDT, withinGroup := withinGroup, on = c(id, splitBy)]
      DT[, group := ifelse(is.na(withinGroup), as.integer(NA), .GRP),
         by = c(splitBy, 'withinGroup')]
      set(DT, j = 'withinGroup', value = NULL)
      if (DT[is.na(group), .N] > 0) {
        warning(
          strwrap(
            prefix = " ",
            initial = "",
            x = 'some rows were dropped,
            cannot build a line with < 2 points.
            in this case, group set to NA.'
          )
        )
      }
      return(DT[])
    }
  }
