#' Generates spartan-compatible timepoint files if simulation results over
#' time are in one file
#'
#' @param FILEPATH Filepath to all LHC results being analysed
#' @param SPARTAN_PARAMETER_FILE CSV file generated by spartan, containing
#' all parameter sets under which the simulation was run
#' @param RUN_SUMMARY_FILE_NAME Simulation output file, containing multiple
#' timepoints
#' @param NUMSAMPLES Number of parameter samples generated using the hypercube
#' @param NUMRUNSPERSAMPLE Number of replicate runs performed for each sample
#' @param TIMEPOINTS Simulation timepoints to extract from the result file
#'
#' @export
lhc_generateTimepointFiles <- function(FILEPATH, SPARTAN_PARAMETER_FILE,
                                       RUN_SUMMARY_FILE_NAME, NUMSAMPLES,
                                       NUMRUNSPERSAMPLE, TIMEPOINTS) {
  # USED WHEN A SIMULATION IS BEING PROCESSED WHERE THE RESULTS OF EACH
  # TIMEPOINT ARE IN THE SAME FILE (AS IN LEISHSIM)
  ## RATHER THAN REMAKE SPARTAN TO PROCESS THESE, WE INSTEAD MAKE INDIVIDUAL
  # TIMEPOINT FILES FOR THIS DATA  SO THE FILES ARE COMPATIBLE WITH SPARTAN

  if (file.exists(FILEPATH)) {
    print("Generating Simulation Timestep files compatible for
          Spartan Analysis")

    # READ IN THE SPARTAN PARAMETER FILE
    LHCTABLE <- read.csv(paste(FILEPATH, "/", SPARTAN_PARAMETER_FILE,
                               sep = ""), header = TRUE)

    # NOW FOR LEISHSIM WE NEED TO DO 168 ROWS, SO:
    for (t in 1:length(TIMEPOINTS)) {
      TIMEPOINTPROCESSING <- TIMEPOINTS[t]
      print(paste("Processing Timepoint: ", TIMEPOINTPROCESSING, sep = ""))
      ALL_SIM_RESULTS_FOR_TIMEPOINT <- NULL

      for (k in 1:NUMSAMPLES) {
        # GET THE MEDIAN RESULTS FILE, CONTAINING THE MEDIAN OF EACH TIMEPOINT
        MEDIAN_RESULTS <- read.csv(paste(FILEPATH, "/", k, "/",
                                         RUN_SUMMARY_FILE_NAME, sep = ""),
                                   header = TRUE)

        RESULT <- cbind(LHCTABLE[k, ], MEDIAN_RESULTS[TIMEPOINTPROCESSING, ])

        # NOW ADD THIS TO THE LIST OF ALL MEDIANS BEING PROCESSED IN THIS
        # ANALYSIS
        ALL_SIM_RESULTS_FOR_TIMEPOINT <- rbind(ALL_SIM_RESULTS_FOR_TIMEPOINT,
                                               RESULT)
      }

      # WRITE THE RESULTS FILE FOR THIS TIMEPOINT
      if (!is.null(ALL_SIM_RESULTS_FOR_TIMEPOINT)) {
        RESULTSFILE <- paste(FILEPATH, "/Results_Time_", TIMEPOINTPROCESSING,
                             ".csv", sep = "")
        write.csv(ALL_SIM_RESULTS_FOR_TIMEPOINT,
                  RESULTSFILE, quote = FALSE, row.names = FALSE)
      }
    }
  }
}
