% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spattemp.slice.R
\name{spattemp.slice}
\alias{spattemp.slice}
\title{Slicing a spatiotemporal object}
\usage{
spattemp.slice(stob, tt, checkargs = TRUE)
}
\arguments{
\item{stob}{An object of class \code{\link{stden}} or \code{\link{rrst}} giving the spatiotemporal
estimate from which to take slices.}

\item{tt}{Desired time(s); the density/risk surface estimate
corresponding to which will be returned. This value \bold{must} be in the
available range provided by \code{stob$tlim}; see `Details'.}

\item{checkargs}{Logical value indicating whether to check validity of
\code{stob} and \code{tt}. Disable only if you know this check will be
unnecessary.}
}
\value{
A list of lists of pixel \code{\link[spatstat]{im}}ages, each of which corresponds to
 the requested times in \code{tt}, and are named as such.\cr
 If \code{stob} is an object of class \code{\link{stden}}:
 
 \item{z}{
 Pixel images of the joint spatiotemporal density corresponding to \code{tt}.
 }
 
 \item{z.cond}{
 Pixel images of the conditional spatiotemporal density given each time in \code{tt}.
 }
 
 If \code{stob} is an object of class \code{\link{rrst}}:
 
 \item{rr}{
 Pixel images of the joint spatiotemporal relative risk corresponding to \code{tt}.
 }

 \item{rr.cond}{
 Pixel images of the conditional spatiotemporal relative risk given each time in \code{tt}.
 }

 \item{P}{
 Only present if \code{tolerate = TRUE} in the preceding call to \code{\link{spattemp.risk}}.
 Pixel images of the \eqn{p}-value surfaces for the joint spatiotemporal relative risk.
 }
 
 \item{P.cond}{
 Only present if \code{tolerate = TRUE} in the preceding call to \code{\link{spattemp.risk}}.
 Pixel images of the \eqn{p}-value surfaces for the conditional spatiotemporal relative risk.
 }
}
\description{
Takes slices of the spatiotemporal kernel density or 
relative risk function estimate at desired times
}
\details{
Contents of the \code{stob} argument are returned based on a discretised set of times.
This function internally computes the desired surfaces as
pixel-by-pixel linear interpolations using the two discretised times
that bound each requested \code{tt}.

The function returns an error if any of the
requested slices at \code{tt} are not within the available range of
times as given by the \code{tlim}
component of \code{stob}.
}
\examples{

\dontrun{
data(fmd)
fmdcas <- fmd$cases
fmdcon <- fmd$controls

f <- spattemp.density(fmdcas,h=6,lambda=8)
g <- bivariate.density(fmdcon,h0=6)
rho <- spattemp.risk(f,g,tolerate=TRUE) 

f$tlim # requested slices must be in this range

# slicing 'stden' object
f.slice1 <- spattemp.slice(f,tt=50) # evaluation timestamp
f.slice2 <- spattemp.slice(f,tt=150.5) # interpolated timestamp
par(mfrow=c(2,2))
plot(f.slice1$z$'50')
plot(f.slice1$z.cond$'50')
plot(f.slice2$z$'150.5')
plot(f.slice2$z.cond$'150.5')

# slicing 'rrst' object
rho.slices <- spattemp.slice(rho,tt=c(50,150.5))
par(mfrow=c(2,2))
plot(rho.slices$rr$'50');tol.contour(rho.slices$P$'50',levels=0.05,add=TRUE)
plot(rho.slices$rr$'150.5');tol.contour(rho.slices$P$'150.5',levels=0.05,add=TRUE)
plot(rho.slices$rr.cond$'50');tol.contour(rho.slices$P.cond$'50',levels=0.05,add=TRUE)
plot(rho.slices$rr.cond$'150.5');tol.contour(rho.slices$P.cond$'150.5',levels=0.05,add=TRUE)
}

}
\references{
Fernando, W.T.P.S. and Hazelton, M.L. (2014), Generalizing the spatial relative risk function, \emph{Spatial and Spatio-temporal Epidemiology}, \bold{8}, 1-10.
}
\seealso{
\code{\link{spattemp.density}}, \code{\link{spattemp.risk}}, \code{\link{bivariate.density}}
}
\author{
T.M. Davies
}
