% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vtl.R
\name{estimateVTL}
\alias{estimateVTL}
\title{Estimate vocal tract length}
\usage{
estimateVTL(formants, method = c("meanFormant", "meanDispersion",
  "regression")[3], speedSound = 35400, checkFormat = TRUE)
}
\arguments{
\item{formants}{a character string like "aaui" referring to default presets
for speaker "M1"; a vector of formant frequencies; or a list of formant
times, frequencies, amplitudes, and bandwidths, with a single value of each
for static or multiple values of each for moving formants}

\item{method}{the method of estimating vocal tract length (see details)}

\item{speedSound}{speed of sound in warm air, cm/s. Stevens (2000) "Acoustic
phonetics", p. 138}

\item{checkFormat}{if TRUE, expands shorthand format specifications into the
canonical form of a list with four components: time, frequency, amplitude
and bandwidth for each format (as returned by the internal function
\code{reformatFormants})}
}
\value{
Returns the estimated vocal tract length in cm.
}
\description{
Estimates the length of vocal tract based on formant frequencies, assuming
that the vocal tract can be modeled as a tube open at both ends.
}
\details{
If \code{method = 'meanFormant'}, vocal tract length (VTL) is calculated
separately for each formant as \eqn{(2 * formant_number - 1) * speedSound /
(4 * formant_frequency)}, and then the resulting VTLs are averaged. If
\code{method = 'meanDispersion'}, formant dispersion is calculated as the
mean distance between formants, and then VTL is calculated as \eqn{speed of
sound / 2 / formant dispersion}. If \code{method = 'regression'}, formant
dispersion is estimated using the regression method described in Reby et al.
(2005) "Red deer stags use formants as assessment cues during intrasexual
agonistic interactions". For a review of these and other VTL-related summary
measures of formant frequencies, refer to Pisanski et al. (2014) "Vocal
indicators of body size in men and women: a meta-analysis". See also
\code{\link{schwa}} for VTL estimation with additional information on formant
frequencies.
}
\examples{
estimateVTL(NA)
estimateVTL(500)
estimateVTL(c(600, 1850, 3100))
estimateVTL(formants = list(f1 = 600, f2 = 1650, f3 = 2400))

# Missing values are OK
estimateVTL(c(600, 1850, 3100, NA, 5000))

# For moving formants, frequencies are averaged over time,
# i.e. this is identical to c(600, 1650, 2400)
estimateVTL(formants = list(f1 = c(500, 700), f2 = 1650, f3 = c(2200, 2600)))

# Note that VTL estimates based on the commonly reported 'meanDispersion'
# depend only on the first and last formant
estimateVTL(c(500, 1400, 2800, 4100), method = 'meanDispersion')
estimateVTL(c(500, 1100, 2300, 4100), method = 'meanDispersion') # identical
# ...but
estimateVTL(c(500, 1400, 2800, 4100), method = 'meanFormant')
estimateVTL(c(500, 1100, 2300, 4100), method = 'meanFormant') # much longer

\dontrun{
# Compare the results produced by the three methods
nIter = 1000
out = data.frame(meanFormant = rep(NA, nIter), meanDispersion = NA, regression = NA)
for (i in 1:nIter) {
  # generate a random formant configuration
  f = runif(1, 300, 900) + (1:6) * rnorm(6, 1000, 200)
  out$meanFormant[i]    = estimateVTL(f, method = 'meanFormant')
  out$meanDispersion[i] = estimateVTL(f, method = 'meanDispersion')
  out$regression[i]     = estimateVTL(f, method = 'regression')
}
pairs(out)
cor(out)
# 'meanDispersion' is pretty different, while 'meanFormant' and 'regression'
# give broadly comparable results
}
}
