% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simLife.R
\name{simFracture}
\alias{simFracture}
\title{Fatigue lifetime simulation}
\usage{
simFracture(stress, S, opt, param, last.defect = FALSE, CL = NULL)
}
\arguments{
\item{stress}{applied stress level}

\item{S}{non-overlapping geometry system}

\item{opt}{control list for simulation of individual failure times}

\item{param}{parameter list for generating individual failure times}

\item{last.defect}{logical, \code{last.defect=FALSE} (default), return either all defect
accumulations or only the last one}

\item{CL}{optional, NULL (default), predefined clustered regions of objects}
}
\value{
a list made up of the following objects if \code{last.defect=FALSE} (default):
				\itemize{
					\item{fracture}{ return value of \code{\link{simDefect}} }
				    \item{times}{ return value of \code{\link{simTimes}} }
					\item{cl_info}{ additional cluster information of responsible defect cluster}
				}
				otherwise only \code{cl_info} is returned.
}
\description{
Simulation of fatigue lifetime model
}
\details{
We provide a wrapper function for \code{\link{simDefect}} to simulate the proposed fatigue lifetime model including
the generation of individual failure times with additional information of the responsible accumulated defect projection
area (simply called defect area) which leads to the failure of the system. This information is returned in the following
list with elements: \code{crack} the responsible defect itself, \code{T} the individual failure times of objects aggregated
as a cluster of objects, \code{ferrit} whether any secondary phase (here called \code{ferrit}) is involved for overall failure,
\code{interior} whether the defect occurs in the interior or at the boundaries of the simulation box. The optional
list argument \code{CL} defines clustered regions, see \code{\link{simCluster}}, of objects sticking together more close than
others. This is useful in case one also wishes to model densely lieing agglomerations of objects (i.e. clusters of particles or
fibers) as an obvious phenomenon of some specimen in mind. This list is made up of the following elements: \code{id}, the id of
the region, \code{center} the center point of the corresponding region, \code{r} the radius (as the Euclidean distance from the
center point) which within all objects belong to this region, \code{interior} whether any object within radius \code{r} hits
the box boundaries of the simulation box.
}
\examples{
\dontrun{

## Simulate a defect accumulation
## of a particle system
library(simLife)

## primary particles (as prolate spheroids)
## and secondary phase (as spheres)
data(AL2MC_20p_k10_F2p_S)

## which is clustered and densified according to CL 
## additional predefined clustered regions
data(AL2MC_20p_k10_F2p_CL)

## the box is stored together with the geometry system 
box <- attr(S,"box")

## distTol=0.25, so use 25\% of accumulation distance
opt <- list("vickers"=107, "distTol"=0.25, "Tmax"=10^12,
		"inAreafactor"=1.56, "outAreafactor"=1.43, 
		"pointsConvHull"=10, "scale"=1e+06,"pl"=101)

## constants 'const' are set to the following internally
## and can partially be overwritten if needed
# const <- list("Em"=68.9,"Ef"=400,"nc"=28.2,"nu"=0.33,
#		    "pf"=0.138,"nE"=NULL,"sigref"=276,"Vref"=5891)

par <- list("P"=c(0.01,5,0.5,50,-11,3),
		"F"=c(0,0,0,105,-12,1),"const"=NULL)

# stress amplitude applied	
stress <- 110
## generate individual (particles') failure times
CLT <- simTimes(S,par,vickers=opt$vickers,stress=stress)

## generated random failure times
T <- unlist(sapply(CLT,`[[`,"T"))
V <- unlist(sapply(CLT,`[[`,"V"))
U <- unlist(sapply(CLT,`[[`,"U"))

dev.new()
showDensity(list("Debonding"=log10(V),"Fracture"=log10(U),"Failure time"=log10(T)),xlim=c(-2,15))

# Area max
(inA <- areaCrit(opt$vickers, stress=stress, factor=opt$inAreafactor, scale=opt$scale))
(outA <- areaCrit(opt$vickers, stress=stress, factor=opt$outAreafactor, scale=opt$scale))

## run accumulation
RET <- simDefect(stress,S,CLT,opt)
length(RET)

#### alternatively, includes simulating times by 'simTimes'
# SIM <- simFracture(stress,S,opt,par,last.defect=FALSE,CL=CL)	
# SIM$cl_info

## some simple analysis of accumulation paths until failure,
## that is, while the critical area is exceeded
LR <- RET[[length(RET)]]
isInCluster <- any(unlist(lapply(CL,function(x,y)
					any(y$id \%in\% x$id) , y=LR)))
cat("Broken cluster: ", isInCluster,"\\t Ferrit: ",
	any("F" \%in\% LR$label),"\\t Acc.size",length(LR$id),"\\n")
	
## select only clusters of size larger than 'msize'	
msize <- 1
id <- sapply(RET,function(x) ifelse(length(x$id)>msize,TRUE,FALSE))
cat("Number of defect projections in last cluster: ",length(RET[[length(RET)]]$id),"\\n")

## draw all accumulation paths until failure
dev.new()
L <- plotDefectAcc(RET,last.path=FALSE)
## draw last accumulation path until failure
dev.new()
L <- plotDefectAcc(RET,last.path=TRUE)

## 3D visualization of final defect projection area
#library(rgl)
#library(unfoldr)
#qid <- LR$id
#open3d()
#spheroids3d(S[qid],box=box, col=c("#0000FF","#00FF00","#FF0000","#FF00FF"))
#
### drawing only last cluster leading to failure
#drawDefectProjections(S,list(LR))
}
}
\author{
M. Baaske
}
