% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shadowHeight.R
\docType{methods}
\name{shadowHeight}
\alias{shadowHeight}
\alias{shadowHeight,SpatialPoints-method}
\alias{shadowHeight,Raster-method}
\title{Shadow height calculation considering sun position and obstacles}
\usage{
\S4method{shadowHeight}{SpatialPoints}(location, obstacles,
  obstacles_height_field, solar_pos, b = 0.01,
  parallel = getOption("mc.cores"), filter_footprint = FALSE)

\S4method{shadowHeight}{Raster}(location, obstacles,
  obstacles_height_field, solar_pos, b = 0.01,
  parallel = getOption("mc.cores"), filter_footprint = FALSE)
}
\arguments{
\item{location}{A \code{SpatialPoints*} or \code{Raster*} object, specifying the location(s) for which to calculate shadow height}

\item{obstacles}{A \code{SpatialPolygonsDataFrame} object specifying the obstacles outline}

\item{obstacles_height_field}{Name of attribute in \code{obstacles} with extrusion height for each feature}

\item{solar_pos}{A matrix with two columns representing sun position(s); first column is the solar azimuth (in decimal degrees from North), second column is sun elevation (in decimal degrees); rows represent different positions (e.g. at different times of day)}

\item{b}{Buffer size when joining intersection points with building outlines, to determine intersection height}

\item{parallel}{Number of parallel processes or a predefined socket cluster. With \code{parallel=1} uses ordinary, non-parallel processing. Parallel processing is done with the \code{parallel} package}

\item{filter_footprint}{Should the points be filtered using \code{shadowFootprint} before calculating shadow height? This can make the calculation faster when there are many point which are not shaded}
}
\value{
Returned object is either a numeric \code{matrix} or a \code{Raster*} -
\itemize{
\item{If input \code{location} is a \code{SpatialPoints*}, then returned object is a \code{matrix}, where rows represent spatial locations (\code{location} features), columns represent solar positions (\code{solar_pos} rows) and values represent shadow height}
\item{If input \code{location} is a \code{Raster*}, then returned object is a \code{RasterLayer} or \code{RasterStack} where layers represent solar positions (\code{solar_pos} rows) and pixel values represent shadow height}
}
In both cases the numeric values express shadow height -
\itemize{
\item{\code{NA} value means no shadow}
\item{A \strong{valid number} expresses shadow height, in CRS units (e.g. meters)}
\item{\code{Inf} means complete shadow (i.e. sun below horizon)}
}
}
\description{
This function calculates shadow height at given points or complete grid (\code{location}),
taking into account:\itemize{
\item{Obstacles outline (\code{obstacles}), given by a polygonal layer with a height attribute (\code{obstacles_height_field})}
\item{Sun position (\code{solar_pos}), given by azimuth and elevation angles}
}
}
\note{
For a correct geometric calculation, make sure that:\itemize{
\item{The layers \code{location} and \code{obstacles} are projected and in same CRS}
\item{The values in \code{obstacles_height_field} of \code{obstacles} are given in the same distance units as the CRS (e.g. meters when using UTM)}
}
}
\examples{
# Single location
location = rgeos::gCentroid(rishon)

\dontrun{
location_geo = spTransform(location, "+proj=longlat +datum=WGS84")
}
location_geo = matrix(c(34.7767978098526, 31.9665936050395), ncol = 2)
time = as.POSIXct("2004-12-24 13:30:00", tz = "Asia/Jerusalem")
solar_pos = maptools::solarpos(location_geo, time)
plot(rishon, main = time)
plot(location, add = TRUE)
sun = shadow:::.sunLocation(location = location, sun_az = solar_pos[1,1], sun_elev = solar_pos[1,2])
sun_ray = ray(from = location, to = sun)
rishon_outline = as(rishon, "SpatialLinesDataFrame")
inter = rgeos::gIntersection(rishon_outline, sun_ray)
plot(sun_ray, add = TRUE, col = "yellow")
plot(inter, add = TRUE, col = "red")
shadowHeight(
  location = location,
  obstacles = rishon,
  obstacles_height_field = "BLDG_HT",
  solar_pos = solar_pos
)

\dontrun{
# Two locations and three times
location0 = rgeos::gCentroid(rishon)
location1 = raster::shift(location0, 0, -15)
location2 = raster::shift(location0, -10, 20)
locations = rbind(location1, location2)
times = seq(from = time, by = "1 hour", length.out = 3)
solar_pos = maptools::solarpos(location_geo, times)
shadowHeight(
  location = locations,
  obstacles = rishon,
  obstacles_height_field = "BLDG_HT",
  solar_pos = solar_pos
)

# Grid
ext = as(raster::extent(rishon), "SpatialPolygons")
r = raster::raster(ext, res = 1)
proj4string(r) = proj4string(rishon)
x = Sys.time()
shadow = shadowHeight(
    location = r,
    obstacles = rishon,
    obstacles_height_field = "BLDG_HT",
    solar_pos = solar_pos,
    parallel = 3
  )
  y = Sys.time()
  y - x
plot(shadow[[1]], col = grey(seq(0.9, 0.2, -0.01)), main = time)
raster::contour(shadow[[1]], add = TRUE)
plot(rishon, add = TRUE, border = "red")
}
}
