% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/est_change_approx.R
\name{est_change_approx}
\alias{est_change_approx}
\title{Standardized Case Influence on Parameter Estimates (Approximate)}
\usage{
est_change_approx(
  fit,
  parameters = NULL,
  case_id = NULL,
  allow_inadmissible = FALSE,
  skip_all_checks = FALSE
)
}
\arguments{
\item{fit}{The output from \code{\link[lavaan:lavaan]{lavaan::lavaan()}} or its wrappers (e.g.,
\code{\link[lavaan:cfa]{lavaan::cfa()}} and \code{\link[lavaan:sem]{lavaan::sem()}}).}

\item{parameters}{A character vector to specify the selected
parameters. Each parameter is named as in \code{lavaan} syntax, e.g.,
\code{x ~ y} or \code{x ~~ y}, as appeared in the columns \code{lhs}, \code{op}, and \code{rhs}
in the output of \code{\link[lavaan:parameterEstimates]{lavaan::parameterEstimates()}}.
Supports specifying an operator to select all parameters with these
operators: \code{~}, \verb{~~}, \verb{=~}, and \code{~1}. This vector can contain
both parameter names and operators. More details can be found
in the help of \code{\link[=pars_id]{pars_id()}}.
If omitted or \code{NULL}, the
default, changes on all free parameters will be computed.}

\item{case_id}{If it is a character vector of length equals to the
number of cases (the number of rows in the data in \code{fit}), then it
is the vector of case identification values. If it is \code{NULL}, the
default, then \code{case.idx} used by \code{lavaan} functions will be used
as case identification values.}

\item{allow_inadmissible}{If \code{TRUE}, accepts a fit object with
inadmissible results (i.e., \code{post.check} from
\code{\link[lavaan:lavInspect]{lavaan::lavInspect()}} is \code{FALSE}). Default is \code{FALSE}.}

\item{skip_all_checks}{If \code{TRUE}, skips all checks and allows
users to run this function on any object of the \code{lavaan} class.
For users to experiment this and other functions on models
not officially supported. Default is \code{FALSE}.}
}
\value{
An \code{est_change}-class object, which is
matrix with the number of columns equals to the number of
requested parameters plus one, the last column being the
approximate generalized Cook's
distance. The number of rows equal to the number
of cases. The row names are the case identification values used in
\code{\link[=lavaan_rerun]{lavaan_rerun()}}. The elements are approximate standardized
differences.
A print method is available for user-friendly output.
}
\description{
Gets a \code{\link[lavaan:lavaan]{lavaan::lavaan()}} output and computes the
approximate standardized changes in selected parameters for each case
if included.
}
\details{
For each case, \code{\link[=est_change_approx]{est_change_approx()}} computes the
approximate differences in the estimates of selected parameters
with and without this case:

(Estimate with all case) - (Estimate without this case)

The differences are standardized by dividing
the approximate raw differences by their standard errors. This is a
measure of the standardized influence of a case on the parameter estimates
if it is included.

If the value of a case is positive, including the case increases an estimate.

If the value of a case is negative, including the case decreases an estimate.

The model is not refitted. Therefore, the result is only an
approximation of that of \code{\link[=est_change]{est_change()}}. However, this
approximation is useful for identifying potentially influential
cases when the sample size is very large or the model takes a long
time to fit. This function can be used to identify potentially
influential cases quickly and then select them to conduct the
leave-one-out sensitivity analysis using \code{\link[=lavaan_rerun]{lavaan_rerun()}} and
\code{\link[=est_change]{est_change()}}.

This function also computes the approximate generalized Cook's
distance (gCD). To avoid confusion, it is labelled \code{gcd_approx}.

For the technical details, please refer to the vignette
on this approach: \code{vignette("casewise_scores", package = "semfindr")}

The approximate approach supports a model with
equality constraints (available in 0.1.4.8 and later version).

Supports both single-group and multiple-group models.
(Support for multiple-group models available in 0.1.4.8 and later version).
}
\examples{
library(lavaan)

# A path model

dat <- pa_dat
mod <-
"
m1 ~ a1 * iv1 + a2 * iv2
dv ~ b * m1
a1b := a1 * b
a2b := a2 * b
"
# Fit the model
fit <- lavaan::sem(mod, dat)
summary(fit)

# Approximate standardized changes and gCD
out_approx <- est_change_approx(fit)
head(out_approx)

# Fit the model several times. Each time with one case removed.
# For illustration, do this only for the first 10 cases.
fit_rerun <- lavaan_rerun(fit, parallel = FALSE,
                          to_rerun = 1:10)
# Compute the changes in chisq if a case is removed
out <- est_change(fit_rerun)
head(out)

# Compare the results
plot(out_approx[1:10, 1], out[, 1])
abline(a = 0, b = 1)
plot(out_approx[1:10, 2], out[, 2])
abline(a = 0, b = 1)
plot(out_approx[1:10, 3], out[, 3])
abline(a = 0, b = 1)
plot(out_approx[1:10, "gcd_approx"], out[, "gcd"])
abline(a = 0, b = 1)

# A CFA model

dat <- cfa_dat
mod <-
"
f1 =~  x1 + x2 + x3
f2 =~  x4 + x5 + x6
f1 ~~ f2
"
# Fit the model
fit <- lavaan::cfa(mod, dat)
summary(fit)

# Approximate standardized changes and gCD
# Compute gCD only for free loadings
out_approx <- est_change_approx(fit,
                                parameters = "=~")
head(out_approx)

# A latent variable model

dat <- sem_dat
mod <-
"
f1 =~  x1 + x2 + x3
f2 =~  x4 + x5 + x6
f3 =~  x7 + x8 + x9
f2 ~   a * f1
f3 ~   b * f2
ab := a * b
"
# Fit the model
fit <- lavaan::sem(mod, dat)
summary(fit)

# Approximate standardized changes and gCD
# Compute gCD only for structural paths
out_approx <- est_change_approx(fit,
                                parameters = "~")
head(out_approx)



}
\author{
Idea by Mark Hok Chio Lai \url{https://orcid.org/0000-0002-9196-7406},
implemented by Shu Fai Cheung \url{https://orcid.org/0000-0002-9871-9448}.
}
