% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scores_sample_univ_weighted.R
\name{scores_sample_univ_weighted}
\alias{scores_sample_univ_weighted}
\alias{twcrps_sample}
\alias{owcrps_sample}
\alias{clogs_sample}
\title{Weighted Scoring Rules for Simulated Forecast Distributions (experimental)}
\usage{
twcrps_sample(
  y,
  dat,
  a = -Inf,
  b = Inf,
  chain_func = function(x) pmin(pmax(x, a), b),
  w = NULL,
  show_messages = TRUE
)

owcrps_sample(
  y,
  dat,
  a = -Inf,
  b = Inf,
  weight_func = function(x) as.numeric(x > a & x < b),
  w = NULL,
  show_messages = TRUE
)

clogs_sample(
  y,
  dat,
  a = -Inf,
  b = Inf,
  bw = NULL,
  show_messages = FALSE,
  cens = TRUE
)
}
\arguments{
\item{y}{vector of realized values.}

\item{dat}{vector or matrix (depending on \code{y}; see details)
of simulation draws from forecast distribution.}

\item{a}{numeric lower bound for the indicator weight function \code{w(z) = 1{a < z < b}}.}

\item{b}{numeric upper bound for the indicator weight function \code{w(z) = 1{a < z < b}}.}

\item{chain_func}{function used to target particular outcomes in the threshold-weighted CRPS; 
the default corresponds to the weight function \code{w(z) = 1{a < z < b}}.}

\item{w}{optional; vector or matrix (matching \code{dat}) of ensemble weights. 
Note that these weights are not used in the weighted scoring rules; see details.}

\item{show_messages}{logical; display of messages (does not affect
warnings and errors).}

\item{weight_func}{function used to target particular outcomes in the outcome-weighted CRPS; 
the default corresponds to the weight function \code{w(z) = 1{a < z < b}}.}

\item{bw}{optional; vector (matching \code{y}) of bandwidths for kernel density
estimation for \code{\link{clogs_sample}}; see details.}

\item{cens}{logical; if TRUE, \code{\link{clogs_sample}} returns the censored
likelihood score; if FALSE, \code{\link{clogs_sample}} returns the conditional
likelihood score.}
}
\value{
Value of the score. \emph{A lower score indicates a better forecast.}
}
\description{
Calculate weighted scores given observations and draws from univariate predictive distributions.
The weighted scoring rules that are available are the threshold-weighted CRPS, outcome-weighted CRPS, 
and conditional and censored likelihood scores. Note that the functions 
documented here are a new experimental feature of the package, and feedback is highly welcome.
}
\details{
For a vector \code{y} of length n, \code{dat} should be given as a matrix
with n rows. If \code{y} has length 1, then \code{dat} may be a vector.

\code{\link{twcrps_sample}} transforms \code{y} and \code{dat} using the chaining
function \code{chain_func} and then calls \code{\link{crps_sample}}. 
\code{\link{owcrps_sample}} weights \code{y} and \code{dat} using the weight function
\code{weight_func} and then calls \code{\link{crps_sample}}. 
See the documentation for \code{\link{crps_sample}} for further details.

The default weight function used in the weighted scores is \code{w(z) = 1{a < z < b}}, 
which is equal to one if \code{z} is between \code{a} and \code{b}, and zero otherwise.
This weight function emphasises outcomes between \code{a} and \code{b}, and is 
commonly used in practical applications when interest is on values above a threshold
(set \code{b = Inf} and \code{a} equal to the threshold) or below a threshold 
(set \code{a = -Inf} and \code{b} equal to the threshold). 

Alternative weight functions can also be employed using the \code{chain_func} 
and \code{weight_func} arguments to \code{\link{twcrps_sample}} and \code{\link{owcrps_sample}},
respectively. Computation of the threshold-weighted CRPS for samples from a predictive distribution 
requires a chaining function rather than a weight function. This is why a chaining 
function is an input for \code{\link{twcrps_sample}} whereas a weight function is an 
input for \code{\link{owcrps_sample}}. Since \code{\link{clogs_sample}} requires 
kernel density estimation to approximate the forecast density, it cannot readily
be calculated for arbitrary weight functions, and is thus only available for 
the canonical weight function \code{w(z) = 1{a < z < b}}.

The \code{chain_func} and \code{weight_func} arguments are functions that will 
be applied to the vector \code{y} and the columns of \code{dat}. It is assumed
that these functions are vectorised. Both functions must take a vector as an input
and output a vector of the same length, containing the weight (for \code{weight_func}) 
or transformed value (for \code{chain_func}) corresponding to each element in the 
input vector. An error will be returned if \code{weight_func} returns
negative values, and a warning message will appear if \code{chain_func} is 
not increasing. 

If no custom argument is given for \code{a}, \code{b}, \code{chain_func} or 
\code{weight_func}, then both \code{\link{twcrps_sample}} and \code{\link{owcrps_sample}} 
are equivalent to the standard unweighted \code{\link{crps_sample}}, and 
\code{\link{clogs_sample}} is equivalent to \code{\link{logs_sample}}. 

The \code{w} argument is also present in the unweighted scores (e.g. \code{\link{crps_sample}}).
\code{w} is used to weight the draws from the predictive distribution, and does 
not weight particular outcomes within the weighted scoring rules. This should not be
confused with the \code{weight_func} argument, which is used within the weighted scores.
}
\examples{
\dontrun{

y <- rnorm(10)
sample_fc <- matrix(rnorm(100), nrow = 10)

crps_sample(y = y, dat = sample_fc)
twcrps_sample(y = y, dat = sample_fc)
owcrps_sample(y = y, dat = sample_fc)

logs_sample(y = y, dat = sample_fc)
clogs_sample(y = y, dat = sample_fc)
clogs_sample(y = y, dat = sample_fc, cens = FALSE)

# emphasise outcomes above 0
twcrps_sample(y = y, dat = sample_fc, a = 0)
owcrps_sample(y = y, dat = sample_fc, a = 0)
clogs_sample(y = y, dat = sample_fc, a = 0)
clogs_sample(y = y, dat = sample_fc, a = 0, cens = FALSE)

# emphasise outcomes below 0
twcrps_sample(y = y, dat = sample_fc, b = 0)
owcrps_sample(y = y, dat = sample_fc, b = 0)
clogs_sample(y = y, dat = sample_fc, b = 0) 

# emphasise outcomes between -1 and 1
twcrps_sample(y = y, dat = sample_fc, a = -1, b = 1)
owcrps_sample(y = y, dat = sample_fc, a = -1, b = 1)
clogs_sample(y = y, dat = sample_fc, a = -1, b = 1)


# a must be smaller than b 
twcrps_sample(y = y, dat = sample_fc, a = 1, b = -1) # error
owcrps_sample(y = y, dat = sample_fc, a = 0, b = 0) # error
clogs_sample(y = y, dat = sample_fc, a = 10, b = 9) # error

# a and b must be single numeric values (not vectors)
twcrps_sample(y = y, dat = sample_fc, a = rnorm(10)) # error


# the owCRPS is not well-defined if none of dat are between a and b
y <- rnorm(10)
sample_fc <- matrix(runif(100, -5, 1), nrow = 10)
owcrps_sample(y = y, dat = sample_fc, a = 1)
# the twCRPS is zero if none of y and dat are between a and b
twcrps_sample(y = y, dat = sample_fc, a = 1) 


# alternative custom weight and chaining functions can also be used

# Example 1: a Gaussian weight function with location mu and scale sigma
mu <- 0
sigma <- 0.5
weight_func <- function(x) pnorm(x, mu, sigma)
# a corresponding chaining function is
chain_func <- function(x) (x - mu)*pnorm(x, mu, sigma) + (sigma^2)*dnorm(x, mu, sigma)

x <- seq(-2, 2, 0.01)
plot(x, weight_func(x), type = "l") # positive outcomes are given higher weight
plot(x, chain_func(x), type = "l") 

owcrps_sample(y = y, dat = sample_fc, a = mu)
owcrps_sample(y = y, dat = sample_fc, weight_func = weight_func)
twcrps_sample(y = y, dat = sample_fc, a = mu)
twcrps_sample(y = y, dat = sample_fc, chain_func = chain_func)


# Example 2: a sigmoid (or logistic) weight function with location mu and scale sigma
weight_func <- function(x) plogis(x, mu, sigma)
chain_func <- function(x) sigma*log(exp((x - mu)/sigma) + 1)

x <- seq(-2, 2, 0.01)
plot(x, weight_func(x), type = "l") # positive outcomes are given higher weight
plot(x, chain_func(x), type = "l") 

owcrps_sample(y = y, dat = sample_fc, a = mu)
owcrps_sample(y = y, dat = sample_fc, weight_func = weight_func)
twcrps_sample(y = y, dat = sample_fc, a = mu)
twcrps_sample(y = y, dat = sample_fc, chain_func = chain_func)


# Example 3: the weight function w(z) = 1{z < a or z > b}
a <- -1
b <- 1
weight_func <- function(x) as.numeric(x < a | x > b)
chain_func <- function(x) (x < a)*(x - a) + (x > b)*(x - b) + a

x <- seq(-2, 2, 0.01)
plot(x, weight_func(x), type = "l")
plot(x, chain_func(x), type = "l")

owcrps_sample(y = y, dat = sample_fc, weight_func = weight_func)
twcrps_sample(y = y, dat = sample_fc, chain_func = chain_func)
twcrps_sample(y = y, dat = sample_fc, b = -1) + twcrps_sample(y = y, dat = sample_fc, a = 1)
crps_sample(y = y, dat = sample_fc) - twcrps_sample(y = y, dat = sample_fc, a = -1, b = 1)
}

}
\references{
\emph{Threshold-weighted CRPS:}

Gneiting, T. and R. Ranjan (2011): 
`Comparing density forecasts using threshold-and quantile-weighted scoring rules', 
\emph{Journal of Business & Economic Statistics} 29, 411-422. 
\doi{10.1198/jbes.2010.08110}

Allen, S., Ginsbourger, D. and J. Ziegel (2022): 
`Evaluating forecasts for high-impact events using transformed kernel scores', 
\emph{arXiv preprint} arXiv:2202.12732.
\doi{10.48550/arXiv.2202.12732}
 
\emph{Outcome-weighted CRPS:}
 
Holzmann, H. and B. Klar (2017):
`Focusing on regions of interest in forecast evaluation',
\emph{Annals of Applied Statistics} 11, 2404-2431. 
\doi{10.1214/17-AOAS1088}

\emph{Conditional and censored likelihood scores:}

Diks, C., Panchenko, V. and D. Van Dijk (2011):
`Likelihood-based scoring rules for comparing density forecasts in tails',
\emph{Journal of Econometrics} 163, 215-230.
\doi{10.1016/j.jeconom.2011.04.001}
}
\seealso{
\link{scores_sample_univ} for standard (un-weighted) scores based on simulated forecast distributions. \link{scores_sample_multiv_weighted} for weighted scores based on simulated multivariate forecast distributions.
}
\author{
Sam Allen
}
