% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mantel_haenszel.R
\name{rr_rd_mantel_haenszel}
\alias{rr_rd_mantel_haenszel}
\title{Risk Ratios and Risk Differences from Mantel-Haenszel Estimators}
\usage{
rr_rd_mantel_haenszel(
  data,
  exposure,
  outcome,
  confounders,
  estimand = c("rr", "rd"),
  conf.level = 0.95
)
}
\arguments{
\item{data}{Data set.}

\item{exposure}{Exposure variable. Must be binary or categorical. The first
level is treated as unexposed.}

\item{outcome}{Outcome variable. Must be binary.}

\item{confounders}{Optional. Binary or categorical variable(s) to perform
stratification over. Supply more than one variable using
\code{confounders = c(var1, var2)}.}

\item{estimand}{Optional. \code{"rr"} for risk ratio; \code{"rd"} for risk
difference. Defaults to \code{"rr"}.}

\item{conf.level}{Optional. Confidence level. Defaults to \code{0.95}.}
}
\value{
Tibble in \code{\link[broom]{tidy}} format with
\itemize{
\item \code{term} the (non-reference) exposure levels
\item \code{estimate} Risk ratio (on log scale) or risk difference
\item \code{std.error}, \code{conf.low}, and \code{conf.high}  Square-root of M-H
variance estimate, and the corresponding confidence limits (on log scale
for RR)
\item \code{model} always \code{"mh"}
\item \code{estimand} \code{"rr"} or \code{"rd"}
}
}
\description{
This function implements the Mantel-Haenszel estimators for risk ratio and
risk differences for a binary or categorical exposure and one or more
categorical confounder(s). Compare to estimates from regression models.
}
\examples{
# Newman SC. Biostatistical methods in epidemiology. New York, NY:
# Wiley, 2001, table 5.3

library(tibble)  # used to set up example data
dat <- tibble(
  death    = c(rep(1, 54), rep(0, 138)),
  stage    = c(rep("Stage I", 7),  rep("Stage II", 26), rep("Stage III", 21),
               rep("Stage I", 60), rep("Stage II", 70), rep("Stage III", 8)),
  receptor = c(rep("Low", 2),  rep("High", 5),  rep("Low", 9),  rep("High", 17),
               rep("Low", 12), rep("High", 9),  rep("Low", 10), rep("High", 50),
               rep("Low", 13), rep("High", 57), rep("Low", 2),  rep("High", 6)))

# Risk difference
rr_rd_mantel_haenszel(
  data = dat,
  exposure = stage,
  outcome = death,
  confounders = receptor,
  estimand = "rd")

# Risk ratio, log scale:
result <- rr_rd_mantel_haenszel(
  data = dat,
  exposure = stage,
  outcome = death,
  confounders = receptor,
  estimand = "rr")
result

# Risk ratio, exponentiated:
result \%>\%
  dplyr::mutate(dplyr::across(.cols = c(estimate, conf.low, conf.high),
                              .fns = exp))
}
\references{
Greenland S, Rothman KJ. Introduction to Stratified Analysis. In: Rothman KJ,
Greenland S, Lash TL. Modern Epidemiology. 3rd edn. Lippincott Williams &
Wilkins: Philadelphia, PA 2008. Page 275. Risk ratios: formulae 15-18, -20,
-22. Risk differences: formulae 15-18, -19, -21.
}
