#' Cancel Pending Orders
#'
#' @name cancel_order
#'
#' @description This is an auth based function. User must have valid api keys generated by GDAX which must be passed as mandatory arguments. The users can cancel all orders pending except stop orders (GDAX does not treat STOP order as an open order unless the stop price is kicked in). User can now pass an optional order id to cancel a specific order including stop orders. Open orders can now be determined with \code{\link{open_orders}}
#'
#'@param order_id Optional character value. This is the order id as generated by GDAX.
#' @param api.key Mandatory character value. This is the API key as generated by GDAX. Typically a 32 character value.
#' @param secret Mandatory character value. This is the API secret as generated by GDAX. Typically a 88 character value.
#' @param passphrase Mandatory character value. This is the passphrase as generated by GDAX. Typically a 11 character value.
#'
#' @return  A Dataframe of order-ids of all orders that were cancelled.
#' @examples
#' \dontrun{
#' cancel_order(api.key = your_key,
#'              secret = your_api_secret,
#'              passphrase = your_api_pass)
#'
#' cancel_order(order_id = "a0a00000-0000-000a-a000-a0a0aa00000a",
#'              api.key = your_key,
#'              secret = your_api_secret,
#'              passphrase = your_api_pass)
#' }
#' @export

cancel_order <- function(order_id = "all",
                         api.key,
                         secret,
                         passphrase) {

  #get url extension----
  if (order_id == "all"){
    req.url <- "/orders/"
    print(req.url)
  } else {
    req.url <- paste0("/orders/",order_id)
    print(req.url)
  }


  #define method----
  method = "DELETE"

  #fetch response----
  response <-
    auth(
      method = method,
      req.url = req.url,
      api.key = api.key,
      secret = secret,
      passphrase = passphrase
    )

  #transform----
  response <- as.data.frame(response)

  #return----
  return(response)
}
