% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/elevation.r
\name{elevation}
\alias{elevation}
\title{Get elevation for lat/long points from a data.frame or list of points.}
\usage{
elevation(
  input = NULL,
  latitude = NULL,
  longitude = NULL,
  latlong = NULL,
  elevation_model = "srtm3",
  username = Sys.getenv("GEONAMES_USER"),
  key,
  curlopts,
  ...
)
}
\arguments{
\item{input}{A data.frame of lat/long data. There must be columns
decimalLatitude and decimalLongitude.}

\item{latitude}{A vector of latitude's. Must be the same length as the
longitude vector.}

\item{longitude}{A vector of longitude's. Must be the same length as
the latitude vector.}

\item{latlong}{A vector of lat/long pairs. See examples.}

\item{elevation_model}{(character) one of srtm3 (default), srtm1, astergdem,
or gtopo30. See "Elevation models" below for more}

\item{username}{(character) Required. An GeoNames user name. See Details.}

\item{key, curlopts}{defunct. see docs}

\item{...}{curl options passed on to \link[crul:verb-GET]{crul::verb-GET}
see \code{\link[curl:curl_options]{curl::curl_options()}} for curl options}
}
\value{
A new column named \code{elevation_geonames} in the supplied data.frame
or a vector with elevation of each location in meters. Note that data from
GBIF can already have a column named \code{elevation}, thus the column we
add is named differently.
}
\description{
Uses the GeoNames web service
}
\section{GeoNames user name}{

To get a GeoNames user name, register for an account at
http://www.geonames.org/login - then you can enable your account for the
GeoNames webservice on your account page. Once you are enabled to use
the webservice, you can pass in your username to the \code{username}
parameter. Better yet, store your username in your \code{.Renviron} file, or
similar (e.g., .zshrc or .bash_profile files) and read it in via
\code{Sys.getenv()} as in the examples below. By default we do
\code{Sys.getenv("GEONAMES_USER")} for the \code{username} parameter.
}

\section{Elevation models}{

\itemize{
\item srtm3:
\itemize{
\item sample area: ca 90m x 90m
\item result: a single number giving the elevation in meters according to
srtm3, ocean areas have been masked as "no data" and have been assigned
a value of -32768
}
\item srtm1:
\itemize{
\item sample area: ca 30m x 30m
\item result: a single number giving the elevation in meters according to
srtm1, ocean areas have been masked as "no data" and have been assigned
a value of -32768
}
\item astergdem (Aster Global Digital Elevation Model V2 2011):
\itemize{
\item sample area: ca 30m x 30m, between 83N and 65S latitude
\item result: a single number giving the elevation in meters according to
aster gdem, ocean areas have been masked as "no data" and have been
assigned a value of -32768
}
\item gtopo30:
\itemize{
\item sample area: ca 1km x 1km
\item result: a single number giving the elevation in meters according to
gtopo30, ocean areas have been masked as "no data" and have been
assigned a value of -9999
}
}
}

\examples{
\dontrun{
user <- Sys.getenv("GEONAMES_USER")

occ_key <- name_suggest('Puma concolor')$key[1]
dat <- occ_search(taxonKey = occ_key, return = 'data', limit = 300,
  hasCoordinate = TRUE)
head( elevation(dat, username = user) )

# Pass in a vector of lat's and a vector of long's
elevation(latitude = dat$decimalLatitude[1:10],
  longitude = dat$decimalLongitude[1:10],
  username = user, verbose = TRUE)

# Pass in lat/long pairs in a single vector
pairs <- list(c(31.8496,-110.576060), c(29.15503,-103.59828))
elevation(latlong=pairs, username = user)

# Pass on curl options
pairs <- list(c(31.8496,-110.576060), c(29.15503,-103.59828))
elevation(latlong=pairs, username = user, verbose = TRUE)

# different elevation models
lats <- dat$decimalLatitude[1:5]
lons <- dat$decimalLongitude[1:5]
elevation(latitude = lats, longitude = lons, elevation_model = "srtm3", verbose = TRUE)
elevation(latitude = lats, longitude = lons, elevation_model = "srtm1", verbose = TRUE)
elevation(latitude = lats, longitude = lons, elevation_model = "astergdem", verbose = TRUE)
elevation(latitude = lats, longitude = lons, elevation_model = "gtopo30", verbose = TRUE)
}
}
\references{
GeoNames http://www.geonames.org/export/web-services.html
}
