% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reprex.R
\name{reprex}
\alias{reprex}
\title{Render a reprex}
\usage{
reprex(x = NULL, venue = c("gh", "so", "r", "R"), si = FALSE,
  show = TRUE, input = NULL, outfile = NULL, comment = "#>",
  opts_chunk = NULL, opts_knit = NULL)
}
\arguments{
\item{x}{An expression. If not given, \code{reprex()} looks for code in
\code{input} or on the clipboard, in that order.}

\item{venue}{"gh" for GitHub (default), "so" for StackOverflow, "r" or "R"
for a runnable R script, with commented output interleaved.}

\item{si}{Whether to include the results of
\code{\link[devtools:session_info]{devtools::session_info}()}, if available, or
\code{\link{sessionInfo}()} at the end of the reprex. When \code{venue =
"gh"} (the default), session info is wrapped in a collapsible details tag.}

\item{show}{Whether to show rendered output in a viewer (RStudio or browser).
Defaults to \code{TRUE}.}

\item{input}{Character. If has length one and lacks a terminating newline,
interpreted as the path to a file containing reprex code. Otherwise,
assumed to hold reprex code as character vector (length greater than one)
or string (with embedded newlines).}

\item{outfile}{Optional basename for output files. When \code{NULL}
(default), reprex writes to temp files below the session temp directory. If
\code{outfile = "foo"}, expect output files in current working directory,
like \code{foo_reprex.R}, \code{foo_reprex.md}, and, if \code{venue = "R"},
\code{foo_rendered.R}. If \code{outfile = NA}, expect output files in
current working directory with basename derived from the path in
\code{input}, if sensible, otherwise from \code{\link{tempfile}()}.}

\item{comment}{Character. Prefix with which to comment out output, defaults
to \code{"#>"}.}

\item{opts_chunk, opts_knit}{Named list. Optional
\href{http://yihui.name/knitr/options/}{knitr chunk and package options},
respectively, to supplement or override reprex defaults. See Details.}
}
\value{
Character vector of rendered reprex, invisibly.
}
\description{
Run a bit of R code using
\code{\link[rmarkdown:render]{rmarkdown::render}()}. The goal is to make it
easy to share a small reproducible example ("reprex"), e.g., in a GitHub
issue. Reprex source can be
\itemize{
\item read from clipboard
\item read from current selection or active document
(\link[=reprex_addin]{"Render reprex" RStudio addin})
\item provided directly as expression, character vector, or string
\item read from file
}
The usual "code + commented output" is returned invisibly, put on the
clipboard, and written to file. An HTML preview displays in RStudio's Viewer
pane, if available, or in the default browser, otherwise. Leading \code{"> "}
prompts, are stripped from the input code. Read more at
\url{http://jennybc.github.io/reprex/}.
}
\details{
reprex sets specific \href{http://yihui.name/knitr/options/}{knitr options},
which you can supplement or override via the \code{opts_chunk} and
\code{opts_knit} arguments or via explicit calls to knitr in your reprex code
(see examples). If all you want to override is the \code{comment} option, use
the dedicated argument, e.g.\code{commment = "#;-)"}.

\itemize{
\item Chunk options default to \code{collapse = TRUE}, \code{comment = "#>"},
\code{error = TRUE}. These are options you normally set via
\code{knitr::opts_chunk$set()}. Note that \code{error = TRUE}, because a
common use case is bug reporting.
\item Package options default to \code{upload.fun = knitr::imgur_upload}.
These are options you normally set via \code{knitr::opts_knit$set()}. The
\code{upload.fun} defaults to \code{\link[knitr]{imgur_upload}} so figures
produced by the reprex appear properly on GitHub.
}
}
\examples{
\dontrun{
# put some code like this on the clipboard
# (y <- 1:4)
# mean(y)
reprex()

# provide code as an expression
reprex(rbinom(3, size = 10, prob = 0.5))
reprex({y <- 1:4; mean(y)})

# note that you can include newlines in those brackets
# in fact, that is probably a good idea
reprex({
  x <- 1:4
  y <- 2:5
  x + y
})

## provide code via character vector
reprex(input = c("x <- 1:4", "y <- 2:5", "x + y"))

## if just one line, terminate with '\\n'
reprex(input = "rnorm(3)\\n")

## customize the output comment prefix
reprex(rbinom(3, size = 10, prob = 0.5), comment = "#;-)")

# override a default chunk option, in general
reprex({(y <- 1:4); median(y)}, opts_chunk = list(collapse = FALSE))
# the above is simply shorthand for this and produces same result
reprex({
  #+ setup, include = FALSE
  knitr::opts_chunk$set(collapse = FALSE)

  #+ actual-reprex-code
  (y <- 1:4)
  median(y)
})

# add prose, use general markdown formatting
reprex({
  #' # A Big Heading
  #'
  #' Look at my cute example. I love the
  #' [reprex](https://github.com/jennybc/reprex#readme) package!
  y <- 1:4
  mean(y)
})

# read reprex from file
writeLines(c("x <- 1:4", "mean(x)"), "foofy.R")
reprex(input = "foofy.R")

# read from file and write to similarly-named outfiles
reprex(input = "foofy.R", outfile = NA)
list.files(pattern = "foofy")
file.remove(list.files(pattern = "foofy"))

# write rendered reprex to file
reprex({
  x <- 1:4
  y <- 2:5
  x + y
}, outfile = "foofy")
list.files(pattern = "foofy")
file.remove(list.files(pattern = "foofy"))

# write reprex to file AND keep figure local too, i.e. don't post to imgur
reprex({
  #' Some prose
  ## regular comment
  (x <- 1:4)
  median(x)
  plot(x)
  }, outfile = "blarg", opts_knit = list(upload.fun = identity))
list.files(pattern = "blarg")
unlink(list.files(pattern = "blarg"), recursive = TRUE)

## target venue = StackOverflow
## http://stackoverflow.com/editing-help
ret <- reprex({
  x <- 1:4
  y <- 2:5
  x + y
}, venue = "so")
ret

## target venue = R, also good for email or Slack snippets
ret <- reprex({
  x <- 1:4
  y <- 2:5
  x + y
}, venue = "R")
ret

## include prompt and don't comment the output
## use this when you want to make your code hard to execute :)
reprex({
  x <- 1:4
  y <- 2:5
  x + y
}, opts_chunk = list(comment = NA, prompt = TRUE))

## leading prompts are stripped from source
reprex(input = c("> x <- 1:3", "> median(x)"))
}

}
