% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/triangularize.R
\name{triangularize}
\alias{triangularize}
\title{Create x-coordinates so the points form a triangle}
\usage{
triangularize(
  data,
  y_col = NULL,
  .min = NULL,
  .max = NULL,
  offset_x = 0,
  keep_original = TRUE,
  x_col_name = ".triangle_x",
  edge_col_name = ".edge",
  overwrite = FALSE
)
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{y_col}{Name of column in \code{`data`} with y-coordinates to create x-coordinates for.}

\item{.min}{Minimum y-coordinate. If \code{NULL}, it is inferred by the given y-coordinates.}

\item{.max}{Maximum y-coordinate. If \code{NULL}, it is inferred by the given y-coordinates.}

\item{offset_x}{Value to offset the x-coordinates by.}

\item{keep_original}{Whether to keep the original columns. (Logical)

Some columns may have been overwritten, in which case only the newest versions are returned.}

\item{x_col_name}{Name of new column with the x-coordinates.}

\item{edge_col_name}{Name of new column with the edge identifiers. If \code{NULL}, no column is added.

Numbering is clockwise and starts at the upper-right edge.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
\code{data.frame} (\code{tibble}) with the added x-coordinates and an identifier
for the edge the data point is a part of.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Create the x-coordinates for a \code{vector} of y-coordinates such that
they form a triangle.

The data points are stochastically distributed based on edge lengths, why it might be preferable to
set a random seed.

This will likely look most like a triangle when the y-coordinates are somewhat equally distributed,
e.g. a uniform distribution.
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)
library(purrr)
library(ggplot2)

# Set seed
set.seed(1)

# Create a data frame
df <- data.frame(
  "y" = runif(200),
  "g" = factor(rep(1:5, each = 40))
)

# Triangularize 'y'
df_tri <- triangularize(df, y_col = "y")
df_tri

# Plot triangle
df_tri \%>\%
  ggplot(aes(x = .triangle_x, y = y, color = .edge)) +
  geom_point() +
  theme_minimal()

#
# Grouped squaring
#

# Triangularize 'y' for each group
# First cluster the groups a bit to move the
# triangles away from each other
df_tri <- df \%>\%
  cluster_groups(
    cols = "y",
    group_cols = "g",
    suffix = "",
    overwrite = TRUE
  ) \%>\%
  dplyr::group_by(g) \%>\%
  triangularize(
    y_col = "y",
    overwrite = TRUE
  )

# Plot triangles
df_tri \%>\%
  ggplot(aes(x = .triangle_x, y = y, color = g)) +
  geom_point() +
  theme_minimal()

#
# Specifying minimum value
#

# Specify minimum value manually
df_tri <- triangularize(df, y_col = "y", .min = -2)
df_tri

# Plot triangle
df_tri \%>\%
  ggplot(aes(x = .triangle_x, y = y, color = .edge)) +
  geom_point() +
  theme_minimal()

#
# Multiple triangles by contraction
#

\donttest{
# Start by squaring 'y'
df_tri <- triangularize(df, y_col = "y")

# Contract '.triangle_x' and 'y' towards the centroid
# To contract with multiple multipliers at once,
# we wrap the call in purrr::map_dfr
df_expanded <- purrr::map_dfr(
  .x = 1:10 / 10,
  .f = function(mult) {
    expand_distances(
      data = df_tri,
      cols = c(".triangle_x", "y"),
      multiplier = mult,
      origin_fn = centroid,
      overwrite = TRUE
    )
  }
)
df_expanded

df_expanded \%>\%
  ggplot(aes(
    x = .triangle_x_expanded, y = y_expanded,
    color = .edge, alpha = .multiplier
  )) +
  geom_point() +
  theme_minimal()
}
}
\seealso{
Other forming functions: 
\code{\link{circularize}()},
\code{\link{hexagonalize}()},
\code{\link{square}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{forming functions}
