 \name{rTableICC.2x2xK}
\alias{rTableICC.2x2xK}

\title{
Randomly Generate 2x2xK Contingency Tables over Intraclass-Correlated Individuals
}
\description{
A generic function that generates 2 x 2 x K contingency tables over intraclass-correlated cells under product multinomial, multinomial or Poisson sampling plans. 
}
\usage{
rTableICC.2x2xK(p,theta,M,sampling="Multinomial",N=0,lambda=NULL,zero.clusters=FALSE,
                 print.regular=TRUE,print.raw=FALSE)

}

\arguments{
  \item{p}{
A finite matrix of cell probabilities. Dimension of \code{p} is \eqn{K\times 4}, where 2x2 tables under each of K centers is expanded to a vector of 4 elements. Elements in each row of \code{p} should correspond to the cells (1,1), (1,2), (2,1), and (2,2), respectively. 
}
  \item{theta}{
  A finite and positive valued \eqn{(T-1)\times 1} vector of predetermined ICCs, where \eqn{T} is the maximum number of individuals allowed in each cluster. The first element of \code{theta} represents the ICC for clusters of size 2. 
  }
  \item{M}{
  The total number of clusters under each center. If number of clusters under each center is unbalanced, \eqn{M} would be a \eqn{K\times 1} vector.
  }
  \item{sampling}{
  Sampling plan. It takes 'Product' for product multinomial sampling, 'Multinomial' for multinomial sampling, and 'Poisson' for Poisson sampling plans.
  }  
  \item{N}{
  Total number of individuals to be generated under product multinomial or multinomial sampling plans. It is a vector of positive integers containing total number of observations in each center under product multinomial sampling plan, a positive integer of total sample size under all centers under multinomial sampling plan, and not required under Poisson sampling plan. If \code{N} is not a positive integer, it is converted to do so.
  } 
  \item{lambda}{
  Mean number of individuals in each cluster. It is either a \eqn{K\times 1} positive vector or a positive scalar under Poisson sampling plan and not required for both multinomial and product multinomial sampling plans. If a positive scalar is entered, mean number of individuals in each cluster under all centers will be equal to each other. If a vector is entered, mean number of individuals in each cluster under each center will be equal to each other.
  }    
  \item{zero.clusters}{
  If \code{TRUE}, generation of clusters with no individuals are allowed. Otherwise, each cluster has at least one individual. 
  }  
  \item{print.regular}{
  If \code{TRUE}, generated random table is printed in 2x2xK format. Otherwise, generated random table is printed in two dimensional format. 
  }  
  \item{print.raw}{
  If \code{TRUE}, generated raw data is printed on the screen.
  }     
}
\details{
To generate random tables under multinomial sampling plan, first total sample size is distributed to clusters with equal probabilities using the code \code{rmultinom(1, N, rep(1/M,M))}. Then, for each center, the package \code{partitions} is utilized to distribute individuals across cells of 2 x 2 tables under the pre-determined intraclass correlations. Let \eqn{n} and \eqn{m} be integer to be partitioned (cluster size) and order of partition, respectively. If there is more than one individual (\eqn{n>1}) in a cluster, all possible compositions of order \eqn{RC} of cluster size \eqn{n} into at most \eqn{m} parts are generated by using \code{compositions} function. This provides all possible distributions of individuals in the cluster of interest into 2 x 2 table of interest. If all individuals are at the same cell, the following equation is used to calculate the probability that all individuals in the \eqn{i}th cluster fall in the same cell of a contingency table of interest:

\deqn{
      \theta_{t}p_{ij}+(1-\theta_{t})(p_{ij})^{t},
} 
      
where \eqn{i,j=1,2}, \eqn{0\le\theta\le 1}, \eqn{\theta_{t}} is the intraclass correlation for clusters of size \eqn{t} for \eqn{t=2,\dots,T}, and \eqn{\theta_{1}=0}. Otherwise, the probability that the individuals are in different but specified cells is calculated as follows:

\deqn{
      (1-\theta_{t})\prod_{ij}(p_{ij})^{n_{rij}},
}

where \eqn{n_{rij}} be the number of individuals from \eqn{r}th cluster falling in the \eqn{i}th row and \eqn{j}th column of the considered 2 x 2 table (Altham, 1976; Nandram and Choi, 2006; Demirhan, 2013). This provides probability of each possible distribution. Then, calculated probabilities are normalized and the function \code{rDiscrete} is utilized to randomly select one of the generated compositions. By this way, a realization is obtained for each cluster having more than one individual. If there is only one individual in a cluster, a realization is obtained by assigning the individual to cells according to the entered cell probabilities using the function \code{rDiscrete}. The resulting random 2x2 table under one of \eqn{K} centers is constructed by combining these realizations. This process is repeated for all of \eqn{K} centers.

To generate random tables under product multinomial sampling plan, center margins must be entered by \code{N}. Because both cell probabilities and fixed center margins are entered at the same time, margin probabilities calculated over the number of individuals in each center and entered \eqn{K\times 4} matrix of cell probabilities must be equal to each other. To ensure intraclass correlations, the same manner as multinomial sampling plan is applied to each center. Suppose that center totals are denoted by \eqn{n_{ij+}}, where \eqn{i,j=1,2}. Then with the counts satisfying \eqn{\sum_{ij}n_{ijk}=n_{ij+}}, we have the following multinomial form that \code{rTableICC.2x2xK} uses (Agresti, 2002):

\deqn{
      \frac{n_{ij+}!}{\prod_{ij}n_{ijk}!}\prod_{ij}p_{ij|k}^{n_{ijk}},
      } 

where \eqn{k=1,\dots,K}, \eqn{n_{ijk}} is the count of cell \eqn{(i,j,k)}, and given that an individual is in \eqn{k}th center, \eqn{p_{ij|k}} is the conditional probability of being in the cell \eqn{(i,j)} of 2 x 2 table. This multinomial form is used to generate data under each center. 
  
To generate random tables under Poisson sampling plan, the same manner as multinomial sampling plan is taken except cluster sizes are generated from Poisson distribution with entered mean cluster counts and total sample size is calculated over the generated cluster sizes. If zero sized clusters are not allowed, truncated Poisson distribution is used to generate cluster counts.

Because total sample size is randomly distributed into the clusters, it is coincidentally possible to have clusters with more individuals than the allowed maximum cluster size. In this case, the following error message is generated: 

\code{Maximum number of individuals in one of the clusters is 14, which is greater than maximum allowed cluster size. (1) Re-run the function,
(2) increase maximum allowed cluster size by increasing the number of elements of theta, (3) increase total number of clusters, or (4) decrease total number of individuals!} 

and execution is stopped.  
}
\value{
Let \eqn{C} be the set of clusters in which all individuals fall in a single cell of the contingency table and \eqn{C'} be the complement of \eqn{C},  \eqn{K} be the number of centers, and \eqn{T} be the maximum cluster size.
 
 A list with the following elements is generated:
\item{g.t}{A \eqn{2K\times 2\times (T-1)} dimensional array including the number of clusters of size \eqn{t} in \eqn{C'} of size \eqn{t} with all individuals in cell \eqn{(i,j)}, where \eqn{i,j=1,2} and \eqn{t=2,\dots,T}.}
\item{g.tilde}{A \eqn{(T-1)\times 1} dimensional vector including the number of clusters of size \eqn{t} in \eqn{C'}, where \eqn{t=2,\dots,T}.}
\item{rTable}{A \eqn{K\times (2*2)} dimensional matrix including generated 2 x 2 contingency table under each center in its each row.}
\item{rTable.raw}{Generated table in a \eqn{N\times 2K\times 2} dimensional array in raw data format.}
\item{rTable.regular}{Generated table in a \eqn{2\times 2\times K} dimensional array.}
\item{N}{Total number of generated individuals.}
\item{cluster.size}{Size of each generated cluster.}
\item{sampling}{Used sampling plan in data generation.}
\item{M}{Total number of clusters under each center.}
\item{K}{Number of centers.}
\item{T}{Maximum allowed cluster size.}
\item{ICC}{Returns \code{TRUE} to indicate the data is generated under intraclass-correlated clusters.}
\item{structure}{Returns "2 x 2 x K" to indicate structure of generated table is "2 x 2 x K."}
\item{print.raw}{\code{TRUE} if generated table will be printed in raw data format.}
\item{print.regular}{\code{TRUE} if generated table will be printed in the format determined by \code{structure}.}

}
\references{
Agresti A. (2002) \emph{Categorical Data Analysis}, Wiley, New York.

Altham, P.M. (1976) Discrete variable analysis for individuals grouped into families, \emph{Biometrika} \bold{63}, 263--269.

Nandram, B. and Choi, J.W. (2006) Bayesian analysis of a two-way categorical table incorporating intraclass correlation,
\emph{Journal of Statistical Computation and Simulation} \bold{76}, 233--249.

Demirhan, H. (2013) Bayesian estimation of log odds ratios over two-way contingency tables with intraclass-correlated cells, \emph{Journal of Applied Statistics} \bold{40}, 2303--2316.

Demirhan, H. and Hamurkaroglu, C. (2008) Bayesian estimation of log odds ratios from RxC and 2x2xK contingency tables, \emph{Statistica Neerlandica} \bold{62}, 405--424.
}
\author{
Haydar Demirhan

Maintainer: Haydar Demirhan <haydarde@hacettepe.edu.tr>
}

\examples{

# --- Generate a 2x2x4 contingency table under multinomial sampling plan ---
num.centers=4                                # Number of centers
max.cluster.size=9                           # Maximum allowed cluster size
num.cluster=95                               # Total number of clusters under 
                                             #  centers each center is equal across the  
ICCs=array(0.1,dim=max.cluster.size)         # Assign equal ICCs for this exmaple
ICCs[1]=0                                    # Assign zero ICC to clusters with 
                                             #  one individual 
sampl="Multinomial"                          # Generate table under multinomial 
                                             #  sampling plan
num.obs=900                                  # Number of observations to be 
                                             #  generated
cell.prob=array(0.0625,dim=c(num.centers,4)) # Cell probabilities sum up to one 

zeros=FALSE                                  # Do not allow zero sized clusters

x=rTableICC.2x2xK(p=cell.prob,theta=ICCs,M=num.cluster,sampling=sampl,zero.clusters=zeros,
                   N=num.obs,print.regular=TRUE,print.raw=FALSE)
print.rTableICC(x)                          

# --- Generate a 2x2x4 contingency table under product multinomial sampling plan ---

sampl="Product"                              # Generate table under product 
                                             #  multinomial sampling plan
num.obs=c(200,200,200,200)                   # Number of observations to be generated 
                                             #  under each center
cell.prob=array(0.0625,dim=c(num.centers,4)) # Cell probabilities sum up to one 
zeros=FALSE                                  # Do not allow zero sized clusters

y=rTableICC.2x2xK(p=cell.prob,theta=ICCs,M=num.cluster,sampling=sampl,
                  zero.clusters=zeros,N=num.obs,print.regular=TRUE,print.raw=FALSE)
print.rTableICC(y)

# --- Generate a 2x2x4 contingency table under Poisson sampling plan ---

sampl="Poisson"                              # Generate table under Poisson 
                                             #  sampling plan
cell.mean=2                                  # Assign equal mean number of 
                                             #  individual to all cells
cell.prob=array(0.0625,dim=c(num.centers,4)) # Cell probabilities sum up 
                                             #  to one 

z1=rTableICC.2x2xK(p=cell.prob,lambda=cell.mean,theta=ICCs,M=num.cluster,
                    sampling=sampl,N=num.obs,print.regular=TRUE,print.raw=FALSE)
print.rTableICC(z1)

cell.mean=c(2,3,3,2)                         # Assign equal mean number of individual 
                                             #  to cells under each center
max.cluster.size=19                          # Maximum allowed cluster size
ICCs=array(0.1,dim=max.cluster.size)         # Assign equal ICCs for this exmaple
ICCs[1]=0                                    # Assign zero ICC to clusters with one 
                                             #  individual

z2=rTableICC.2x2xK(p=cell.prob,lambda=cell.mean,theta=ICCs,M=num.cluster,
                    sampling=sampl,N=num.obs,print.regular=TRUE,print.raw=FALSE)
print.rTableICC(z2)
}

\keyword{ datagen }
\keyword{contingency}
\keyword{correlation}
\keyword{intraclass}
\keyword{Poisson}
\keyword{multinomial}
\keyword{product}