% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detailed_itineraries.R
\name{detailed_itineraries}
\alias{detailed_itineraries}
\title{Calculate detailed itineraries between origin destination pairs}
\usage{
detailed_itineraries(
  r5r_core,
  origins,
  destinations,
  mode = "WALK",
  mode_egress = "WALK",
  departure_datetime = Sys.time(),
  max_walk_dist = Inf,
  max_trip_duration = 120L,
  walk_speed = 3.6,
  bike_speed = 12,
  max_rides = 3,
  shortest_path = TRUE,
  n_threads = Inf,
  verbose = TRUE,
  drop_geometry = FALSE
)
}
\arguments{
\item{r5r_core}{rJava object to connect with R5 routing engine}

\item{origins, destinations}{either a spatial sf POINT object or a data.frame
containing the columns 'id', 'lon', 'lat'}

\item{mode}{string. Transport modes allowed for the trips. Defaults to
"WALK". See details for other options.}

\item{mode_egress}{string. Transport mode used after egress from public
transport. It can be either 'WALK', 'BICYCLE', or 'CAR'.
Defaults to "WALK". Ignored when public transport is not
used.}

\item{departure_datetime}{POSIXct object. If working with public transport
networks, please check \code{calendar.txt} within
the GTFS file for valid dates.}

\item{max_walk_dist}{numeric. Maximum walking distance (in meters) for the
whole trip. Defaults to no restrictions on walking, as
long as \code{max_trip_duration} is respected.}

\item{max_trip_duration}{numeric. Maximum trip duration in minutes. Defaults
to 120 minutes (2 hours).}

\item{walk_speed}{numeric. Average walk speed in km/h. Defaults to 3.6 km/h.}

\item{bike_speed}{numeric. Average cycling speed in km/h. Defaults to 12 km/h.}

\item{max_rides}{numeric. The max number of public transport rides allowed in
the same trip. Defaults to 3.}

\item{shortest_path}{logical. Whether the function should only return the
fastest route alternative (the default) or multiple
alternatives.}

\item{n_threads}{numeric. The number of threads to use in parallel computing.
Defaults to use all available threads (Inf).}

\item{verbose}{logical. TRUE to show detailed output messages (the default)
or FALSE to show only eventual ERROR messages.}

\item{drop_geometry}{logical. Indicates whether R5 should drop segment's
geometry column. It can be helpful for saving memory.}
}
\value{
A LINESTRING sf with detailed information about the itineraries
between specified origins and destinations. Distances are in meters
and travel times are in minutes.
}
\description{
Fast computation of (multiple) detailed itineraries between one
or many origin destination pairs.
}
\details{
R5 allows for multiple combinations of transport modes. The options
include:
\subsection{Transit modes}{

TRAM, SUBWAY, RAIL, BUS, FERRY, CABLE_CAR, GONDOLA, FUNICULAR. The option
'TRANSIT' automatically considers all public transport modes available.
}

\subsection{Non transit modes}{

WALK, BICYCLE, CAR, BICYCLE_RENT, CAR_PARK
}
}
\section{Routing algorithm:}{
The detailed_itineraries function uses an R5-specific extension to the
McRAPTOR routing algorithm to find paths that are optimal or less than
optimal, with some heuristics around multiple access modes, riding the same
patterns, etc. The specific extension to McRAPTOR to do suboptimal
path routing are not documented yet, but a detailed description of base
McRAPTOR can be found in Delling et al (2015).
\itemize{
\item Delling, D., Pajor, T., & Werneck, R. F. (2015). Round-based public transit
routing. Transportation Science, 49(3), 591-604.
}
}

\examples{
\donttest{
library(r5r)

# build transport network
data_path <- system.file("extdata/poa", package = "r5r")
r5r_core <- setup_r5(data_path = data_path)

# load origin/destination points
points <- read.csv(file.path(data_path, "poa_points_of_interest.csv"))

# inputs
departure_datetime <- as.POSIXct("13-05-2019 14:00:00", format = "\%d-\%m-\%Y \%H:\%M:\%S")

dit <- detailed_itineraries(r5r_core,
                            origins = points[10,],
                            destinations = points[12,],
                            mode = c("WALK", "TRANSIT"),
                            departure_datetime = departure_datetime,
                            max_walk_dist = 1000,
                            max_trip_duration = 120L)

stop_r5(r5r_core)
}
}
\seealso{
Other routing: 
\code{\link{travel_time_matrix}()}
}
\concept{routing}
