% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/base_surv.R
\name{qgcomp.cox.boot}
\alias{qgcomp.cox.boot}
\title{estimation of quantile g-computation fit for a survival outcome}
\usage{
qgcomp.cox.boot(f, data, expnms = NULL, q = 4, breaks = NULL,
  id = NULL, alpha = 0.05, B = 200, MCsize = 10000, degree = 1,
  seed = NULL, parallel = FALSE, ...)
}
\arguments{
\item{f}{R style survival formula, which includes \code{\link[survival]{Surv}}
in the outcome defintion. E.g. \code{Surv(time,event) ~ exposure}}

\item{data}{data frame}

\item{expnms}{character vector of exposures of interest}

\item{q}{NULL or number of quantiles used to create quantile indicator variables
representing the exposure variables. If NULL, then gcomp proceeds with un-transformed
version of exposures in the input datasets (useful if data are already transformed,
or for performing standard g-computation)}

\item{breaks}{(optional) NULL, or a list of (equal length) numeric vectors that 
characterize the minimum value of each category for which to 
break up the variables named in expnms. This is an alternative to using 'q'
to define cutpoints.}

\item{id}{(optional) NULL. Reserved for future use. This does nothing yet.}

\item{alpha}{alpha level for confidence limit calculation}

\item{B}{integer: number of bootstrap iterations (this should typically be
>=200, though it is set lower in examples to improve run-time).}

\item{MCsize}{integer: sample size for simulation to approximate marginal 
hazards ratios (if < sample size, then set to sample size). Note that large
values will slow down the fitting, but will result in higher accuracy - if you 
run the function multiple times you will see that results vary due to simuation
error. Ideally, MCsize would be set such that simulation error is negligible
in the precision reported (e.g. if you report results to 2 decimal places, then
MCsize should be set high enough that you consistenty get answers that are the same
to 2 decimal places).}

\item{degree}{polynomial basis function for marginal model (e.g. degree = 2
allows that the relationship between the whole exposure mixture and the outcome
is quadratic.}

\item{seed}{integer or NULL: random number seed for replicable bootstrap results}

\item{parallel}{logical (default FALSE): use future package to speed up bootstrapping}

\item{...}{arguments to glm (e.g. family)}
}
\value{
a qgcompfit object, which contains information about the effect
 measure of interest (psi) and associated variance (var.psi), as well
 as information on the model fit (fit) and information on the 
 marginal structural model (msmfit) used to estimate the final effect
 estimates.
}
\description{
This function yields population average effect estimates for 
  (possibly right censored) time-to event outcomes
}
\details{
`qgcomp.cox.boot' estimates the
 log(hazard ratio) per quantile increase in the joint exposure to all exposures 
 in `expnms'. This function uses g-computation to estimate the parameters of a
 marginal structural model for the population average effect of increasing all
 expsoures in `expnms' by a single quantile. This approach involves specifying 
 an underlying conditional outcome model, given all exposures of interest (possibly
 with non-linear basis function representations such as splines or product terms)
 and confounders or covariates of interest. This model is fit first, which is used
 to generate expected outcomes at each quantile of all exposures, which is then
 used in a second model to estimate a population average dose-response curve that
 is linear or follows a simple polynomial function. See section on MCSize below
 
 Test statistics and confidence intervals are based on 
 a non-parametric bootstrap, using the standard deviation of the bootstrap
 estimates to estimate the standard error. The bootstrap standard error is 
 then used to estimate Wald-type confidence intervals. Note that no bootstrapping
 is done on estimated quantiles of exposure, so these are treated as fixed
 quantities
 
 MCSize is crucial to get accurate point estimates. In order to get marginal
 estimates of the population hazard under different values of the joint exposure
 at a given quantile for all exposures in `expnmns`, `qgcomp.cox.boot` uses
 Monte Carlo simulation to generate outcomes implied by the underlying conditional model
 and then fit a separate (marginal structural) model to those outcomes. In order to get
 accurate results that don't vary much from run-to-run of this approach, MCsize
 must be set large enough so that results are stable across runs according to a pre-determined
 precision (e.g. 2 significant digits).
}
\examples{
set.seed(50)
N=200
dat <- data.frame(time=(tmg <- pmin(.1,rweibull(N, 10, 0.1))), 
                d=1.0*(tmg<0.1), x1=runif(N), x2=runif(N), z=runif(N))
expnms=paste0("x", 1:2)
f = survival::Surv(time, d)~x1 + x2
(fit1 <- survival::coxph(f, data = dat))
(obj <- qgcomp.cox.noboot(f, expnms = expnms, data = dat))
# not run (slow when using boot version to proper precision)
#(obj2 <- qgcomp.cox.boot(f, expnms = expnms, data = dat, B=10, MCsize=2000))
# using future package, marginalizing over confounder z
#(obj3 <- qgcomp.cox.boot(survival::Surv(time, d)~x1 + x2 + z, expnms = expnms, data = dat, 
#                         B=1000, MCsize=20000, parallel=TRUE))
## non-constant hazard ratio, non-linear terms
#(obj4 <- qgcomp.cox.boot(survival::Surv(time, d)~factor(x1) + splines::bs(x2) + z, 
#                         expnms = expnms, data = dat, 
#                         B=1000, MCsize=20000, parallel=FALSE, degree=1))
}
\seealso{
\code{\link[qgcomp]{qgcomp.cox.noboot}}, and \code{\link[qgcomp]{qgcomp}}
}
\concept{variance mixtures}
