% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/exact.R
\name{exact}
\alias{exact}
\title{A wrapper function supporting exact simulation of dose-escalation trials.}
\usage{
exact(selector_factory)
}
\arguments{
\item{selector_factory}{An object of type
\code{\link[escalation:get_three_plus_three]{three_plus_three_selector_factory}},
with \code{allow_deescalation = TRUE}.}
}
\description{
Implemented currently only for the (most?) common variant of the 3+3 design,
which requires that at least 6 patients be treated at a dose before it may be
declared as \sQuote{the} MTD.
}
\details{
In any given realization of a 3+3 design, each of the \eqn{D} prespecified doses
will enroll 0, 1 or 2 cohorts, each with 3 patients. Each cohort will result in
a tally of 0--3 dose-limiting toxicities (DLTs), and these may be recorded in a
\eqn{2 \times D}{2 x D} matrix. Moreover, the 3+3 dose-escalation rules allow for
only one path through any such matrix. For example, the matrix
\preformatted{
   d
c   D1 D2 D3 D4
  1  0  1  2 NA
  2 NA  0 NA NA
}
represents the path in a 4-dose 3+3 trial, where the following events occur:
\enumerate{
\item Initial cohort at \eqn{d=1} results 0/3
\item Escalation to \eqn{d=2} results 1/3
\item Additional cohort at \eqn{d=2} results 0/3 for net 1/6 at this dose
\item Escalation to \eqn{d=3} results 2/3; MTD declared as \eqn{d=1}.
}
(Indeed, as you may verify at the R prompt, the above matrix is the 262nd of 442
such paths enumerated comprehensively in the \eqn{2 \times 4 \times 442}{2 x 4 x 442}
array \code{precautionary:::T[[4]]}.)

As detailed in Norris 2020c (below), these matrices may be used to construct simple
matrix computations that altogether eliminate the need for discrete-event simulation
of the 3+3 design. For each \eqn{D = 2,...,8}, the \code{precautionary} package has
pretabulated a \eqn{J \times 2D}{J x 2D} matrix \code{precautionary:::U[[D]]} and
\eqn{J}-vector \code{precautionary:::b[[D]]} such that the eqn{J}-vector \eqn{\pi}
of path probabilities is given by:
\deqn{
log(\pi) = b + U [log(p), log(q)]',
}
where \eqn{p} is the \eqn{D}-vector of DLT probabilities at the prespecified
doses, and \eqn{q \equiv 1-p}{q := 1-p} is its complement. See Eq. (4) of
Norris (2020c).

For details on the enumeration itself, please see the Prolog code in directory
\code{exec/} of the installed package.
}
\examples{
# Run an exact version of the simulation from FDA-proactive vignette
design <- get_three_plus_three(
  num_doses = 6
, allow_deescalate = TRUE)
old <- options(
  dose_levels = c(2, 6, 20, 60, 180, 400)
, ordinalizer = function(MTDi, r0 = 1.5)
    MTDi * r0 ^ c(Gr1=-2, Gr2=-1, Gr3=0, Gr4=1, Gr5=2)
)
mtdi_gen <- hyper_mtdi_lognormal(CV = 0.5
                                ,median_mtd = 180
                                ,median_sdlog = 0.6
                                ,units="ng/kg/week")
exact(design) \%>\% simulate_trials(
  num_sims = 1000
, true_prob_tox = mtdi_gen) -> EXACT
summary(EXACT)$safety
if (interactive()) { # runs too long for CRAN servers
  # Compare with discrete-event-simulation trials
  design \%>\% simulate_trials(
    num_sims = 1000
  , true_prob_tox = mtdi_gen) -> DISCRETE
  summary(DISCRETE)$safety[,]
  # Note the larger MCSEs in this latter simulation, reflecting combined noise
  # from hyperprior sampling and the nested discrete-event trial simulations.
  # The MCSE of the former simulation is purely from the hyperprior sampling,
  # since the nested trial simulation is carried out by an exact computation.
}
options(old)
}
\references{
Norris DC. What Were They Thinking? Pharmacologic priors implicit in a choice
of 3+3 dose-escalation design. arXiv:2012.05301 [stat.ME]. December 2020.
\url{https://arxiv.org/abs/2012.05301}
}
