% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate.R
\name{simulate.plcp}
\alias{simulate.plcp}
\alias{simulate.plcp_multi}
\title{Perform a simulation study using a \code{study_parameters}-object}
\usage{
\method{simulate}{plcp}(object, nsim, seed = NULL, formula = NULL,
  satterthwaite = FALSE, CI = FALSE, cores = 1, progress = FALSE,
  batch_progress = TRUE, ...)

\method{simulate}{plcp_multi}(object, nsim, seed = NULL, formula = NULL,
  satterthwaite = FALSE, CI = FALSE, cores = 1, progress = FALSE,
  batch_progress = TRUE, ...)
}
\arguments{
\item{object}{An object created by \code{\link{study_parameters}}.}

\item{nsim}{The number of simulations to run.}

\item{seed}{Currently ignored.}

\item{formula}{\code{lme4::lmer} formula(s) used to analyze the data, see \emph{Details}.
Should either be a character vector or named list if one model is simulated.
It also possible to compare two models, e.g. a correct and a misspecified model,
by passing the two formulas as a named list, with the names "wrong" and "correct".
See \emph{Examples}. If \code{NULL} the formula is made automatically,
using \code{\link{create_lmer_formula}}, which does not support objects with multiple
simulation setups.}

\item{satterthwaite}{Logical; if \code{TRUE} Satterthwaite's degrees of freedom
approximation will be used when computing \emph{p}-values. This is implemented using
the \code{lmerTest}-package. See \emph{Details}.}

\item{CI}{Logical; if \code{TRUE} coverage rates for confidence intervals
will be calculated. See \emph{Details}.}

\item{cores}{Number of CPU cores to use. If called from a GUI environment (e.g. RStudio) or
a computer running Microsoft Windows, PSOCK clusters will be used. If called from a
non-interactive Unix environment forking is utilized.}

\item{progress}{\code{logical}; will display progress if \code{TRUE}. Currently
ignored on \emph{Windows}. Package \code{pbmclapply} is used to display progress,
which relies on forking. \strong{N.B} using a progress bar will noticeably
increase the simulation time, due to the added overhead.}

\item{batch_progress}{\code{logical}; if \code{TRUE} progress will be shown for
simulations with multiple setups.}

\item{...}{Optional arguments, see \emph{Saving} in \emph{Details} section.}
}
\description{
Perform a simulation study using a \code{study_parameters}-object
}
\details{
\strong{Model formula}

The available model terms are:
\itemize{
 \item \code{y} the outcome vector
 \item \code{time} the time vector
 \item \code{treatment} treatment indicator (0 = "control", 1 = "treatment")
 \item \code{subject} subject-level id variable, from 1 to total number of subjects.
 \item \code{cluster} for three-level models; the cluster-level id variable,
 from 1 to the total number of clusters.
}

See \emph{Examples} and the simulation-vignette for formula examples. For
\code{object}s that contain a single study setup, then the lmer formula
can be created automatically using \code{\link{create_lmer_formula}}.

\strong{Satterthwaite's approximation, and CI coverage}

To decrease the simulation time Satterthwaite's \emph{dfs} and the CIs' coverage rates
will only be calculated for the test of 'time:treatment'-interaction.

Confidence intervals are both calculated using profile likelihood and by
the Wald approximation.

\strong{Saving intermediate results for multi-sims}

Objects with multi-sims can be save after each batch is finished. This is highly
recommended when many designs are simulated. The following additional arguments
control saving behavior:

\itemize{
 \item \code{'save'}, \code{logical}, if \code{TRUE} each batch is saved as a
 \code{RDS}-file. Results are saved in your working directory, in the directory
 specified by \code{save_folder}.
 \item \code{'save_folder'} a \code{character} indicating the folder name. Default is \code{'save'}.
 \item \code{'save_folder_create'},  \code{logical}, if \code{TRUE} then \code{save_folder}
 will be created if it does not exist in your working directory.
}
}
\examples{
\dontrun{
# Two-level ---------------------------------------------------------------
p <- study_parameters(n1 = 11,
                      n2 = 25,
                      sigma_subject_intercept = 1.44,
                      sigma_subject_slope = 0.2,
                      sigma_error = 1.44,
                      cohend = 0.5)

f <- "y ~ treatment * time + (1 + time | subject)"


res <- simulate(object = p,
                nsim = 1000,
                formula = f,
                satterthwaite = TRUE,
                progress = FALSE,
                cores = 1,
                save = FALSE)

summary(res)


# Three-level (nested) ---------------------------------------------------------
p <- study_parameters(n1 = 10,
                      n2 = 20,
                      n3 = 4,
                      sigma_subject_intercept = 1.44,
                      icc_pre_cluster = 0,
                      sigma_subject_slope = 0.2,
                      icc_slope = 0.05,
                      sigma_error = 1.44,
                      cohend = 0.5)

## compare correct and miss-specified model
f <- list("correct" = "y ~ treatment * time + (1 + time | subject) + (time | cluster)",
          "wrong" = "y ~ treatment * time + (1 + time | subject)")

res <- simulate(object = p,
                nsim = 1000,
                formula = f,
                satterthwaite = TRUE,
                progress = FALSE,
                cores = 1,
                save = FALSE)

summary(res)


# Partially nested design ------------------------------------------------
p <- study_parameters(n1 = 11,
                      n2 = 10,
                      n3 = 4,
                      sigma_subject_intercept = 1.44,
                      icc_pre_cluster = 0,
                      sigma_subject_slope = 0.2,
                      cor_subject = -0.5,
                      icc_slope = 0.05,
                      sigma_error = 1.44,
                      partially_nested = TRUE,
                      cohend = -0.5)

f <- "y ~ treatment * time + (1 + time | subject) + (0 + treatment:time | cluster)"

res <- simulate(object = p,
                nsim = 1000,
                formula = f,
                satterthwaite = TRUE,
                progress = FALSE,
                cores = 4,
                save = FALSE)

summary(res)

# Run multiple designs  --------------------------------------------------------
p <- study_parameters(n1 = 10,
                      n2 = 20,
                      n3 = c(2, 4, 6),
                      sigma_subject_intercept = 1.44,
                      icc_pre_cluster = 0,
                      sigma_subject_slope = 0.2,
                      icc_slope = 0.05,
                      sigma_error = 1.44,
                      cohend = 0.5)

f <- list("correct" = "y ~ treatment * time + (1 + time | subject) + (time | cluster)",
          "wrong" = "y ~ treatment * time + (1 + time | subject)")

res <- simulate(object = p,
                nsim = 1000,
                formula = f,
                satterthwaite = TRUE,
                progress = FALSE,
                cores = 1,
                save = FALSE)

# Summarize 'time:treatment' results for n3 = c(2, 4, 6) for 'correct' model
summary(res, para = "time:treatment", type = "fixed", model = "correct")

# Summarize cluster-level random slope  for n3 = c(2, 4, 6) for 'correct' model
summary(res, para = "cluster_slope", type = "random", model = "correct")
}
}
\seealso{
\code{\link{summary.plcp_sim}}, \code{\link{simulate_data}}
}
