% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/survival_aggregated.R, R/survival_lexis.R
\name{survtab_ag}
\alias{survtab}
\alias{survtab_ag}
\title{Estimate Survival Time Functions}
\usage{
survtab_ag(formula = NULL, data, adjust = NULL, weights = NULL,
  surv.breaks = NULL, n = "at.risk", d = "from0to1",
  n.cens = "from0to0", pyrs = "pyrs", d.exp = "d.exp", n.pp = NULL,
  d.pp = "d.pp", d.pp.2 = "d.pp.2", n.cens.pp = "n.cens.pp",
  pyrs.pp = "pyrs.pp", d.exp.pp = "d.exp.pp", surv.type = "surv.rel",
  surv.method = "hazard", relsurv.method = "e2", subset = NULL,
  conf.level = 0.95, conf.type = "log-log", verbose = FALSE)

survtab(formula, data, adjust = NULL, breaks = NULL, pophaz = NULL,
  weights = NULL, surv.type = "surv.rel", surv.method = "hazard",
  relsurv.method = "e2", subset = NULL, conf.level = 0.95,
  conf.type = "log-log", verbose = FALSE)
}
\arguments{
\item{formula}{a \code{formula} object. For \code{survtab_ag}, the response 
must be the time scale to compute survival time function estimates
over, e.g. \code{fot ~ sex + adjust(agegr)}. For \code{survtab} it can
also be the just e.g. \code{fot ~ sex + adjust(agegr)}, whereupon it is
assumed that \code{lex.Xst} in your data is the status variable in the
intended format. To be explicit, use \code{\link[survival]{Surv}}
containing the 
survival time scale and the event indicator as the right-hand-side, e.g. 
\code{Surv(fot, lex.Xst) ~ sex}.
Stratifying variables can be included on the right-hand side
separated by '\code{+}'. May contain usage of \code{adjust()} 
--- see Details.}

\item{data}{for \code{survtab_ag}, a data set of aggregated counts, 
subject-times, etc., as created
by \code{\link{aggre}}; for pre-aggregated data see \code{\link{as.aggre}}.
For \code{survtab}, a \code{Lexis} object.}

\item{adjust}{can be used as an alternative to passing variables to 
argument \code{formula} within a call to \code{adjust()}; e.g.
\code{adjust = "agegr"}. \link[=flexible_argument]{Flexible input}.}

\item{weights}{typically a list of weights or a \code{character} string
specifying an age group standardization scheme; see
the \link[=direct_standardization]{dedicated help page} 
and examples.}

\item{surv.breaks}{(\code{survtab_ag} only) a vector of breaks on the survival time scale. Only used
to compute estimates using a subset of the intervals in data or larger intervals
than in data. E.g. one might use \code{surv.breaks = 0:5} when the aggregated
data has intervals with the breaks \code{seq(0, 10, 1/12)}.}

\item{n}{variable containing counts of subjects at-risk at the start of a time 
interval; e.g. \code{n = "at.risk"}. 
Required when \code{surv.method = "lifetable"}.
\link[=flexible_argument]{Flexible input}.}

\item{d}{variable(s) containing counts of subjects experiencing an event. With
only one type of event, e.g. \code{d = "deaths"}. With multiple types of 
events (for CIF or cause-specific survival estimation), supply e.g.
\code{d = c("canD", "othD")}. If the survival time function to be estimated
does not use multiple types of events, supplying more than one variable
to \code{d} simply causes the variables to be added together. 
Always required. \link[=flexible_argument]{Flexible input}.}

\item{n.cens}{variable containing counts of subjects censored during a 
survival time interval; E.g. \code{n.cens = "alive"}.
Required when \code{surv.method = "lifetable"}. 
\link[=flexible_argument]{Flexible input}.}

\item{pyrs}{variable containing total subject-time accumulated within a survival
time interval; E.g. \code{pyrs = "pyrs"}. 
Required when \code{surv.method = "hazard"}. Flexible input.}

\item{d.exp}{variable denoting total "expected numbers of events" 
(typically computed \code{pyrs * pop.haz}, where 
\code{pop.haz} is the expected hazard level) 
accumulated within a survival time interval; E.g. \code{pyrs = "pyrs"}.
Required when computing EdererII relative survivals or 
CIFs based on excess counts of events. Flexible input.}

\item{n.pp}{variable containing total Pohar-Perme weighted counts of
subjects at risk in an interval,
supplied as argument \code{n} is supplied. 
Computed originally on the subject
level as analogous to \code{pp * as.integer(status == "at-risk")}.
Required when \code{relsurv.method = "pp"}. Flexible input.}

\item{d.pp}{variable(s) containing total Pohar-Perme weighted counts of events,
supplied as argument \code{d} is supplied. Computed originally on the subject
level as analogous to \code{pp * as.integer(status == some_event)}.
Required when \code{relsurv.method = "pp"}. Flexible input.}

\item{d.pp.2}{variable(s) containing total Pohar-Perme 
"double-weighted" counts of events,
supplied as argument \code{d} is supplied. Computed originally on the subject
level as analogous to \code{pp * pp * as.integer(status == some_event)}.
Required when \code{relsurv.method = "pp"}. Flexible input.}

\item{n.cens.pp}{variable containing total Pohar-Perme weighted counts 
censorings,
supplied as argument \code{n.cens} is supplied. 
Computed originally on the subject
level as analogous to \code{pp * as.integer(status == "censored")}.
Required when \code{relsurv.method = "pp"}. Flexible input.}

\item{pyrs.pp}{variable containing total Pohar-Perme weighted subject-times,
supplied as argument \code{pyrs} is supplied. 
Computed originally on the subject
level as analogous to \code{pp * pyrs}.
Required when \code{relsurv.method = "pp"}. Flexible input.}

\item{d.exp.pp}{variable containing total Pohar-Perme weighted counts 
of excess events,
supplied as argument \code{pyrs} is supplied. 
Computed originally on the subject
level as analogous to \code{pp * d.exp}.
Required when \code{relsurv.method = "pp"}. Flexible input.}

\item{surv.type}{one of \code{'surv.obs'},
\code{'surv.cause'}, \code{'surv.rel'}, 
\code{'cif.obs'} or \code{'cif.rel'}; 
defines what kind of survival time function(s) is/are estimated; see Details}

\item{surv.method}{either \code{'lifetable'} or \code{'hazard'}; determines
the method of calculating survival time functions, where the former computes
ratios such as \code{p = d/(n - n.cens)} 
and the latter utilizes subject-times 
(typically person-years) for hazard estimates such as \code{d/pyrs} 
which are used to compute survival time function estimates.
The former method requires argument \code{n.cens} and the latter 
argument \code{pyrs} to be supplied.}

\item{relsurv.method}{either \code{'e2'} or \code{'pp'}; 
defines whether to compute relative survival using the
EdererII method or using Pohar-Perme weighting;
ignored if \code{surv.type != "surv.rel"}}

\item{subset}{a logical condition; e.g. \code{subset = sex == 1}; 
subsets the data before computations}

\item{conf.level}{confidence level used in confidence intervals; 
e.g. \code{0.95} for 95 percent confidence intervals}

\item{conf.type}{character string; must be one of \code{"plain"}, 
\code{"log-log"} and \code{"log"}; 
defines the transformation used on the survival time
function to yield confidence 
intervals via the delta method}

\item{verbose}{logical; if \code{TRUE}, the function is chatty and
returns some messages and timings along the process}

\item{breaks}{(\code{survtab} only) a named list of breaks, e.g.
\code{list(FUT = 0:5)}. If data is not split in advance, \code{breaks}
must at the very least contain a vector of breaks to split the survival time 
scale (mentioned in argument \code{formula}). If data has already been split
(using e.g. \code{\link{splitMulti}}) along at least the used survival time
scale, this may be \code{NULL}.}

\item{pophaz}{(\code{survtab} only) a \code{data.frame} containing
expected hazards for the event of interest to occur. See the
\link[=pophaz]{dedicated help page}. Required when
\code{surv.type = "surv.rel"} or \code{"cif.rel"}. \code{pophaz} must
contain one column named \code{"haz"}, and any number of other columns
identifying levels of variables to do a merge with split data within
\code{survtab}. Some columns may be time scales, which will
allow for the expected hazard to vary by e.g. calendar time and age.}
}
\value{
Returns a table of life time function values and other 
information with survival intervals as rows.
Returns some of the following estimates of survival time functions:

\itemize{
 \item \code{surv.obs} - observed (raw) survival
 \item \code{CIF_k} - cumulative incidence function for cause \code{k}
 \item \code{CIF.rel} - cumulative incidence function using excess cases
 \item \code{r.e2} -  relative survival, EdererII
 \item \code{r.pp} -  relative survival, Pohar-Perme weighted
}
The suffix \code{.as} implies adjusted estimates, and \code{.lo} and
\code{.hi} imply lower and upper confidence limits, respectively. 
The prefix \code{SE.} stands for standard error.
}
\description{
Estimate survival time functions (survival, relative/net
survival, CIFs) using aggregated data (\code{survtab_ag}) or 
\code{\link[Epi]{Lexis}} data (\code{survtab}).
}
\details{
\strong{Basics}

\code{survtab_ag} computes estimates of survival time functions using 
pre-aggregated data. When using subject-level data directly, use 
\code{survtab}. For aggregating data, see \code{\link{lexpand}}
and \code{\link{aggre}}. Data sets can be coerced into \code{Lexis}
objects using \code{\link[Epi]{Lexis}}.

By default
\code{survtab_ag} makes use of the exact same breaks that were used in 
splitting the original data (with e.g. \code{lexpand}), so it is not 
necessary to specify any
\code{surv.breaks}. If specified, the 
\code{surv.breaks} must be a subset of the pertinent 
pre-existing breaks. Interval lengths (\code{delta} in output) are 
also calculated based on existing breaks or \code{surv.breaks}, 
so the upper limit of the breaks should
therefore be meaningful and never e.g. \code{Inf}. 

\code{survtab} may be a split or unsplit \code{Lexis} data set, but it 
is recommended to supply the \code{breaks} argument anyway.

if \code{surv.type = 'surv.obs'}, only 'raw' observed survival 
is calculated over the chosen time intervals. With
\code{surv.type = 'surv.rel'}, also relative survival estimates 
are supplied in addition to observed survival figures. 

\code{surv.type = 'cif.obs'} requests cumulative incidence functions (CIF) 
to be estimated, where all unique events (supplied via \code{d})
are seen as competing risks. 
CIFs are estimated for each competing risk based 
on a survival-interval-specific proportional hazards
assumption as described by Chiang (1968).  
With \code{surv.type = 'cif.rel'}, a CIF is estimated with using 
excess cases as the ''cause-specific'' cases.

if \code{surv.type = 'surv.cause'}, cause-specific survivals are estimated
separately for each unique value of \code{event.values}.

The vignette \href{../doc/survtab_examples.html}{survtab_examples} 
has some practical examples.

\strong{Relative / net survival}
 
When \code{surv.type = 'surv.rel'}, the user can choose 
\code{relsurv.method = 'pp'}, whereupon
Pohar-Perme weighting is used.
By default \code{relsurv.method = 'e2'}.

\strong{Adjusted estimates}

Adjusted estimates in this context mean computing estimates separately
by the levels of adjusting variables and returning weighted averages
of the estimates. For example, computing estimates separately by
age groups and returning a weighted average estimate (age-adjusted estimate).

Adjusting requires specification of both the adjusting variables and
the weights for all the levels of the adjusting variables. The former can be
accomplished by using \code{adjust()} with the argument \code{formula},
or by supplying variables directly to argument \code{adjust}. E.g. the
following are all equivalent (using \code{survtab_ag}):

\code{formula = fot ~ sex + adjust(agegr, area)}

\code{formula  = fot ~ sex} and \code{adjust = c("agegr", "area")}

\code{formula  = fot ~ sex} and \code{adjust = list(agegr, area)}

When using \code{survtab}, the response must be a 
\code{\link[survival]{Surv}} object, e.g. 
\code{Surv(time = fot, event = lex.Xst)}, but otherwise the 
syntax is the same.

The adjusting variables must match with the variable names in the
argument \code{weights}, which may be supplied as a \code{list} or
a \code{data.frame}. The former can be done by e.g.

\code{weights = list(agegr = VEC1, area = VEC2)},

where \code{VEC1} and \code{VEC2} are vectors of weights (which do not
have to add up to one). When passing a \code{list} of weights, the order
of the weights must match with the order of the levels of the variable:
For \code{factor} variables, they must correspond to the \code{factor}'s
levels. Otherwise they must match to the sorted levels of the variable,
i.e. if variable \code{agegr} has three levels \code{c(1, 2, 3)},
the weights in e.g. \code{list(agegr = c(0.1, 0.4, 0.5))} would pass 
the weight
\code{0.1} for level \code{1} and so on, regardless of the order of values in
the data.


See 
\href{../doc/survtab_examples.html}{survtab_examples} 
for an example of using a \code{data.frame} to pass weights.



\strong{Period analysis and other data selection schemes}

If one wishes to calculate e.g. period analysis (delayed entry estimates), 
one should limit the data when/before supplying to this function. 
See e.g. \code{\link{lexpand}}.
}
\section{Functions}{
\itemize{
\item \code{survtab}: survtab
}}
\examples{
## see more examples with explanations in vignette("survtab_examples")

#### survtab_ag usage

data(sire)
## prepare data for e.g. 5-year "period analysis" for 2008-2012
## note: sire is a simulated cohort integrated into popEpi.
BL <- list(fot=seq(0, 5, by = 1/12),
           per = c("2008-01-01", "2013-01-01"))
x <- lexpand(sire, birth = bi_date, entry = dg_date, exit = ex_date,
             status = status \%in\% 1:2,
             breaks = BL,
             pophaz = popmort,
             aggre = list(fot))
             
## calculate relative EdererII period method 
st <- survtab_ag(fot ~ 1, data = x)

#### survtab usage
library(Epi)
library(survival)

## NOTE: recommended to use factor status variable
x <- Lexis(entry = list(FUT = 0, AGE = dg_age, CAL = get.yrs(dg_date)), 
           exit = list(CAL = get.yrs(ex_date)), 
           data = sire[sire$dg_date < sire$ex_date, ],
           exit.status = factor(status, levels = 0:2, 
                                labels = c("alive", "canD", "othD")), 
           merge = TRUE)

## phony group variable
set.seed(1L)
x$group <- rbinom(nrow(x), 1, 0.5)

\dontrun{
## observed survival. explicit supplying of status:
st <- survtab(Surv(time = FUT, event = lex.Xst) ~ group, data = x, 
              surv.type = "surv.obs",
              breaks = list(FUT = seq(0, 5, 1/12)))
## this assumes the status is lex.Xst (right 99.9 \% of the time)
st <- survtab(FUT ~ group, data = x, 
              surv.type = "surv.obs",
              breaks = list(FUT = seq(0, 5, 1/12)))

#### using dates with survtab
x <- Lexis(entry = list(FUT = 0L, AGE = dg_date-bi_date, CAL = dg_date),
           exit = list(CAL = ex_date),
           data = sire[sire$dg_date < sire$ex_date, ],
           exit.status = factor(status, levels = 0:2, 
                                labels = c("alive", "canD", "othD")), 
           merge = TRUE)
## phony group variable
set.seed(1L)
x$group <- rbinom(nrow(x), 1, 0.5)

st <- survtab(Surv(time = FUT, event = lex.Xst) ~ group, data = x, 
              surv.type = "surv.obs",
              breaks = list(FUT = seq(0, 5, 1/12)*365.25))    
                  
## NOTE: population hazard should be reported at the same scale
## as time variables in your Lexis data.
data(popmort, package = "popEpi")
pm <- data.frame(popmort)
names(pm) <- c("sex", "CAL", "AGE", "haz")
## from year to day level
pm$haz <- pm$haz/365.25 
pm$CAL <- as.Date(paste0(pm$CAL, "-01-01")) 
pm$AGE <- pm$AGE*365.25 

st <- survtab(Surv(time = FUT, event = lex.Xst) ~ group, data = x, 
              surv.type = "surv.rel", relsurv.method = "e2",
              pophaz = pm,
              breaks = list(FUT = seq(0, 5, 1/12)*365.25))  
}
}
\author{
Joonas Miettinen, Karri Seppa
}
\references{
Perme, Maja Pohar, Janez Stare, and Jacques Estève. 
"On estimation in relative survival." Biometrics 68.1 (2012): 113-120.

Hakulinen, Timo, Karri Seppa, and Paul C. Lambert. 
"Choosing the relative survival method for cancer survival estimation." European Journal of Cancer 47.14 (2011): 2202-2210.
 
Seppa, Karri, Timo Hakulinen, and Arun Pokhrel. 
"Choosing the net survival method for cancer survival estimation." European Journal of Cancer (2013).

CHIANG, Chin Long. Introduction to stochastic processes in biostatistics. 1968.
}
\seealso{
\code{\link{splitMulti}}, \code{\link{lexpand}}, 
\code{\link{ICSS}}, \code{\link{sire}},
\code{\link{survtab}}
\href{../doc/survtab_examples.html}{The survtab_examples vignette}
}

