% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ResultsManager.R
\name{ResultsManager}
\alias{ResultsManager}
\title{R6 class representing a results manager.}
\description{
\code{\link[R6:R6Class]{R6}} class to represent a manager for generating summary
metrics and/or matrices from simulation results, as well as optionally regenerating
values via generators.
}
\examples{
# U Island example region
coordinates <- data.frame(x = rep(seq(177.01, 177.05, 0.01), 5),
                          y = rep(seq(-18.01, -18.05, -0.01), each = 5))
template_raster <- Region$new(coordinates = coordinates)$region_raster # full extent
template_raster[][-c(7, 9, 12, 14, 17:19)] <- NA # make U Island
region <- Region$new(template_raster = template_raster)
raster::plot(region$region_raster, main = "Example region (indices)",
             xlab = "Longitude (degrees)", ylab = "Latitude (degrees)",
             colNA = "blue")
# Results manager
results_manager <- ResultsManager$new(
  sample_data = data.frame(index = 1:3),
  simulation_results = PopulationResults$new(region = region),
  summary_metrics = c("trend_n", "total_h"),
  summary_matrices = c("n", "h"),
  summary_functions = list(
    trend_n = function(results) {
      round(results$all$abundance_trend, 2)
    },
    total_h = function(results) {
      sum(results$harvested)
    },
    n = "all$abundance", # string
    h = "all$harvested"),
  parallel_cores = 2,
  results_dir = tempdir())
# Write example result files
results <- list()
for (i in 1:3) {
  results[[i]] <- list(abundance = t(apply(matrix(11:17), 1,
                                           function(n) round(n*exp(-(0:9)/i)))))
  results[[i]]$harvested <- round(results[[i]]$abundance*i/7)
  file_name <- paste0(results_manager$get_results_filename(i), ".RData")
  saveRDS(results[[i]], file.path(tempdir(), file_name))
}
# Generate result metrics and matrices
gen_output <- results_manager$generate()
gen_output$summary
dir(tempdir(), "*.txt") # plus generation log
results_manager$summary_metric_data
results_manager$summary_matrix_list

}
\section{Super classes}{
\code{\link[poems:GenericClass]{poems::GenericClass}} -> \code{\link[poems:GenericManager]{poems::GenericManager}} -> \code{ResultsManager}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{attached}}{A list of dynamically attached attributes (name-value pairs).}
}
\if{html}{\out{</div>}}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{sample_data}}{A data frame of sampled parameters for each simulation/result.}

\item{\code{simulation_results}}{An object of a class inherited from the \code{\link{SimulationResults}} class for encapsulating and dynamically generating simulation results.}

\item{\code{generators}}{A list of generators (\code{\link{Generator}} or inherited class) objects for (optionally) regenerating simulation model values.}

\item{\code{result_attachment_functions}}{A list of functions for attaching intermediate values to the simulation results prior to generation.}

\item{\code{summary_metrics}}{An array of names for summary metrics, each of which are calculated as single values for each simulation. These should refer to list names for the summary functions.}

\item{\code{summary_matrices}}{An array of names for summary matrices, each of which are calculated as a single matrix row for each simulation. These should refer to list names for the summary functions.}

\item{\code{summary_functions}}{A list of functions, result attributes, or constants for transforming individual simulation results to single summary metric values stored in the metric data frame, or to matrix rows stored in the summary matrix list.}

\item{\code{summary_metric_data}}{A data frame of generated summary metrics (one row per simulation).}

\item{\code{summary_matrix_list}}{A list of generated matrices of summary results (each having one row per simulation).}

\item{\code{summary_matrix_weighted_averages}}{A list of calculated weighted averages for each of the summary matrices (using the sample data \emph{weight} column).}

\item{\code{parallel_cores}}{Number of cores for running the simulations in parallel.}

\item{\code{results_dir}}{Results directory path.}

\item{\code{results_ext}}{Result file extension (default is .RData).}

\item{\code{results_filename_attributes}}{A vector of: prefix (optional); attribute names (from the sample data frame); postfix (optional); utilized to construct results filenames.}

\item{\code{error_messages}}{A vector of error messages encountered when setting model attributes.}

\item{\code{warning_messages}}{A vector of warning messages encountered when setting model attributes.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-ResultsManager-new}{\code{ResultsManager$new()}}
\item \href{#method-ResultsManager-generate}{\code{ResultsManager$generate()}}
\item \href{#method-ResultsManager-calculate_result_attachments}{\code{ResultsManager$calculate_result_attachments()}}
\item \href{#method-ResultsManager-calculate_summaries}{\code{ResultsManager$calculate_summaries()}}
\item \href{#method-ResultsManager-log_generation}{\code{ResultsManager$log_generation()}}
\item \href{#method-ResultsManager-calculate_summary_weighted_averages}{\code{ResultsManager$calculate_summary_weighted_averages()}}
\item \href{#method-ResultsManager-clone}{\code{ResultsManager$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="poems" data-topic="GenericClass" data-id="new_clone"><a href='../../poems/html/GenericClass.html#method-GenericClass-new_clone'><code>poems::GenericClass$new_clone()</code></a></span></li>
<li><span class="pkg-link" data-pkg="poems" data-topic="GenericManager" data-id="get_attribute"><a href='../../poems/html/GenericManager.html#method-GenericManager-get_attribute'><code>poems::GenericManager$get_attribute()</code></a></span></li>
<li><span class="pkg-link" data-pkg="poems" data-topic="GenericManager" data-id="get_message_sample"><a href='../../poems/html/GenericManager.html#method-GenericManager-get_message_sample'><code>poems::GenericManager$get_message_sample()</code></a></span></li>
<li><span class="pkg-link" data-pkg="poems" data-topic="GenericManager" data-id="get_results_filename"><a href='../../poems/html/GenericManager.html#method-GenericManager-get_results_filename'><code>poems::GenericManager$get_results_filename()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ResultsManager-new"></a>}}
\if{latex}{\out{\hypertarget{method-ResultsManager-new}{}}}
\subsection{Method \code{new()}}{
Initialization method optionally copies attributes from a simulation (results) manager, sets any included attributes (\emph{sample_data}, \emph{simulation_results}, \emph{generators}, \emph{result_attachment_functions}, \emph{summary_metrics}, \emph{summary_functions}, \emph{parallel_cores}, \emph{results_dir}, \emph{results_ext}, \emph{results_filename_attributes}), and attaches other attributes individually listed.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ResultsManager$new(simulation_manager = NULL, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{simulation_manager}}{Optional \code{\link{SimulationManager}} object (or an object inherited from the \code{\link{GenericManager}} class), from which simulation attributes can be copied.}

\item{\code{...}}{Parameters listed individually.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ResultsManager-generate"></a>}}
\if{latex}{\out{\hypertarget{method-ResultsManager-generate}{}}}
\subsection{Method \code{generate()}}{
Generates the summary metric data and/or matrix list via the summary functions for each simulation sample, and creates/writes a generation log.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ResultsManager$generate(results_dir = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{results_dir}}{Results directory path (must be present if not already set within manager class object).}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Generation log as a list.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ResultsManager-calculate_result_attachments"></a>}}
\if{latex}{\out{\hypertarget{method-ResultsManager-calculate_result_attachments}{}}}
\subsection{Method \code{calculate_result_attachments()}}{
Calculates and attaches intermediate values to the sample result model (via the result attachment functions).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ResultsManager$calculate_result_attachments(simulation_results)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{simulation_results}}{The sample simulation results, an object of a class inherited from \code{\link{SimulationResults}}, to which the intermediate results are attached.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ResultsManager-calculate_summaries"></a>}}
\if{latex}{\out{\hypertarget{method-ResultsManager-calculate_summaries}{}}}
\subsection{Method \code{calculate_summaries()}}{
Calculates the summary metrics and/or matrices for the results of a sample simulation (via the summary functions).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ResultsManager$calculate_summaries(simulation_results, sample_index)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{simulation_results}}{The sample simulation results, an object of a class inherited from \code{\link{SimulationResults}}.}

\item{\code{sample_index}}{Index of sample from data frame.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Generation log entry as a (nested) list, including generated summary metric data and (optionally) matrices.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ResultsManager-log_generation"></a>}}
\if{latex}{\out{\hypertarget{method-ResultsManager-log_generation}{}}}
\subsection{Method \code{log_generation()}}{
Summarizes the log generated within the generate method and writes it to a text file in the results directory.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ResultsManager$log_generation(generation_log)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{generation_log}}{Nested list of log entries generated via the generate method.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Extended generation log as a nested with added summary and failure/warning indices.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ResultsManager-calculate_summary_weighted_averages"></a>}}
\if{latex}{\out{\hypertarget{method-ResultsManager-calculate_summary_weighted_averages}{}}}
\subsection{Method \code{calculate_summary_weighted_averages()}}{
Calculates the weighted averages for each of the summary matrices (providing the sample data has a \emph{weight} column).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ResultsManager$calculate_summary_weighted_averages(na_replacements = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{na_replacements}}{List of values or functions (form: \code{modified_matrix <- function(matrix)}) for dealing with NA values in each summary matrix (default NULL will ignore NAs).}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ResultsManager-clone"></a>}}
\if{latex}{\out{\hypertarget{method-ResultsManager-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ResultsManager$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
