% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trelliPlots.R
\name{trelli_abundance_histogram}
\alias{trelli_abundance_histogram}
\title{Histogram trelliscope building function for abundance data}
\usage{
trelli_abundance_histogram(
  trelliData,
  cognostics = c("sample count", "mean abundance", "median abundance", "cv abundance",
    "skew abundance"),
  ggplot_params = NULL,
  interactive = FALSE,
  path = .getDownloadsFolder(),
  name = "Trelliscope",
  test_mode = FALSE,
  test_example = 1,
  single_plot = FALSE,
  ...
)
}
\arguments{
\item{trelliData}{A trelliscope data object made by as.trelliData or
as.trelliData.edata, and grouped by edata_cname in trelli_panel_by.
Required.}

\item{cognostics}{A vector of cognostic options for each plot. Valid entries
are "sample count", "mean abundance", "median abundance", "cv abundance", 
and "skew abundance". All are included by default.}

\item{ggplot_params}{An optional vector of strings of ggplot parameters to
the backend ggplot function. For example, c("ylab('')", "ylim(c(1,2))").
Default is NULL.}

\item{interactive}{A logical argument indicating whether the plots should be
interactive or not. Interactive plots are ggplots piped to ggplotly (for
now). Default is FALSE.}

\item{path}{The base directory of the trelliscope application. Default is
Downloads.}

\item{name}{The name of the display. Default is Trelliscope.}

\item{test_mode}{A logical to return a smaller trelliscope to confirm plot
and design. Default is FALSE.}

\item{test_example}{A vector of plot indices to return for test_mode. Default
is 1.}

\item{single_plot}{A TRUE/FALSE to indicate whether 1 plot (not a
trelliscope) should be returned. Default is FALSE.}

\item{...}{Additional arguments to be passed on to the trelli builder}
}
\value{
No return value, builds a trelliscope display of histograms that is stored in `path`
}
\description{
Specify a plot design and cognostics for the abundance histogram
  trelliscope. Main_effects grouping are ignored. Data must be grouped by
  edata_cname.
}
\examples{
\dontshow{if (requireNamespace("pmartRdata", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{

library(pmartRdata)

trelliData1 <- as.trelliData.edata(e_data = pep_edata,
                                   edata_cname = "Peptide",
                                   omics_type = "pepData")
# Transform the data
omicsData <- edata_transform(omicsData = pep_object, data_scale = "log2")

# Group the data by condition
omicsData <- group_designation(omicsData = omicsData, main_effects = c("Phenotype"))

# Apply the IMD ANOVA filter
imdanova_Filt <- imdanova_filter(omicsData = omicsData)
omicsData <- applyFilt(filter_object = imdanova_Filt, omicsData = omicsData,
                       min_nonmiss_anova = 2)

# Normalize my pepData
omicsData <- normalize_global(omicsData, "subset_fn" = "all", "norm_fn" = "median",
                             "apply_norm" = TRUE, "backtransform" = TRUE)

# Implement the IMD ANOVA method and compute all pairwise comparisons 
# (i.e. leave the `comparisons` argument NULL)
statRes <- imd_anova(omicsData = omicsData, test_method = 'combined')

# Generate the trelliData object
trelliData2 <- as.trelliData(omicsData = omicsData)
trelliData4 <- as.trelliData(omicsData = omicsData, statRes = statRes)

# Build the abundance histogram with an edata file. 
# Generate trelliData in as.trelliData.edata
trelli_panel_by(trelliData = trelliData1, panel = "Peptide") \%>\% 
   trelli_abundance_histogram(test_mode = TRUE, test_example = 1:10, path = tempdir())

# Build the abundance histogram with an omicsData object. 
# Generate trelliData in as.trelliData
trelli_panel_by(trelliData = trelliData2, panel = "Peptide") \%>\% 
   trelli_abundance_histogram(test_mode = TRUE, test_example = 1:10)
    
# Build the abundance histogram with an omicsData and statRes object. 
# Generate trelliData in as.trelliData.
trelli_panel_by(trelliData = trelliData4, panel = "Peptide") \%>\%
   trelli_abundance_histogram(
     test_mode = TRUE, test_example = 1:10, cognostics = "sample count", path = tempdir())
   
# Users can modify the plotting function with ggplot parameters and interactivity, 
# and can also select certain cognostics.     
trelli_panel_by(trelliData = trelliData1, panel = "Peptide") \%>\% 
   trelli_abundance_histogram(test_mode = TRUE, test_example = 1:10, 
     ggplot_params = c("ylab('')", "xlab('Abundance')"), interactive = TRUE,
     cognostics = c("mean abundance", "median abundance"), path = tempdir())  
 
\dontshow{closeAllConnections()}
}
\dontshow{\}) # examplesIf}
}
\author{
David Degnan, Lisa Bramer
}
