#' @name sprinkle_fn
#' @title Apply a function to a selection of cells
#' 
#' @description The pre-defined sprinkles do not always provide the 
#'   desired impact on the tables. Applying a function allows for 
#'   highly customized output without having to pre-process that data 
#'   frame.
#' 
#' @param x An object of class \code{dust}
#' @param rows Either a numeric vector of rows in the tabular object to be 
#'   modified or an object of class \code{call}.  When a \code{call}, 
#'   generated by \code{quote(expression)}, the expression resolves to 
#'   a logical vector the same length as the number of rows in the table.
#'   Sprinkles are applied to where the expression resolves to \code{TRUE}.
#' @param cols Either a numeric vector of columns in the tabular object to
#'   be modified, or a character vector of column names. A mixture of 
#'   character and numeric indices is permissible.
#' @param fn An object of class \code{call}. The function should act on 
#'   an object \code{value} (which is an internal column in the \code{dust}
#'   object). It is recommend to wrap the function call in \code{quote}.
#'   For example, \code{quote(pvalString(value))} or 
#'   \code{quote(format(value, nsmall = 3))}.
#' @param part A character string denoting which part of the table to modify.
#' @param fixed \code{logical(1)} indicating if the values in \code{rows} 
#'   and \code{cols} should be read as fixed coordinate pairs.  By default, 
#'   sprinkles are applied at the intersection of \code{rows} and \code{cols}, 
#'   meaning that the arguments do not have to share the same length.  
#'   When \code{fixed = TRUE}, they must share the same length.
#' @param recycle A \code{character} one that determines how sprinkles are 
#'   managed when the sprinkle input doesn't match the length of the region
#'   to be sprinkled.  By default, recycling is turned off.  Recycling 
#'   may be performed across rows first (left to right, top to bottom), 
#'   or down columns first (top to bottom, left to right).
#' @param ... Additional arguments to pass to other methods. Currently ignored.
#' 
#'
#' @details \code{dust} objects transform tabular objects so that each cell
#'   in the table comprises one row in the data frame of cell attributes.  
#'   The function to be applied needs to act on the \code{value} column of
#'   that data frame. 
#' 
#' @author Benjamin Nutter
#' 
#' @section Functional Requirements:
#' \enumerate{
#'  \item Correctly reassigns the appropriate elements \code{fn} column
#'    in the table part.
#'  \item Casts an error if \code{x} is not a \code{dust} object.
#'  \item Casts an error if \code{fn} is not a \code{call} object.
#'  \item Casts an error if \code{part} is not one of \code{"body"}, 
#'    \code{"head"}, \code{"foot"}, or \code{"interfoot"}
#'  \item Casts an error if \code{fixed} is not a \code{logical(1)}
#'  \item Casts an error if \code{recycle} is not one of \code{"none"},
#'    \code{"rows"}, or \code{"cols"}
#' }
#'
#' @export

sprinkle_fn <- function(x, rows = NULL, cols = NULL, fn = NULL, 
                        part = c("body", "head", "foot", "interfoot"),
                        fixed = FALSE,
                        recycle = c("none", "rows", "cols"), ...)
{
  UseMethod("sprinkle_fn")
}

#' @rdname sprinkle_fn
#' @export

sprinkle_fn.default <- function(x, rows = NULL, cols = NULL, fn = NULL,
                                part = c("body", "head", "foot", "interfoot"),
                                fixed = FALSE,
                                recycle = c("none", "rows", "cols", "columns"), 
                                ...)
{
  coll <- checkmate::makeAssertCollection()
  
  indices <- index_to_sprinkle(x = x, 
                               rows = rows, 
                               cols = cols, 
                               fixed = fixed,
                               part = part,
                               recycle = recycle,
                               coll = coll)
  
  sprinkle_fn_index_assert(fn = fn,
                           coll = coll)
  
  checkmate::reportAssertions(coll)
 
  if (is.null(fn)) return(x)
   
  part <- part[1]
  
  sprinkle_fn_index(x = x, 
                    indices = indices,
                    fn = fn,
                    part = part)
}

#' @rdname sprinkle_fn
#' @export

sprinkle_fn.dust_list <- function(x, rows = NULL, cols = NULL, fn = NULL,
                                  part = c("body", "head", "foot", "interfoot"),
                                  fixed = FALSE,
                                  recycle = c("none", "rows", "cols", "columns"),
                                  ...)
{
  structure(
    lapply(X = x,
           FUN = sprinkle_fn.default,
           rows = rows,
           cols = cols,
           fn = fn,
           part = part,
           fixed = fixed,
           recycle = recycle,
           ...),
    class = "dust_list"
  )
}

# Unexported Utility ------------------------------------------------

# These functions are to be used inside of the general `sprinkle` call
# When used inside `sprinkle`, the indices are already determined, 
# the only the `fn` argument needs to be validated. 
# The assert function is kept separate so it may be called earlier
# without attempting to perform the assignment.

sprinkle_fn_index_assert <- function(fn, coll)
{
  checkmate::assert_class(x = fn,
                          class = "call",
                          add = coll)
}

sprinkle_fn_index <- function(x, indices, fn, part)
{
  x[[part]][["fn"]][indices] <- deparse(fn)
  
  x
}
