<!-- README.md is generated from README.Rmd. Please edit that file -->

# pedquant

[![Travis build
status](https://travis-ci.org/ShichenXie/pedquant.svg?branch=master)](https://travis-ci.org/ShichenXie/pedquant)
[![CRAN
status](https://www.r-pkg.org/badges/version/pedquant)](https://cran.r-project.org/package=pedquant)
[![](http://cranlogs.r-pkg.org/badges/grand-total/pedquant)](https://cran.r-project.org/package=pedquant)
[![](http://cranlogs.r-pkg.org/badges/pedquant)](https://cran.r-project.org/package=pedquant)

`pedquant` (Public Economic Data and QUANTitative analysis) provides an
interface to access public economic and financial data for economic
research and quantitative analysis. The functions are grouped into three
main categories,

-   ed\_\* (economic data) functions load economic data from
    [NBS](http://www.stats.gov.cn/) and
    [FRED](https://fred.stlouisfed.org/);
-   md\_\* (market data) functions load the forex, money, bond, stock,
    future market data from public data sources, including 163, Sina, qq
    finance and etc.
-   pq\_\* (quantitative analysis) functions create technical
    indicators, visualization charts and industrial index etc for time
    series data.

The functions in this package are designed to write minimum codes for
some common tasks in quantitative analysis process. Since the parameters
to get data can be interactively specify, it’s very easy to start. The
loaded data have been carefully cleansed and provided in a unified
format.

`pedquant` package has advantages on multiple aspects, such as the
format of loaded data is a list of data frames, which can be easily
manipulated in [data.table](https://rdatatable.gitlab.io/data.table) or
[tidyverse](https://www.tidyverse.org) packages; high performance on
speed by using [data.table](https://rdatatable.gitlab.io/data.table) and
[TTR](https://github.com/joshuaulrich/TTR); and interactive charts by
using [echarts4r](https://echarts4r.john-coene.com). Similar works
including [tidyquant](https://github.com/business-science/tidyquant) or
[quantmod](https://github.com/joshuaulrich/quantmod).

## Installation

-   Install the release version of `pedquant` from CRAN with:

<!-- -->

    install.packages("pedquant")

-   Install the developing version of `pedquant` from
    [github](https://github.com/shichenXie/pedquant) with:

<!-- -->

    devtools::install_github("shichenxie/pedquant")

## Example

The following examples show you how to import data.

    library(pedquant)
    # loading data
    ## import eocnomic data
    dat1 = ed_fred('GDPCA')
    #> 1/1 GDPCA
    dat2 = ed_nbs(geo_type='nation', freq='quarterly', symbol='A010101')

    ## import market data
    FAAG = md_stock(c('META', 'AMZN', 'AAPL', 'GOOG'), date_range = '10y')
    #> 1/4 meta
    #> 2/4 amzn
    #> 3/4 aapl
    #> 4/4 goog
    INDX = md_stock(c('^000001','^399001'), date_range = '10y')
    #> 1/2 ^000001
    #> 2/2 ^399001

    # double moving average strategy
    ## add technical indicators
    data("dt_banks")
    dtbnkti = pq_addti(dt_banks, x='close_adj', sma=list(n=200), sma=list(n=50))

    ## crossover signals
    library(data.table)
    dtorders = copy(dtbnkti[['601988.SS']])[
       sma_50 %x>% sma_200, `:=`(type = 'buy',  prices = close_adj)
     ][sma_50 %x<% sma_200, `:=`(type = 'sell', prices = close_adj)
     ][order(date)
     ][, (c('type', 'prices')) := lapply(.SD, shift), .SDcols = c('type', 'prices')
     ][,.(symbol, name, date, close_adj, type, prices)]
    head(dtorders[!is.na(type)])
    #>       symbol     name       date close_adj type   prices
    #> 1: 601988.SS 中国银行 2015-09-28  3.283584 sell 3.309507
    #> 2: 601988.SS 中国银行 2016-09-05  3.148622  buy 3.157722
    #> 3: 601988.SS 中国银行 2018-05-03  3.638575 sell 3.629050
    #> 4: 601988.SS 中国银行 2018-12-26  3.630000  buy 3.610000

    # charting
    e = pq_plot(dt_banks,  y='close_adj', addti = list(sma=list(n=200), sma=list(n=50)), orders = dtorders[!is.na(type)])
    e[['601988.SS']]

<div id="htmlwidget-15e066dcd6813fc26426" style="width:100%;height:500px;" class="echarts4r html-widget"></div>
<script type="application/json" data-for="htmlwidget-15e066dcd6813fc26426">{"x":{"theme":"default","tl":false,"draw":true,"renderer":"canvas","events":[],"buttons":[],"opts":{"yAxis":[{"show":true,"type":"value","position":"right","axisLabel":{"rotate":90}}],"xAxis":[{"data":["2013-12-31","2014-01-02","2014-01-03","2014-01-06","2014-01-07","2014-01-08","2014-01-09","2014-01-10","2014-01-13","2014-01-14","2014-01-15","2014-01-16","2014-01-17","2014-01-20","2014-01-21","2014-01-22","2014-01-23","2014-01-24","2014-01-27","2014-01-28","2014-01-29","2014-01-30","2014-02-07","2014-02-10","2014-02-11","2014-02-12","2014-02-13","2014-02-14","2014-02-17","2014-02-18","2014-02-19","2014-02-20","2014-02-21","2014-02-24","2014-02-25","2014-02-26","2014-02-27","2014-02-28","2014-03-03","2014-03-04","2014-03-05","2014-03-06","2014-03-07","2014-03-10","2014-03-11","2014-03-12","2014-03-13","2014-03-14","2014-03-17","2014-03-18","2014-03-19","2014-03-20","2014-03-21","2014-03-24","2014-03-25","2014-03-26","2014-03-27","2014-03-28","2014-03-31","2014-04-01","2014-04-02","2014-04-03","2014-04-04","2014-04-08","2014-04-09","2014-04-10","2014-04-11","2014-04-14","2014-04-15","2014-04-16","2014-04-17","2014-04-18","2014-04-21","2014-04-22","2014-04-23","2014-04-24","2014-04-25","2014-04-28","2014-04-29","2014-04-30","2014-05-05","2014-05-06","2014-05-07","2014-05-08","2014-05-09","2014-05-12","2014-05-13","2014-05-14","2014-05-15","2014-05-16","2014-05-19","2014-05-20","2014-05-21","2014-05-22","2014-05-23","2014-05-26","2014-05-27","2014-05-28","2014-05-29","2014-05-30","2014-06-03","2014-06-04","2014-06-05","2014-06-06","2014-06-09","2014-06-10","2014-06-11","2014-06-12","2014-06-13","2014-06-16","2014-06-17","2014-06-18","2014-06-19","2014-06-20","2014-06-23","2014-06-24","2014-06-25","2014-06-26","2014-06-27","2014-06-30","2014-07-01","2014-07-02","2014-07-03","2014-07-04","2014-07-07","2014-07-08","2014-07-09","2014-07-10","2014-07-11","2014-07-14","2014-07-15","2014-07-16","2014-07-17","2014-07-18","2014-07-21","2014-07-22","2014-07-23","2014-07-24","2014-07-25","2014-07-28","2014-07-29","2014-07-30","2014-07-31","2014-08-01","2014-08-04","2014-08-05","2014-08-06","2014-08-07","2014-08-08","2014-08-11","2014-08-12","2014-08-13","2014-08-14","2014-08-15","2014-08-18","2014-08-19","2014-08-20","2014-08-21","2014-08-22","2014-08-25","2014-08-26","2014-08-27","2014-08-28","2014-08-29","2014-09-01","2014-09-02","2014-09-03","2014-09-04","2014-09-05","2014-09-09","2014-09-10","2014-09-11","2014-09-12","2014-09-15","2014-09-16","2014-09-17","2014-09-18","2014-09-19","2014-09-22","2014-09-23","2014-09-24","2014-09-25","2014-09-26","2014-09-29","2014-09-30","2014-10-08","2014-10-09","2014-10-10","2014-10-13","2014-10-14","2014-10-15","2014-10-16","2014-10-17","2014-10-20","2014-10-21","2014-10-22","2014-10-23","2014-10-24","2014-10-27","2014-10-28","2014-10-29","2014-10-30","2014-10-31","2014-11-03","2014-11-04","2014-11-05","2014-11-06","2014-11-07","2014-11-10","2014-11-11","2014-11-12","2014-11-13","2014-11-14","2014-11-17","2014-11-18","2014-11-19","2014-11-20","2014-11-21","2014-11-24","2014-11-25","2014-11-26","2014-11-27","2014-11-28","2014-12-01","2014-12-02","2014-12-03","2014-12-04","2014-12-05","2014-12-08","2014-12-09","2014-12-10","2014-12-11","2014-12-12","2014-12-15","2014-12-16","2014-12-17","2014-12-18","2014-12-19","2014-12-22","2014-12-23","2014-12-24","2014-12-25","2014-12-26","2014-12-29","2014-12-30","2014-12-31","2015-01-05","2015-01-06","2015-01-07","2015-01-08","2015-01-09","2015-01-12","2015-01-13","2015-01-14","2015-01-15","2015-01-16","2015-01-19","2015-01-20","2015-01-21","2015-01-22","2015-01-23","2015-01-26","2015-01-27","2015-01-28","2015-01-29","2015-01-30","2015-02-02","2015-02-03","2015-02-04","2015-02-05","2015-02-06","2015-02-09","2015-02-10","2015-02-11","2015-02-12","2015-02-13","2015-02-16","2015-02-17","2015-02-25","2015-02-26","2015-02-27","2015-03-02","2015-03-03","2015-03-04","2015-03-05","2015-03-06","2015-03-09","2015-03-10","2015-03-11","2015-03-12","2015-03-13","2015-03-16","2015-03-17","2015-03-18","2015-03-19","2015-03-20","2015-03-23","2015-03-24","2015-03-25","2015-03-26","2015-03-27","2015-03-30","2015-03-31","2015-04-01","2015-04-02","2015-04-03","2015-04-07","2015-04-08","2015-04-09","2015-04-10","2015-04-13","2015-04-14","2015-04-15","2015-04-16","2015-04-17","2015-04-20","2015-04-21","2015-04-22","2015-04-23","2015-04-24","2015-04-27","2015-04-28","2015-04-29","2015-04-30","2015-05-04","2015-05-05","2015-05-06","2015-05-07","2015-05-08","2015-05-11","2015-05-12","2015-05-13","2015-05-14","2015-05-15","2015-05-18","2015-05-19","2015-05-20","2015-05-21","2015-05-22","2015-05-25","2015-05-26","2015-05-27","2015-05-28","2015-05-29","2015-06-01","2015-06-02","2015-06-03","2015-06-04","2015-06-05","2015-06-08","2015-06-09","2015-06-10","2015-06-11","2015-06-12","2015-06-15","2015-06-16","2015-06-17","2015-06-18","2015-06-19","2015-06-23","2015-06-24","2015-06-25","2015-06-26","2015-06-29","2015-06-30","2015-07-01","2015-07-02","2015-07-03","2015-07-06","2015-07-07","2015-07-08","2015-07-09","2015-07-10","2015-07-13","2015-07-14","2015-07-15","2015-07-16","2015-07-17","2015-07-20","2015-07-21","2015-07-22","2015-07-23","2015-07-24","2015-07-27","2015-07-28","2015-07-29","2015-07-30","2015-07-31","2015-08-03","2015-08-04","2015-08-05","2015-08-06","2015-08-07","2015-08-10","2015-08-11","2015-08-12","2015-08-13","2015-08-14","2015-08-17","2015-08-18","2015-08-19","2015-08-20","2015-08-21","2015-08-24","2015-08-25","2015-08-26","2015-08-27","2015-08-28","2015-08-31","2015-09-01","2015-09-02","2015-09-07","2015-09-08","2015-09-09","2015-09-10","2015-09-11","2015-09-14","2015-09-15","2015-09-16","2015-09-17","2015-09-18","2015-09-21","2015-09-22","2015-09-23","2015-09-24","2015-09-25","2015-09-28","2015-09-29","2015-09-30","2015-10-08","2015-10-09","2015-10-12","2015-10-13","2015-10-14","2015-10-15","2015-10-16","2015-10-19","2015-10-20","2015-10-21","2015-10-22","2015-10-23","2015-10-26","2015-10-27","2015-10-28","2015-10-29","2015-10-30","2015-11-02","2015-11-03","2015-11-04","2015-11-05","2015-11-06","2015-11-09","2015-11-10","2015-11-11","2015-11-12","2015-11-13","2015-11-16","2015-11-17","2015-11-18","2015-11-19","2015-11-20","2015-11-23","2015-11-24","2015-11-25","2015-11-26","2015-11-27","2015-11-30","2015-12-01","2015-12-02","2015-12-03","2015-12-04","2015-12-07","2015-12-08","2015-12-09","2015-12-10","2015-12-11","2015-12-14","2015-12-15","2015-12-16","2015-12-17","2015-12-18","2015-12-21","2015-12-22","2015-12-23","2015-12-24","2015-12-25","2015-12-28","2015-12-29","2015-12-30","2015-12-31","2016-01-04","2016-01-05","2016-01-06","2016-01-07","2016-01-08","2016-01-11","2016-01-12","2016-01-13","2016-01-14","2016-01-15","2016-01-18","2016-01-19","2016-01-20","2016-01-21","2016-01-22","2016-01-25","2016-01-26","2016-01-27","2016-01-28","2016-01-29","2016-02-01","2016-02-02","2016-02-03","2016-02-04","2016-02-05","2016-02-15","2016-02-16","2016-02-17","2016-02-18","2016-02-19","2016-02-22","2016-02-23","2016-02-24","2016-02-25","2016-02-26","2016-02-29","2016-03-01","2016-03-02","2016-03-03","2016-03-04","2016-03-07","2016-03-08","2016-03-09","2016-03-10","2016-03-11","2016-03-14","2016-03-15","2016-03-16","2016-03-17","2016-03-18","2016-03-21","2016-03-22","2016-03-23","2016-03-24","2016-03-25","2016-03-28","2016-03-29","2016-03-30","2016-03-31","2016-04-01","2016-04-05","2016-04-06","2016-04-07","2016-04-08","2016-04-11","2016-04-12","2016-04-13","2016-04-14","2016-04-15","2016-04-18","2016-04-19","2016-04-20","2016-04-21","2016-04-22","2016-04-25","2016-04-26","2016-04-27","2016-04-28","2016-04-29","2016-05-03","2016-05-04","2016-05-05","2016-05-06","2016-05-09","2016-05-10","2016-05-11","2016-05-12","2016-05-13","2016-05-16","2016-05-17","2016-05-18","2016-05-19","2016-05-20","2016-05-23","2016-05-24","2016-05-25","2016-05-26","2016-05-27","2016-05-30","2016-05-31","2016-06-01","2016-06-02","2016-06-03","2016-06-06","2016-06-07","2016-06-08","2016-06-13","2016-06-14","2016-06-15","2016-06-16","2016-06-17","2016-06-20","2016-06-21","2016-06-22","2016-06-23","2016-06-24","2016-06-27","2016-06-28","2016-06-29","2016-06-30","2016-07-01","2016-07-04","2016-07-05","2016-07-06","2016-07-07","2016-07-08","2016-07-11","2016-07-12","2016-07-13","2016-07-14","2016-07-15","2016-07-18","2016-07-19","2016-07-20","2016-07-21","2016-07-22","2016-07-25","2016-07-26","2016-07-27","2016-07-28","2016-07-29","2016-08-01","2016-08-02","2016-08-03","2016-08-04","2016-08-05","2016-08-08","2016-08-09","2016-08-10","2016-08-11","2016-08-12","2016-08-15","2016-08-16","2016-08-17","2016-08-18","2016-08-19","2016-08-22","2016-08-23","2016-08-24","2016-08-25","2016-08-26","2016-08-29","2016-08-30","2016-08-31","2016-09-01","2016-09-02","2016-09-05","2016-09-06","2016-09-07","2016-09-08","2016-09-09","2016-09-12","2016-09-13","2016-09-14","2016-09-19","2016-09-20","2016-09-21","2016-09-22","2016-09-23","2016-09-26","2016-09-27","2016-09-28","2016-09-29","2016-09-30","2016-10-10","2016-10-11","2016-10-12","2016-10-13","2016-10-14","2016-10-17","2016-10-18","2016-10-19","2016-10-20","2016-10-21","2016-10-24","2016-10-25","2016-10-26","2016-10-27","2016-10-28","2016-10-31","2016-11-01","2016-11-02","2016-11-03","2016-11-04","2016-11-07","2016-11-08","2016-11-09","2016-11-10","2016-11-11","2016-11-14","2016-11-15","2016-11-16","2016-11-17","2016-11-18","2016-11-21","2016-11-22","2016-11-23","2016-11-24","2016-11-25","2016-11-28","2016-11-29","2016-11-30","2016-12-01","2016-12-02","2016-12-05","2016-12-06","2016-12-07","2016-12-08","2016-12-09","2016-12-12","2016-12-13","2016-12-14","2016-12-15","2016-12-16","2016-12-19","2016-12-20","2016-12-21","2016-12-22","2016-12-23","2016-12-26","2016-12-27","2016-12-28","2016-12-29","2016-12-30","2017-01-03","2017-01-04","2017-01-05","2017-01-06","2017-01-09","2017-01-10","2017-01-11","2017-01-12","2017-01-13","2017-01-16","2017-01-17","2017-01-18","2017-01-19","2017-01-20","2017-01-23","2017-01-24","2017-01-25","2017-01-26","2017-02-03","2017-02-06","2017-02-07","2017-02-08","2017-02-09","2017-02-10","2017-02-13","2017-02-14","2017-02-15","2017-02-16","2017-02-17","2017-02-20","2017-02-21","2017-02-22","2017-02-23","2017-02-24","2017-02-27","2017-02-28","2017-03-01","2017-03-02","2017-03-03","2017-03-06","2017-03-07","2017-03-08","2017-03-09","2017-03-10","2017-03-13","2017-03-14","2017-03-15","2017-03-16","2017-03-17","2017-03-20","2017-03-21","2017-03-22","2017-03-23","2017-03-24","2017-03-27","2017-03-28","2017-03-29","2017-03-30","2017-03-31","2017-04-05","2017-04-06","2017-04-07","2017-04-10","2017-04-11","2017-04-12","2017-04-13","2017-04-14","2017-04-17","2017-04-18","2017-04-19","2017-04-20","2017-04-21","2017-04-24","2017-04-25","2017-04-26","2017-04-27","2017-04-28","2017-05-02","2017-05-03","2017-05-04","2017-05-05","2017-05-08","2017-05-09","2017-05-10","2017-05-11","2017-05-12","2017-05-15","2017-05-16","2017-05-17","2017-05-18","2017-05-19","2017-05-22","2017-05-23","2017-05-24","2017-05-25","2017-05-26","2017-05-31","2017-06-01","2017-06-02","2017-06-05","2017-06-06","2017-06-07","2017-06-08","2017-06-09","2017-06-12","2017-06-13","2017-06-14","2017-06-15","2017-06-16","2017-06-19","2017-06-20","2017-06-21","2017-06-22","2017-06-23","2017-06-26","2017-06-27","2017-06-28","2017-06-29","2017-06-30","2017-07-03","2017-07-04","2017-07-05","2017-07-06","2017-07-07","2017-07-10","2017-07-11","2017-07-12","2017-07-13","2017-07-14","2017-07-17","2017-07-18","2017-07-19","2017-07-20","2017-07-21","2017-07-24","2017-07-25","2017-07-26","2017-07-27","2017-07-28","2017-07-31","2017-08-01","2017-08-02","2017-08-03","2017-08-04","2017-08-07","2017-08-08","2017-08-09","2017-08-10","2017-08-11","2017-08-14","2017-08-15","2017-08-16","2017-08-17","2017-08-18","2017-08-21","2017-08-22","2017-08-23","2017-08-24","2017-08-25","2017-08-28","2017-08-29","2017-08-30","2017-08-31","2017-09-01","2017-09-04","2017-09-05","2017-09-06","2017-09-07","2017-09-08","2017-09-11","2017-09-12","2017-09-13","2017-09-14","2017-09-15","2017-09-18","2017-09-19","2017-09-20","2017-09-21","2017-09-22","2017-09-25","2017-09-26","2017-09-27","2017-09-28","2017-09-29","2017-10-09","2017-10-10","2017-10-11","2017-10-12","2017-10-13","2017-10-16","2017-10-17","2017-10-18","2017-10-19","2017-10-20","2017-10-23","2017-10-24","2017-10-25","2017-10-26","2017-10-27","2017-10-30","2017-10-31","2017-11-01","2017-11-02","2017-11-03","2017-11-06","2017-11-07","2017-11-08","2017-11-09","2017-11-10","2017-11-13","2017-11-14","2017-11-15","2017-11-16","2017-11-17","2017-11-20","2017-11-21","2017-11-22","2017-11-23","2017-11-24","2017-11-27","2017-11-28","2017-11-29","2017-11-30","2017-12-01","2017-12-04","2017-12-05","2017-12-06","2017-12-07","2017-12-08","2017-12-11","2017-12-12","2017-12-13","2017-12-14","2017-12-15","2017-12-18","2017-12-19","2017-12-20","2017-12-21","2017-12-22","2017-12-25","2017-12-26","2017-12-27","2017-12-28","2017-12-29","2018-01-02","2018-01-03","2018-01-04","2018-01-05","2018-01-08","2018-01-09","2018-01-10","2018-01-11","2018-01-12","2018-01-15","2018-01-16","2018-01-17","2018-01-18","2018-01-19","2018-01-22","2018-01-23","2018-01-24","2018-01-25","2018-01-26","2018-01-29","2018-01-30","2018-01-31","2018-02-01","2018-02-02","2018-02-05","2018-02-06","2018-02-07","2018-02-08","2018-02-09","2018-02-12","2018-02-13","2018-02-14","2018-02-22","2018-02-23","2018-02-26","2018-02-27","2018-02-28","2018-03-01","2018-03-02","2018-03-05","2018-03-06","2018-03-07","2018-03-08","2018-03-09","2018-03-12","2018-03-13","2018-03-14","2018-03-15","2018-03-16","2018-03-19","2018-03-20","2018-03-21","2018-03-22","2018-03-23","2018-03-26","2018-03-27","2018-03-28","2018-03-29","2018-03-30","2018-04-02","2018-04-03","2018-04-04","2018-04-09","2018-04-10","2018-04-11","2018-04-12","2018-04-13","2018-04-16","2018-04-17","2018-04-18","2018-04-19","2018-04-20","2018-04-23","2018-04-24","2018-04-25","2018-04-26","2018-04-27","2018-05-02","2018-05-03","2018-05-04","2018-05-07","2018-05-08","2018-05-09","2018-05-10","2018-05-11","2018-05-14","2018-05-15","2018-05-16","2018-05-17","2018-05-18","2018-05-21","2018-05-22","2018-05-23","2018-05-24","2018-05-25","2018-05-28","2018-05-29","2018-05-30","2018-05-31","2018-06-01","2018-06-04","2018-06-05","2018-06-06","2018-06-07","2018-06-08","2018-06-11","2018-06-12","2018-06-13","2018-06-14","2018-06-15","2018-06-19","2018-06-20","2018-06-21","2018-06-22","2018-06-25","2018-06-26","2018-06-27","2018-06-28","2018-06-29","2018-07-02","2018-07-03","2018-07-04","2018-07-05","2018-07-06","2018-07-09","2018-07-10","2018-07-11","2018-07-12","2018-07-13","2018-07-16","2018-07-17","2018-07-18","2018-07-19","2018-07-20","2018-07-23","2018-07-24","2018-07-25","2018-07-26","2018-07-27","2018-07-30","2018-07-31","2018-08-01","2018-08-02","2018-08-03","2018-08-06","2018-08-07","2018-08-08","2018-08-09","2018-08-10","2018-08-13","2018-08-14","2018-08-15","2018-08-16","2018-08-17","2018-08-20","2018-08-21","2018-08-22","2018-08-23","2018-08-24","2018-08-27","2018-08-28","2018-08-29","2018-08-30","2018-08-31","2018-09-03","2018-09-04","2018-09-05","2018-09-06","2018-09-07","2018-09-10","2018-09-11","2018-09-12","2018-09-13","2018-09-14","2018-09-17","2018-09-18","2018-09-19","2018-09-20","2018-09-21","2018-09-25","2018-09-26","2018-09-27","2018-09-28","2018-10-08","2018-10-09","2018-10-10","2018-10-11","2018-10-12","2018-10-15","2018-10-16","2018-10-17","2018-10-18","2018-10-19","2018-10-22","2018-10-23","2018-10-24","2018-10-25","2018-10-26","2018-10-29","2018-10-30","2018-10-31","2018-11-01","2018-11-02","2018-11-05","2018-11-06","2018-11-07","2018-11-08","2018-11-09","2018-11-12","2018-11-13","2018-11-14","2018-11-15","2018-11-16","2018-11-19","2018-11-20","2018-11-21","2018-11-22","2018-11-23","2018-11-26","2018-11-27","2018-11-28","2018-11-29","2018-11-30","2018-12-03","2018-12-04","2018-12-05","2018-12-06","2018-12-07","2018-12-10","2018-12-11","2018-12-12","2018-12-13","2018-12-14","2018-12-17","2018-12-18","2018-12-19","2018-12-20","2018-12-21","2018-12-24","2018-12-25","2018-12-26","2018-12-27","2018-12-28"],"type":"time","boundaryGap":true}],"legend":{"data":["close_adj","sma_200","sma_50"],"show":true,"type":"plain","orient":"vertical","left":"30","top":"30"},"series":[{"data":[{"value":["2013-12-31","2.020824"]},{"value":["2014-01-02","2.013111"]},{"value":["2014-01-03","1.974545"]},{"value":["2014-01-06","1.951406"]},{"value":["2014-01-07","1.943693"]},{"value":["2014-01-08","1.935980"]},{"value":["2014-01-09","1.928267"]},{"value":["2014-01-10","1.920554"]},{"value":["2014-01-13","1.943693"]},{"value":["2014-01-14","1.935980"]},{"value":["2014-01-15","1.935980"]},{"value":["2014-01-16","1.928267"]},{"value":["2014-01-17","1.912841"]},{"value":["2014-01-20","1.905128"]},{"value":["2014-01-21","1.905128"]},{"value":["2014-01-22","1.935980"]},{"value":["2014-01-23","1.912841"]},{"value":["2014-01-24","1.920554"]},{"value":["2014-01-27","1.912841"]},{"value":["2014-01-28","1.920554"]},{"value":["2014-01-29","1.951406"]},{"value":["2014-01-30","1.935980"]},{"value":["2014-02-07","1.920554"]},{"value":["2014-02-10","1.935980"]},{"value":["2014-02-11","1.966832"]},{"value":["2014-02-12","1.959119"]},{"value":["2014-02-13","1.966832"]},{"value":["2014-02-14","1.959119"]},{"value":["2014-02-17","1.966832"]},{"value":["2014-02-18","1.959119"]},{"value":["2014-02-19","2.028537"]},{"value":["2014-02-20","2.013111"]},{"value":["2014-02-21","1.989971"]},{"value":["2014-02-24","1.951406"]},{"value":["2014-02-25","1.959119"]},{"value":["2014-02-26","1.935980"]},{"value":["2014-02-27","1.951406"]},{"value":["2014-02-28","1.935980"]},{"value":["2014-03-03","1.928267"]},{"value":["2014-03-04","1.928267"]},{"value":["2014-03-05","1.920554"]},{"value":["2014-03-06","1.928267"]},{"value":["2014-03-07","1.920554"]},{"value":["2014-03-10","1.897415"]},{"value":["2014-03-11","1.897415"]},{"value":["2014-03-12","1.889702"]},{"value":["2014-03-13","1.920554"]},{"value":["2014-03-14","1.905128"]},{"value":["2014-03-17","1.912841"]},{"value":["2014-03-18","1.905128"]},{"value":["2014-03-19","1.905128"]},{"value":["2014-03-20","1.897415"]},{"value":["2014-03-21","1.951406"]},{"value":["2014-03-24","1.959119"]},{"value":["2014-03-25","1.959119"]},{"value":["2014-03-26","1.959119"]},{"value":["2014-03-27","1.974545"]},{"value":["2014-03-28","1.989971"]},{"value":["2014-03-31","1.997685"]},{"value":["2014-04-01","1.997685"]},{"value":["2014-04-02","2.005398"]},{"value":["2014-04-03","1.989971"]},{"value":["2014-04-04","2.005398"]},{"value":["2014-04-08","2.036250"]},{"value":["2014-04-09","2.028537"]},{"value":["2014-04-10","2.059389"]},{"value":["2014-04-11","2.067102"]},{"value":["2014-04-14","2.059389"]},{"value":["2014-04-15","2.020824"]},{"value":["2014-04-16","2.028537"]},{"value":["2014-04-17","2.028537"]},{"value":["2014-04-18","2.028537"]},{"value":["2014-04-21","2.005398"]},{"value":["2014-04-22","2.028537"]},{"value":["2014-04-23","2.013111"]},{"value":["2014-04-24","2.005398"]},{"value":["2014-04-25","2.013111"]},{"value":["2014-04-28","2.020824"]},{"value":["2014-04-29","2.013111"]},{"value":["2014-04-30","2.005398"]},{"value":["2014-05-05","2.005398"]},{"value":["2014-05-06","1.997685"]},{"value":["2014-05-07","2.005398"]},{"value":["2014-05-08","2.013111"]},{"value":["2014-05-09","2.013111"]},{"value":["2014-05-12","2.036250"]},{"value":["2014-05-13","2.043963"]},{"value":["2014-05-14","2.043963"]},{"value":["2014-05-15","2.043963"]},{"value":["2014-05-16","2.043963"]},{"value":["2014-05-19","2.043963"]},{"value":["2014-05-20","2.043963"]},{"value":["2014-05-21","2.051676"]},{"value":["2014-05-22","2.059389"]},{"value":["2014-05-23","2.067102"]},{"value":["2014-05-26","2.074815"]},{"value":["2014-05-27","2.067102"]},{"value":["2014-05-28","2.074815"]},{"value":["2014-05-29","2.074815"]},{"value":["2014-05-30","2.059389"]},{"value":["2014-06-03","2.097954"]},{"value":["2014-06-04","2.097954"]},{"value":["2014-06-05","2.105667"]},{"value":["2014-06-06","2.082528"]},{"value":["2014-06-09","2.097954"]},{"value":["2014-06-10","2.113381"]},{"value":["2014-06-11","2.113381"]},{"value":["2014-06-12","2.128807"]},{"value":["2014-06-13","2.159659"]},{"value":["2014-06-16","2.167372"]},{"value":["2014-06-17","2.144233"]},{"value":["2014-06-18","2.128807"]},{"value":["2014-06-19","2.097954"]},{"value":["2014-06-20","2.121094"]},{"value":["2014-06-23","2.105667"]},{"value":["2014-06-24","2.121094"]},{"value":["2014-06-25","2.136520"]},{"value":["2014-06-26","2.136520"]},{"value":["2014-06-27","2.111580"]},{"value":["2014-06-30","2.119893"]},{"value":["2014-07-01","2.128206"]},{"value":["2014-07-02","2.144833"]},{"value":["2014-07-03","2.153146"]},{"value":["2014-07-04","2.153146"]},{"value":["2014-07-07","2.144833"]},{"value":["2014-07-08","2.144833"]},{"value":["2014-07-09","2.128206"]},{"value":["2014-07-10","2.111580"]},{"value":["2014-07-11","2.119893"]},{"value":["2014-07-14","2.128206"]},{"value":["2014-07-15","2.136520"]},{"value":["2014-07-16","2.136520"]},{"value":["2014-07-17","2.136520"]},{"value":["2014-07-18","2.136520"]},{"value":["2014-07-21","2.136520"]},{"value":["2014-07-22","2.136520"]},{"value":["2014-07-23","2.128206"]},{"value":["2014-07-24","2.153146"]},{"value":["2014-07-25","2.169773"]},{"value":["2014-07-28","2.244593"]},{"value":["2014-07-29","2.261219"]},{"value":["2014-07-30","2.227966"]},{"value":["2014-07-31","2.252906"]},{"value":["2014-08-01","2.252906"]},{"value":["2014-08-04","2.269533"]},{"value":["2014-08-05","2.261219"]},{"value":["2014-08-06","2.252906"]},{"value":["2014-08-07","2.227966"]},{"value":["2014-08-08","2.236279"]},{"value":["2014-08-11","2.252906"]},{"value":["2014-08-12","2.244593"]},{"value":["2014-08-13","2.244593"]},{"value":["2014-08-14","2.236279"]},{"value":["2014-08-15","2.244593"]},{"value":["2014-08-18","2.244593"]},{"value":["2014-08-19","2.236279"]},{"value":["2014-08-20","2.244593"]},{"value":["2014-08-21","2.236279"]},{"value":["2014-08-22","2.252906"]},{"value":["2014-08-25","2.244593"]},{"value":["2014-08-26","2.227966"]},{"value":["2014-08-27","2.219653"]},{"value":["2014-08-28","2.186400"]},{"value":["2014-08-29","2.227966"]},{"value":["2014-09-01","2.211340"]},{"value":["2014-09-02","2.236279"]},{"value":["2014-09-03","2.252906"]},{"value":["2014-09-04","2.261219"]},{"value":["2014-09-05","2.277846"]},{"value":["2014-09-09","2.269533"]},{"value":["2014-09-10","2.252906"]},{"value":["2014-09-11","2.244593"]},{"value":["2014-09-12","2.261219"]},{"value":["2014-09-15","2.252906"]},{"value":["2014-09-16","2.261219"]},{"value":["2014-09-17","2.244593"]},{"value":["2014-09-18","2.261219"]},{"value":["2014-09-19","2.261219"]},{"value":["2014-09-22","2.219653"]},{"value":["2014-09-23","2.227966"]},{"value":["2014-09-24","2.244593"]},{"value":["2014-09-25","2.244593"]},{"value":["2014-09-26","2.236279"]},{"value":["2014-09-29","2.227966"]},{"value":["2014-09-30","2.244593"]},{"value":["2014-10-08","2.236279"]},{"value":["2014-10-09","2.244593"]},{"value":["2014-10-10","2.236279"]},{"value":["2014-10-13","2.219653"]},{"value":["2014-10-14","2.211340"]},{"value":["2014-10-15","2.219653"]},{"value":["2014-10-16","2.219653"]},{"value":["2014-10-17","2.219653"]},{"value":["2014-10-20","2.227966"]},{"value":["2014-10-21","2.219653"]},{"value":["2014-10-22","2.227966"]},{"value":["2014-10-23","2.219653"]},{"value":["2014-10-24","2.211340"]},{"value":["2014-10-27","2.169773"]},{"value":["2014-10-28","2.194713"]},{"value":["2014-10-29","2.236279"]},{"value":["2014-10-30","2.252906"]},{"value":["2014-10-31","2.327726"]},{"value":["2014-11-03","2.311099"]},{"value":["2014-11-04","2.319412"]},{"value":["2014-11-05","2.294473"]},{"value":["2014-11-06","2.286159"]},{"value":["2014-11-07","2.302786"]},{"value":["2014-11-10","2.377606"]},{"value":["2014-11-11","2.618692"]},{"value":["2014-11-12","2.560498"]},{"value":["2014-11-13","2.577125"]},{"value":["2014-11-14","2.568812"]},{"value":["2014-11-17","2.510619"]},{"value":["2014-11-18","2.435799"]},{"value":["2014-11-19","2.427485"]},{"value":["2014-11-20","2.427485"]},{"value":["2014-11-21","2.452425"]},{"value":["2014-11-24","2.444112"]},{"value":["2014-11-25","2.469052"]},{"value":["2014-11-26","2.535558"]},{"value":["2014-11-27","2.560498"]},{"value":["2014-11-28","2.768331"]},{"value":["2014-12-01","2.809898"]},{"value":["2014-12-02","2.959537"]},{"value":["2014-12-03","2.834837"]},{"value":["2014-12-04","2.934597"]},{"value":["2014-12-05","3.067610"]},{"value":["2014-12-08","3.042670"]},{"value":["2014-12-09","2.768331"]},{"value":["2014-12-10","2.818211"]},{"value":["2014-12-11","2.776644"]},{"value":["2014-12-12","2.776644"]},{"value":["2014-12-15","2.735078"]},{"value":["2014-12-16","2.826524"]},{"value":["2014-12-17","2.942910"]},{"value":["2014-12-18","2.876404"]},{"value":["2014-12-19","2.942910"]},{"value":["2014-12-22","3.233876"]},{"value":["2014-12-23","3.092550"]},{"value":["2014-12-24","2.967850"]},{"value":["2014-12-25","3.092550"]},{"value":["2014-12-26","3.175683"]},{"value":["2014-12-29","3.200623"]},{"value":["2014-12-30","3.275443"]},{"value":["2014-12-31","3.450022"]},{"value":["2015-01-05","3.674481"]},{"value":["2015-01-06","3.790868"]},{"value":["2015-01-07","3.774241"]},{"value":["2015-01-08","3.599662"]},{"value":["2015-01-09","3.716048"]},{"value":["2015-01-12","3.657855"]},{"value":["2015-01-13","3.691108"]},{"value":["2015-01-14","3.815808"]},{"value":["2015-01-15","4.048580"]},{"value":["2015-01-16","4.140027"]},{"value":["2015-01-19","3.724361"]},{"value":["2015-01-20","3.699421"]},{"value":["2015-01-21","4.065207"]},{"value":["2015-01-22","4.023640"]},{"value":["2015-01-23","4.023640"]},{"value":["2015-01-26","3.973760"]},{"value":["2015-01-27","3.849061"]},{"value":["2015-01-28","3.782554"]},{"value":["2015-01-29","3.657855"]},{"value":["2015-01-30","3.657855"]},{"value":["2015-02-02","3.433396"]},{"value":["2015-02-03","3.558095"]},{"value":["2015-02-04","3.433396"]},{"value":["2015-02-05","3.433396"]},{"value":["2015-02-06","3.333636"]},{"value":["2015-02-09","3.350262"]},{"value":["2015-02-10","3.408456"]},{"value":["2015-02-11","3.383516"]},{"value":["2015-02-12","3.375202"]},{"value":["2015-02-13","3.416769"]},{"value":["2015-02-16","3.400142"]},{"value":["2015-02-17","3.408456"]},{"value":["2015-02-25","3.366889"]},{"value":["2015-02-26","3.441709"]},{"value":["2015-02-27","3.416769"]},{"value":["2015-03-02","3.416769"]},{"value":["2015-03-03","3.300383"]},{"value":["2015-03-04","3.250503"]},{"value":["2015-03-05","3.159056"]},{"value":["2015-03-06","3.183996"]},{"value":["2015-03-09","3.350262"]},{"value":["2015-03-10","3.267129"]},{"value":["2015-03-11","3.333636"]},{"value":["2015-03-12","3.474962"]},{"value":["2015-03-13","3.450022"]},{"value":["2015-03-16","3.474962"]},{"value":["2015-03-17","3.499902"]},{"value":["2015-03-18","3.599662"]},{"value":["2015-03-19","3.549782"]},{"value":["2015-03-20","3.641228"]},{"value":["2015-03-23","3.682795"]},{"value":["2015-03-24","3.691108"]},{"value":["2015-03-25","3.574722"]},{"value":["2015-03-26","3.583035"]},{"value":["2015-03-27","3.591348"]},{"value":["2015-03-30","3.707735"]},{"value":["2015-03-31","3.641228"]},{"value":["2015-04-01","3.666168"]},{"value":["2015-04-02","3.624602"]},{"value":["2015-04-03","3.632915"]},{"value":["2015-04-07","3.716048"]},{"value":["2015-04-08","3.765928"]},{"value":["2015-04-09","3.732675"]},{"value":["2015-04-10","3.790868"]},{"value":["2015-04-13","3.890627"]},{"value":["2015-04-14","3.849061"]},{"value":["2015-04-15","3.915567"]},{"value":["2015-04-16","3.990387"]},{"value":["2015-04-17","4.098460"]},{"value":["2015-04-20","3.940507"]},{"value":["2015-04-21","3.973760"]},{"value":["2015-04-22","4.040267"]},{"value":["2015-04-23","3.990387"]},{"value":["2015-04-24","3.874001"]},{"value":["2015-04-27","3.998700"]},{"value":["2015-04-28","4.206533"]},{"value":["2015-04-29","4.131713"]},{"value":["2015-04-30","4.023640"]},{"value":["2015-05-04","4.015327"]},{"value":["2015-05-05","3.815808"]},{"value":["2015-05-06","3.749301"]},{"value":["2015-05-07","3.682795"]},{"value":["2015-05-08","3.732675"]},{"value":["2015-05-11","3.790868"]},{"value":["2015-05-12","3.807494"]},{"value":["2015-05-13","3.790868"]},{"value":["2015-05-14","3.782554"]},{"value":["2015-05-15","3.691108"]},{"value":["2015-05-18","3.599662"]},{"value":["2015-05-19","3.707735"]},{"value":["2015-05-20","3.691108"]},{"value":["2015-05-21","3.691108"]},{"value":["2015-05-22","3.799181"]},{"value":["2015-05-25","3.882314"]},{"value":["2015-05-26","3.890627"]},{"value":["2015-05-27","3.865687"]},{"value":["2015-05-28","3.641228"]},{"value":["2015-05-29","3.599662"]},{"value":["2015-06-01","3.740988"]},{"value":["2015-06-02","3.724361"]},{"value":["2015-06-03","3.724361"]},{"value":["2015-06-04","3.915567"]},{"value":["2015-06-05","3.907254"]},{"value":["2015-06-08","4.297979"]},{"value":["2015-06-09","4.397739"]},{"value":["2015-06-10","4.281353"]},{"value":["2015-06-11","4.164967"]},{"value":["2015-06-12","4.239786"]},{"value":["2015-06-15","4.123400"]},{"value":["2015-06-16","4.015327"]},{"value":["2015-06-17","4.115087"]},{"value":["2015-06-18","3.932194"]},{"value":["2015-06-19","3.765928"]},{"value":["2015-06-23","3.923881"]},{"value":["2015-06-24","3.990387"]},{"value":["2015-06-25","3.940507"]},{"value":["2015-06-26","3.765928"]},{"value":["2015-06-29","3.790868"]},{"value":["2015-06-30","4.065207"]},{"value":["2015-07-01","3.948821"]},{"value":["2015-07-02","4.164967"]},{"value":["2015-07-03","3.853890"]},{"value":["2015-07-06","4.234095"]},{"value":["2015-07-07","4.657504"]},{"value":["2015-07-08","4.830324"]},{"value":["2015-07-09","4.838965"]},{"value":["2015-07-10","4.778478"]},{"value":["2015-07-13","4.501966"]},{"value":["2015-07-14","4.519248"]},{"value":["2015-07-15","4.640222"]},{"value":["2015-07-16","4.519248"]},{"value":["2015-07-17","4.467402"]},{"value":["2015-07-20","4.458761"]},{"value":["2015-07-21","4.389633"]},{"value":["2015-07-22","4.355069"]},{"value":["2015-07-23","4.372351"]},{"value":["2015-07-24","4.329146"]},{"value":["2015-07-27","4.000787"]},{"value":["2015-07-28","4.035351"]},{"value":["2015-07-29","4.000787"]},{"value":["2015-07-30","3.905736"]},{"value":["2015-07-31","3.845249"]},{"value":["2015-08-03","3.845249"]},{"value":["2015-08-04","3.871172"]},{"value":["2015-08-05","3.793403"]},{"value":["2015-08-06","3.767480"]},{"value":["2015-08-07","3.793403"]},{"value":["2015-08-10","3.879813"]},{"value":["2015-08-11","3.845249"]},{"value":["2015-08-12","3.784762"]},{"value":["2015-08-13","3.793403"]},{"value":["2015-08-14","3.776121"]},{"value":["2015-08-17","3.741557"]},{"value":["2015-08-18","3.681070"]},{"value":["2015-08-19","3.629224"]},{"value":["2015-08-20","3.611942"]},{"value":["2015-08-21","3.586019"]},{"value":["2015-08-24","3.283584"]},{"value":["2015-08-25","3.015712"]},{"value":["2015-08-26","3.162609"]},{"value":["2015-08-27","3.361353"]},{"value":["2015-08-28","3.361353"]},{"value":["2015-08-31","3.335430"]},{"value":["2015-09-01","3.499609"]},{"value":["2015-09-02","3.663788"]},{"value":["2015-09-07","3.421840"]},{"value":["2015-09-08","3.439122"]},{"value":["2015-09-09","3.456404"]},{"value":["2015-09-10","3.456404"]},{"value":["2015-09-11","3.395917"]},{"value":["2015-09-14","3.395917"]},{"value":["2015-09-15","3.378635"]},{"value":["2015-09-16","3.404558"]},{"value":["2015-09-17","3.361353"]},{"value":["2015-09-18","3.335430"]},{"value":["2015-09-21","3.326789"]},{"value":["2015-09-22","3.344071"]},{"value":["2015-09-23","3.283584"]},{"value":["2015-09-24","3.274943"]},{"value":["2015-09-25","3.309507"]},{"value":["2015-09-28","3.283584"]},{"value":["2015-09-29","3.214455"]},{"value":["2015-09-30","3.214455"]},{"value":["2015-10-08","3.266302"]},{"value":["2015-10-09","3.283584"]},{"value":["2015-10-12","3.344071"]},{"value":["2015-10-13","3.318148"]},{"value":["2015-10-14","3.326789"]},{"value":["2015-10-15","3.361353"]},{"value":["2015-10-16","3.387276"]},{"value":["2015-10-19","3.378635"]},{"value":["2015-10-20","3.378635"]},{"value":["2015-10-21","3.482327"]},{"value":["2015-10-22","3.404558"]},{"value":["2015-10-23","3.413199"]},{"value":["2015-10-26","3.404558"]},{"value":["2015-10-27","3.387276"]},{"value":["2015-10-28","3.378635"]},{"value":["2015-10-29","3.361353"]},{"value":["2015-10-30","3.352712"]},{"value":["2015-11-02","3.326789"]},{"value":["2015-11-03","3.292225"]},{"value":["2015-11-04","3.395917"]},{"value":["2015-11-05","3.456404"]},{"value":["2015-11-06","3.465045"]},{"value":["2015-11-09","3.620583"]},{"value":["2015-11-10","3.586019"]},{"value":["2015-11-11","3.542814"]},{"value":["2015-11-12","3.499609"]},{"value":["2015-11-13","3.473686"]},{"value":["2015-11-16","3.499609"]},{"value":["2015-11-17","3.508250"]},{"value":["2015-11-18","3.516891"]},{"value":["2015-11-19","3.525532"]},{"value":["2015-11-20","3.499609"]},{"value":["2015-11-23","3.473686"]},{"value":["2015-11-24","3.490968"]},{"value":["2015-11-25","3.525532"]},{"value":["2015-11-26","3.516891"]},{"value":["2015-11-27","3.413199"]},{"value":["2015-11-30","3.551455"]},{"value":["2015-12-01","3.473686"]},{"value":["2015-12-02","3.611942"]},{"value":["2015-12-03","3.750198"]},{"value":["2015-12-04","3.603301"]},{"value":["2015-12-07","3.560096"]},{"value":["2015-12-08","3.508250"]},{"value":["2015-12-09","3.551455"]},{"value":["2015-12-10","3.516891"]},{"value":["2015-12-11","3.490968"]},{"value":["2015-12-14","3.542814"]},{"value":["2015-12-15","3.490968"]},{"value":["2015-12-16","3.490968"]},{"value":["2015-12-17","3.516891"]},{"value":["2015-12-18","3.525532"]},{"value":["2015-12-21","3.560096"]},{"value":["2015-12-22","3.534173"]},{"value":["2015-12-23","3.525532"]},{"value":["2015-12-24","3.508250"]},{"value":["2015-12-25","3.508250"]},{"value":["2015-12-28","3.473686"]},{"value":["2015-12-29","3.490968"]},{"value":["2015-12-30","3.482327"]},{"value":["2015-12-31","3.465045"]},{"value":["2016-01-04","3.344071"]},{"value":["2016-01-05","3.335430"]},{"value":["2016-01-06","3.361353"]},{"value":["2016-01-07","3.274943"]},{"value":["2016-01-08","3.283584"]},{"value":["2016-01-11","3.153968"]},{"value":["2016-01-12","3.136686"]},{"value":["2016-01-13","3.093481"]},{"value":["2016-01-14","3.102122"]},{"value":["2016-01-15","3.007071"]},{"value":["2016-01-18","2.989789"]},{"value":["2016-01-19","3.050276"]},{"value":["2016-01-20","2.998430"]},{"value":["2016-01-21","2.963866"]},{"value":["2016-01-22","2.981148"]},{"value":["2016-01-25","2.972507"]},{"value":["2016-01-26","2.799687"]},{"value":["2016-01-27","2.782405"]},{"value":["2016-01-28","2.730559"]},{"value":["2016-01-29","2.782405"]},{"value":["2016-02-01","2.739200"]},{"value":["2016-02-02","2.765123"]},{"value":["2016-02-03","2.739200"]},{"value":["2016-02-04","2.756482"]},{"value":["2016-02-05","2.747841"]},{"value":["2016-02-15","2.704636"]},{"value":["2016-02-16","2.791046"]},{"value":["2016-02-17","2.799687"]},{"value":["2016-02-18","2.791046"]},{"value":["2016-02-19","2.782405"]},{"value":["2016-02-22","2.842892"]},{"value":["2016-02-23","2.799687"]},{"value":["2016-02-24","2.799687"]},{"value":["2016-02-25","2.695995"]},{"value":["2016-02-26","2.721918"]},{"value":["2016-02-29","2.739200"]},{"value":["2016-03-01","2.765123"]},{"value":["2016-03-02","2.816969"]},{"value":["2016-03-03","2.834251"]},{"value":["2016-03-04","2.937943"]},{"value":["2016-03-07","2.929302"]},{"value":["2016-03-08","2.937943"]},{"value":["2016-03-09","2.963866"]},{"value":["2016-03-10","2.894738"]},{"value":["2016-03-11","2.937943"]},{"value":["2016-03-14","2.937943"]},{"value":["2016-03-15","2.920661"]},{"value":["2016-03-16","2.972507"]},{"value":["2016-03-17","2.963866"]},{"value":["2016-03-18","2.963866"]},{"value":["2016-03-21","2.963866"]},{"value":["2016-03-22","2.937943"]},{"value":["2016-03-23","2.929302"]},{"value":["2016-03-24","2.894738"]},{"value":["2016-03-25","2.903379"]},{"value":["2016-03-28","2.894738"]},{"value":["2016-03-29","2.886097"]},{"value":["2016-03-30","2.937943"]},{"value":["2016-03-31","2.937943"]},{"value":["2016-04-01","2.946584"]},{"value":["2016-04-05","2.963866"]},{"value":["2016-04-06","2.955225"]},{"value":["2016-04-07","2.920661"]},{"value":["2016-04-08","2.903379"]},{"value":["2016-04-11","2.929302"]},{"value":["2016-04-12","2.929302"]},{"value":["2016-04-13","2.955225"]},{"value":["2016-04-14","2.955225"]},{"value":["2016-04-15","2.955225"]},{"value":["2016-04-18","2.929302"]},{"value":["2016-04-19","2.920661"]},{"value":["2016-04-20","2.903379"]},{"value":["2016-04-21","2.886097"]},{"value":["2016-04-22","2.894738"]},{"value":["2016-04-25","2.894738"]},{"value":["2016-04-26","2.903379"]},{"value":["2016-04-27","2.894738"]},{"value":["2016-04-28","2.912020"]},{"value":["2016-04-29","2.894738"]},{"value":["2016-05-03","2.912020"]},{"value":["2016-05-04","2.912020"]},{"value":["2016-05-05","2.912020"]},{"value":["2016-05-06","2.877456"]},{"value":["2016-05-09","2.834251"]},{"value":["2016-05-10","2.825610"]},{"value":["2016-05-11","2.825610"]},{"value":["2016-05-12","2.808328"]},{"value":["2016-05-13","2.808328"]},{"value":["2016-05-16","2.808328"]},{"value":["2016-05-17","2.799687"]},{"value":["2016-05-18","2.799687"]},{"value":["2016-05-19","2.791046"]},{"value":["2016-05-20","2.799687"]},{"value":["2016-05-23","2.816969"]},{"value":["2016-05-24","2.799687"]},{"value":["2016-05-25","2.791046"]},{"value":["2016-05-26","2.791046"]},{"value":["2016-05-27","2.799687"]},{"value":["2016-05-30","2.825610"]},{"value":["2016-05-31","2.903379"]},{"value":["2016-06-01","2.886097"]},{"value":["2016-06-02","2.877456"]},{"value":["2016-06-03","2.886097"]},{"value":["2016-06-06","2.886097"]},{"value":["2016-06-07","2.894738"]},{"value":["2016-06-08","2.903379"]},{"value":["2016-06-13","2.851533"]},{"value":["2016-06-14","2.868815"]},{"value":["2016-06-15","2.868815"]},{"value":["2016-06-16","2.851533"]},{"value":["2016-06-17","2.860174"]},{"value":["2016-06-20","2.868815"]},{"value":["2016-06-21","2.886097"]},{"value":["2016-06-22","2.903379"]},{"value":["2016-06-23","2.912020"]},{"value":["2016-06-24","2.875620"]},{"value":["2016-06-27","2.893820"]},{"value":["2016-06-28","2.884720"]},{"value":["2016-06-29","2.902920"]},{"value":["2016-06-30","2.921120"]},{"value":["2016-07-01","2.930220"]},{"value":["2016-07-04","2.957520"]},{"value":["2016-07-05","2.957520"]},{"value":["2016-07-06","2.957520"]},{"value":["2016-07-07","2.966621"]},{"value":["2016-07-08","2.939320"]},{"value":["2016-07-11","2.957520"]},{"value":["2016-07-12","3.003021"]},{"value":["2016-07-13","3.039421"]},{"value":["2016-07-14","3.021221"]},{"value":["2016-07-15","3.030321"]},{"value":["2016-07-18","3.039421"]},{"value":["2016-07-19","3.021221"]},{"value":["2016-07-20","3.003021"]},{"value":["2016-07-21","3.003021"]},{"value":["2016-07-22","3.003021"]},{"value":["2016-07-25","3.030321"]},{"value":["2016-07-26","3.057621"]},{"value":["2016-07-27","3.039421"]},{"value":["2016-07-28","3.030321"]},{"value":["2016-07-29","3.039421"]},{"value":["2016-08-01","3.048521"]},{"value":["2016-08-02","3.057621"]},{"value":["2016-08-03","3.039421"]},{"value":["2016-08-04","3.021221"]},{"value":["2016-08-05","3.030321"]},{"value":["2016-08-08","3.039421"]},{"value":["2016-08-09","3.057621"]},{"value":["2016-08-10","3.048521"]},{"value":["2016-08-11","3.094021"]},{"value":["2016-08-12","3.185022"]},{"value":["2016-08-15","3.239622"]},{"value":["2016-08-16","3.139522"]},{"value":["2016-08-17","3.130422"]},{"value":["2016-08-18","3.112222"]},{"value":["2016-08-19","3.121322"]},{"value":["2016-08-22","3.139522"]},{"value":["2016-08-23","3.148622"]},{"value":["2016-08-24","3.148622"]},{"value":["2016-08-25","3.148622"]},{"value":["2016-08-26","3.139522"]},{"value":["2016-08-29","3.121322"]},{"value":["2016-08-30","3.139522"]},{"value":["2016-08-31","3.157722"]},{"value":["2016-09-01","3.166822"]},{"value":["2016-09-02","3.157722"]},{"value":["2016-09-05","3.148622"]},{"value":["2016-09-06","3.148622"]},{"value":["2016-09-07","3.139522"]},{"value":["2016-09-08","3.139522"]},{"value":["2016-09-09","3.130422"]},{"value":["2016-09-12","3.075821"]},{"value":["2016-09-13","3.075821"]},{"value":["2016-09-14","3.039421"]},{"value":["2016-09-19","3.048521"]},{"value":["2016-09-20","3.039421"]},{"value":["2016-09-21","3.075821"]},{"value":["2016-09-22","3.084921"]},{"value":["2016-09-23","3.094021"]},{"value":["2016-09-26","3.066721"]},{"value":["2016-09-27","3.075821"]},{"value":["2016-09-28","3.057621"]},{"value":["2016-09-29","3.066721"]},{"value":["2016-09-30","3.066721"]},{"value":["2016-10-10","3.094021"]},{"value":["2016-10-11","3.094021"]},{"value":["2016-10-12","3.084921"]},{"value":["2016-10-13","3.075821"]},{"value":["2016-10-14","3.075821"]},{"value":["2016-10-17","3.057621"]},{"value":["2016-10-18","3.084921"]},{"value":["2016-10-19","3.075821"]},{"value":["2016-10-20","3.084921"]},{"value":["2016-10-21","3.094021"]},{"value":["2016-10-24","3.121322"]},{"value":["2016-10-25","3.130422"]},{"value":["2016-10-26","3.103121"]},{"value":["2016-10-27","3.084921"]},{"value":["2016-10-28","3.103121"]},{"value":["2016-10-31","3.084921"]},{"value":["2016-11-01","3.094021"]},{"value":["2016-11-02","3.075821"]},{"value":["2016-11-03","3.075821"]},{"value":["2016-11-04","3.066721"]},{"value":["2016-11-07","3.057621"]},{"value":["2016-11-08","3.075821"]},{"value":["2016-11-09","3.057621"]},{"value":["2016-11-10","3.075821"]},{"value":["2016-11-11","3.075821"]},{"value":["2016-11-14","3.094021"]},{"value":["2016-11-15","3.084921"]},{"value":["2016-11-16","3.084921"]},{"value":["2016-11-17","3.084921"]},{"value":["2016-11-18","3.075821"]},{"value":["2016-11-21","3.103121"]},{"value":["2016-11-22","3.121322"]},{"value":["2016-11-23","3.139522"]},{"value":["2016-11-24","3.139522"]},{"value":["2016-11-25","3.175922"]},{"value":["2016-11-28","3.221422"]},{"value":["2016-11-29","3.239622"]},{"value":["2016-11-30","3.194122"]},{"value":["2016-12-01","3.203222"]},{"value":["2016-12-02","3.194122"]},{"value":["2016-12-05","3.194122"]},{"value":["2016-12-06","3.185022"]},{"value":["2016-12-07","3.185022"]},{"value":["2016-12-08","3.203222"]},{"value":["2016-12-09","3.266923"]},{"value":["2016-12-12","3.312423"]},{"value":["2016-12-13","3.257823"]},{"value":["2016-12-14","3.266923"]},{"value":["2016-12-15","3.166822"]},{"value":["2016-12-16","3.148622"]},{"value":["2016-12-19","3.148622"]},{"value":["2016-12-20","3.112222"]},{"value":["2016-12-21","3.148622"]},{"value":["2016-12-22","3.139522"]},{"value":["2016-12-23","3.121322"]},{"value":["2016-12-26","3.130422"]},{"value":["2016-12-27","3.103121"]},{"value":["2016-12-28","3.112222"]},{"value":["2016-12-29","3.112222"]},{"value":["2016-12-30","3.130422"]},{"value":["2017-01-03","3.157722"]},{"value":["2017-01-04","3.157722"]},{"value":["2017-01-05","3.148622"]},{"value":["2017-01-06","3.130422"]},{"value":["2017-01-09","3.130422"]},{"value":["2017-01-10","3.121322"]},{"value":["2017-01-11","3.121322"]},{"value":["2017-01-12","3.121322"]},{"value":["2017-01-13","3.148622"]},{"value":["2017-01-16","3.185022"]},{"value":["2017-01-17","3.185022"]},{"value":["2017-01-18","3.194122"]},{"value":["2017-01-19","3.203222"]},{"value":["2017-01-20","3.203222"]},{"value":["2017-01-23","3.194122"]},{"value":["2017-01-24","3.221422"]},{"value":["2017-01-25","3.221422"]},{"value":["2017-01-26","3.257823"]},{"value":["2017-02-03","3.230522"]},{"value":["2017-02-06","3.230522"]},{"value":["2017-02-07","3.221422"]},{"value":["2017-02-08","3.230522"]},{"value":["2017-02-09","3.248722"]},{"value":["2017-02-10","3.276023"]},{"value":["2017-02-13","3.303323"]},{"value":["2017-02-14","3.294223"]},{"value":["2017-02-15","3.339723"]},{"value":["2017-02-16","3.357923"]},{"value":["2017-02-17","3.303323"]},{"value":["2017-02-20","3.357923"]},{"value":["2017-02-21","3.376123"]},{"value":["2017-02-22","3.385223"]},{"value":["2017-02-23","3.394324"]},{"value":["2017-02-24","3.385223"]},{"value":["2017-02-27","3.357923"]},{"value":["2017-02-28","3.348823"]},{"value":["2017-03-01","3.339723"]},{"value":["2017-03-02","3.339723"]},{"value":["2017-03-03","3.321523"]},{"value":["2017-03-06","3.303323"]},{"value":["2017-03-07","3.312423"]},{"value":["2017-03-08","3.312423"]},{"value":["2017-03-09","3.312423"]},{"value":["2017-03-10","3.303323"]},{"value":["2017-03-13","3.339723"]},{"value":["2017-03-14","3.330623"]},{"value":["2017-03-15","3.330623"]},{"value":["2017-03-16","3.348823"]},{"value":["2017-03-17","3.321523"]},{"value":["2017-03-20","3.339723"]},{"value":["2017-03-21","3.321523"]},{"value":["2017-03-22","3.276023"]},{"value":["2017-03-23","3.276023"]},{"value":["2017-03-24","3.303323"]},{"value":["2017-03-27","3.321523"]},{"value":["2017-03-28","3.312423"]},{"value":["2017-03-29","3.321523"]},{"value":["2017-03-30","3.330623"]},{"value":["2017-03-31","3.367023"]},{"value":["2017-04-05","3.348823"]},{"value":["2017-04-06","3.330623"]},{"value":["2017-04-07","3.312423"]},{"value":["2017-04-10","3.303323"]},{"value":["2017-04-11","3.294223"]},{"value":["2017-04-12","3.294223"]},{"value":["2017-04-13","3.303323"]},{"value":["2017-04-14","3.285123"]},{"value":["2017-04-17","3.276023"]},{"value":["2017-04-18","3.248722"]},{"value":["2017-04-19","3.212322"]},{"value":["2017-04-20","3.194122"]},{"value":["2017-04-21","3.221422"]},{"value":["2017-04-24","3.248722"]},{"value":["2017-04-25","3.248722"]},{"value":["2017-04-26","3.257823"]},{"value":["2017-04-27","3.266923"]},{"value":["2017-04-28","3.266923"]},{"value":["2017-05-02","3.239622"]},{"value":["2017-05-03","3.212322"]},{"value":["2017-05-04","3.212322"]},{"value":["2017-05-05","3.221422"]},{"value":["2017-05-08","3.248722"]},{"value":["2017-05-09","3.248722"]},{"value":["2017-05-10","3.257823"]},{"value":["2017-05-11","3.294223"]},{"value":["2017-05-12","3.339723"]},{"value":["2017-05-15","3.330623"]},{"value":["2017-05-16","3.303323"]},{"value":["2017-05-17","3.294223"]},{"value":["2017-05-18","3.285123"]},{"value":["2017-05-19","3.285123"]},{"value":["2017-05-22","3.303323"]},{"value":["2017-05-23","3.348823"]},{"value":["2017-05-24","3.330623"]},{"value":["2017-05-25","3.367023"]},{"value":["2017-05-26","3.367023"]},{"value":["2017-05-31","3.394324"]},{"value":["2017-06-01","3.403424"]},{"value":["2017-06-02","3.385223"]},{"value":["2017-06-05","3.303323"]},{"value":["2017-06-06","3.312423"]},{"value":["2017-06-07","3.339723"]},{"value":["2017-06-08","3.339723"]},{"value":["2017-06-09","3.330623"]},{"value":["2017-06-12","3.312423"]},{"value":["2017-06-13","3.312423"]},{"value":["2017-06-14","3.303323"]},{"value":["2017-06-15","3.285123"]},{"value":["2017-06-16","3.294223"]},{"value":["2017-06-19","3.285123"]},{"value":["2017-06-20","3.294223"]},{"value":["2017-06-21","3.294223"]},{"value":["2017-06-22","3.321523"]},{"value":["2017-06-23","3.339723"]},{"value":["2017-06-26","3.330623"]},{"value":["2017-06-27","3.330623"]},{"value":["2017-06-28","3.339723"]},{"value":["2017-06-29","3.348823"]},{"value":["2017-06-30","3.367023"]},{"value":["2017-07-03","3.357923"]},{"value":["2017-07-04","3.348823"]},{"value":["2017-07-05","3.348823"]},{"value":["2017-07-06","3.339723"]},{"value":["2017-07-07","3.357923"]},{"value":["2017-07-10","3.357923"]},{"value":["2017-07-11","3.385223"]},{"value":["2017-07-12","3.412524"]},{"value":["2017-07-13","3.467124"]},{"value":["2017-07-14","3.505224"]},{"value":["2017-07-17","3.610000"]},{"value":["2017-07-18","3.581425"]},{"value":["2017-07-19","3.581425"]},{"value":["2017-07-20","3.610000"]},{"value":["2017-07-21","3.600475"]},{"value":["2017-07-24","3.619525"]},{"value":["2017-07-25","3.667150"]},{"value":["2017-07-26","3.714776"]},{"value":["2017-07-27","3.686201"]},{"value":["2017-07-28","3.695726"]},{"value":["2017-07-31","3.733826"]},{"value":["2017-08-01","3.771926"]},{"value":["2017-08-02","3.857652"]},{"value":["2017-08-03","3.867177"]},{"value":["2017-08-04","3.838602"]},{"value":["2017-08-07","3.819551"]},{"value":["2017-08-08","3.876702"]},{"value":["2017-08-09","3.790976"]},{"value":["2017-08-10","3.790976"]},{"value":["2017-08-11","3.762401"]},{"value":["2017-08-14","3.724301"]},{"value":["2017-08-15","3.848127"]},{"value":["2017-08-16","3.838602"]},{"value":["2017-08-17","3.829077"]},{"value":["2017-08-18","3.810026"]},{"value":["2017-08-21","3.800501"]},{"value":["2017-08-22","3.810026"]},{"value":["2017-08-23","3.867177"]},{"value":["2017-08-24","3.867177"]},{"value":["2017-08-25","4.010053"]},{"value":["2017-08-28","4.029103"]},{"value":["2017-08-29","4.114828"]},{"value":["2017-08-30","4.067203"]},{"value":["2017-08-31","4.029103"]},{"value":["2017-09-01","4.000528"]},{"value":["2017-09-04","4.010053"]},{"value":["2017-09-05","4.067203"]},{"value":["2017-09-06","4.057678"]},{"value":["2017-09-07","4.019578"]},{"value":["2017-09-08","4.000528"]},{"value":["2017-09-11","4.000528"]},{"value":["2017-09-12","4.000528"]},{"value":["2017-09-13","3.981478"]},{"value":["2017-09-14","3.924327"]},{"value":["2017-09-15","3.867177"]},{"value":["2017-09-18","3.867177"]},{"value":["2017-09-19","3.886227"]},{"value":["2017-09-20","3.933852"]},{"value":["2017-09-21","3.962427"]},{"value":["2017-09-22","3.991003"]},{"value":["2017-09-25","4.029103"]},{"value":["2017-09-26","4.000528"]},{"value":["2017-09-27","3.952902"]},{"value":["2017-09-28","3.943377"]},{"value":["2017-09-29","3.924327"]},{"value":["2017-10-09","3.924327"]},{"value":["2017-10-10","3.943377"]},{"value":["2017-10-11","3.971953"]},{"value":["2017-10-12","3.962427"]},{"value":["2017-10-13","3.943377"]},{"value":["2017-10-16","3.952902"]},{"value":["2017-10-17","3.943377"]},{"value":["2017-10-18","3.971953"]},{"value":["2017-10-19","3.991003"]},{"value":["2017-10-20","3.971953"]},{"value":["2017-10-23","3.914802"]},{"value":["2017-10-24","3.924327"]},{"value":["2017-10-25","3.876702"]},{"value":["2017-10-26","3.810026"]},{"value":["2017-10-27","3.838602"]},{"value":["2017-10-30","3.857652"]},{"value":["2017-10-31","3.752876"]},{"value":["2017-11-01","3.724301"]},{"value":["2017-11-02","3.705251"]},{"value":["2017-11-03","3.695726"]},{"value":["2017-11-06","3.686201"]},{"value":["2017-11-07","3.724301"]},{"value":["2017-11-08","3.762401"]},{"value":["2017-11-09","3.733826"]},{"value":["2017-11-10","3.686201"]},{"value":["2017-11-13","3.705251"]},{"value":["2017-11-14","3.695726"]},{"value":["2017-11-15","3.686201"]},{"value":["2017-11-16","3.657625"]},{"value":["2017-11-17","3.733826"]},{"value":["2017-11-20","3.752876"]},{"value":["2017-11-21","3.733826"]},{"value":["2017-11-22","3.838602"]},{"value":["2017-11-23","3.810026"]},{"value":["2017-11-24","3.810026"]},{"value":["2017-11-27","3.762401"]},{"value":["2017-11-28","3.733826"]},{"value":["2017-11-29","3.743351"]},{"value":["2017-11-30","3.743351"]},{"value":["2017-12-01","3.733826"]},{"value":["2017-12-04","3.733826"]},{"value":["2017-12-05","3.790976"]},{"value":["2017-12-06","3.771926"]},{"value":["2017-12-07","3.752876"]},{"value":["2017-12-08","3.733826"]},{"value":["2017-12-11","3.743351"]},{"value":["2017-12-12","3.695726"]},{"value":["2017-12-13","3.714776"]},{"value":["2017-12-14","3.705251"]},{"value":["2017-12-15","3.676675"]},{"value":["2017-12-18","3.676675"]},{"value":["2017-12-19","3.705251"]},{"value":["2017-12-20","3.714776"]},{"value":["2017-12-21","3.714776"]},{"value":["2017-12-22","3.714776"]},{"value":["2017-12-25","3.733826"]},{"value":["2017-12-26","3.762401"]},{"value":["2017-12-27","3.733826"]},{"value":["2017-12-28","3.743351"]},{"value":["2017-12-29","3.781451"]},{"value":["2018-01-02","3.790976"]},{"value":["2018-01-03","3.810026"]},{"value":["2018-01-04","3.810026"]},{"value":["2018-01-05","3.790976"]},{"value":["2018-01-08","3.819551"]},{"value":["2018-01-09","3.771926"]},{"value":["2018-01-10","3.838602"]},{"value":["2018-01-11","3.895752"]},{"value":["2018-01-12","3.895752"]},{"value":["2018-01-15","3.981478"]},{"value":["2018-01-16","4.019578"]},{"value":["2018-01-17","4.038628"]},{"value":["2018-01-18","4.152929"]},{"value":["2018-01-19","4.191029"]},{"value":["2018-01-22","4.191029"]},{"value":["2018-01-23","4.457731"]},{"value":["2018-01-24","4.467256"]},{"value":["2018-01-25","4.410106"]},{"value":["2018-01-26","4.429156"]},{"value":["2018-01-29","4.381530"]},{"value":["2018-01-30","4.267230"]},{"value":["2018-01-31","4.324380"]},{"value":["2018-02-01","4.410106"]},{"value":["2018-02-02","4.362480"]},{"value":["2018-02-05","4.533931"]},{"value":["2018-02-06","4.533931"]},{"value":["2018-02-07","4.372005"]},{"value":["2018-02-08","4.210079"]},{"value":["2018-02-09","4.086253"]},{"value":["2018-02-12","4.010053"]},{"value":["2018-02-13","4.076728"]},{"value":["2018-02-14","4.076728"]},{"value":["2018-02-22","4.162454"]},{"value":["2018-02-23","4.181504"]},{"value":["2018-02-26","4.191029"]},{"value":["2018-02-27","4.114828"]},{"value":["2018-02-28","4.029103"]},{"value":["2018-03-01","4.019578"]},{"value":["2018-03-02","3.971953"]},{"value":["2018-03-05","3.962427"]},{"value":["2018-03-06","3.991003"]},{"value":["2018-03-07","4.019578"]},{"value":["2018-03-08","4.000528"]},{"value":["2018-03-09","3.981478"]},{"value":["2018-03-12","3.971953"]},{"value":["2018-03-13","3.991003"]},{"value":["2018-03-14","3.971953"]},{"value":["2018-03-15","3.971953"]},{"value":["2018-03-16","3.933852"]},{"value":["2018-03-19","3.933852"]},{"value":["2018-03-20","3.971953"]},{"value":["2018-03-21","3.971953"]},{"value":["2018-03-22","3.971953"]},{"value":["2018-03-23","3.924327"]},{"value":["2018-03-26","3.800501"]},{"value":["2018-03-27","3.781451"]},{"value":["2018-03-28","3.771926"]},{"value":["2018-03-29","3.800501"]},{"value":["2018-03-30","3.743351"]},{"value":["2018-04-02","3.695726"]},{"value":["2018-04-03","3.695726"]},{"value":["2018-04-04","3.667150"]},{"value":["2018-04-09","3.667150"]},{"value":["2018-04-10","3.743351"]},{"value":["2018-04-11","3.781451"]},{"value":["2018-04-12","3.724301"]},{"value":["2018-04-13","3.686201"]},{"value":["2018-04-16","3.590950"]},{"value":["2018-04-17","3.581425"]},{"value":["2018-04-18","3.600475"]},{"value":["2018-04-19","3.610000"]},{"value":["2018-04-20","3.600475"]},{"value":["2018-04-23","3.600475"]},{"value":["2018-04-24","3.657625"]},{"value":["2018-04-25","3.619525"]},{"value":["2018-04-26","3.610000"]},{"value":["2018-04-27","3.638575"]},{"value":["2018-05-02","3.629050"]},{"value":["2018-05-03","3.638575"]},{"value":["2018-05-04","3.619525"]},{"value":["2018-05-07","3.648100"]},{"value":["2018-05-08","3.686201"]},{"value":["2018-05-09","3.695726"]},{"value":["2018-05-10","3.695726"]},{"value":["2018-05-11","3.714776"]},{"value":["2018-05-14","3.714776"]},{"value":["2018-05-15","3.705251"]},{"value":["2018-05-16","3.667150"]},{"value":["2018-05-17","3.667150"]},{"value":["2018-05-18","3.714776"]},{"value":["2018-05-21","3.724301"]},{"value":["2018-05-22","3.695726"]},{"value":["2018-05-23","3.657625"]},{"value":["2018-05-24","3.629050"]},{"value":["2018-05-25","3.638575"]},{"value":["2018-05-28","3.638575"]},{"value":["2018-05-29","3.648100"]},{"value":["2018-05-30","3.581425"]},{"value":["2018-05-31","3.590950"]},{"value":["2018-06-01","3.581425"]},{"value":["2018-06-04","3.610000"]},{"value":["2018-06-05","3.610000"]},{"value":["2018-06-06","3.590950"]},{"value":["2018-06-07","3.600475"]},{"value":["2018-06-08","3.562375"]},{"value":["2018-06-11","3.562375"]},{"value":["2018-06-12","3.562375"]},{"value":["2018-06-13","3.552850"]},{"value":["2018-06-14","3.552850"]},{"value":["2018-06-15","3.571900"]},{"value":["2018-06-19","3.581425"]},{"value":["2018-06-20","3.552850"]},{"value":["2018-06-21","3.562375"]},{"value":["2018-06-22","3.562375"]},{"value":["2018-06-25","3.524274"]},{"value":["2018-06-26","3.476649"]},{"value":["2018-06-27","3.448074"]},{"value":["2018-06-28","3.390923"]},{"value":["2018-06-29","3.438549"]},{"value":["2018-07-02","3.305198"]},{"value":["2018-07-03","3.343298"]},{"value":["2018-07-04","3.362348"]},{"value":["2018-07-05","3.409974"]},{"value":["2018-07-06","3.429024"]},{"value":["2018-07-09","3.571900"]},{"value":["2018-07-10","3.552850"]},{"value":["2018-07-11","3.543325"]},{"value":["2018-07-12","3.610000"]},{"value":["2018-07-13","3.560000"]},{"value":["2018-07-16","3.500000"]},{"value":["2018-07-17","3.480000"]},{"value":["2018-07-18","3.460000"]},{"value":["2018-07-19","3.450000"]},{"value":["2018-07-20","3.550000"]},{"value":["2018-07-23","3.590000"]},{"value":["2018-07-24","3.600000"]},{"value":["2018-07-25","3.590000"]},{"value":["2018-07-26","3.570000"]},{"value":["2018-07-27","3.560000"]},{"value":["2018-07-30","3.600000"]},{"value":["2018-07-31","3.620000"]},{"value":["2018-08-01","3.560000"]},{"value":["2018-08-02","3.490000"]},{"value":["2018-08-03","3.490000"]},{"value":["2018-08-06","3.510000"]},{"value":["2018-08-07","3.560000"]},{"value":["2018-08-08","3.530000"]},{"value":["2018-08-09","3.560000"]},{"value":["2018-08-10","3.540000"]},{"value":["2018-08-13","3.500000"]},{"value":["2018-08-14","3.510000"]},{"value":["2018-08-15","3.450000"]},{"value":["2018-08-16","3.470000"]},{"value":["2018-08-17","3.450000"]},{"value":["2018-08-20","3.500000"]},{"value":["2018-08-21","3.520000"]},{"value":["2018-08-22","3.500000"]},{"value":["2018-08-23","3.500000"]},{"value":["2018-08-24","3.530000"]},{"value":["2018-08-27","3.550000"]},{"value":["2018-08-28","3.560000"]},{"value":["2018-08-29","3.560000"]},{"value":["2018-08-30","3.530000"]},{"value":["2018-08-31","3.550000"]},{"value":["2018-09-03","3.550000"]},{"value":["2018-09-04","3.580000"]},{"value":["2018-09-05","3.530000"]},{"value":["2018-09-06","3.500000"]},{"value":["2018-09-07","3.510000"]},{"value":["2018-09-10","3.520000"]},{"value":["2018-09-11","3.510000"]},{"value":["2018-09-12","3.490000"]},{"value":["2018-09-13","3.540000"]},{"value":["2018-09-14","3.540000"]},{"value":["2018-09-17","3.520000"]},{"value":["2018-09-18","3.550000"]},{"value":["2018-09-19","3.560000"]},{"value":["2018-09-20","3.570000"]},{"value":["2018-09-21","3.680000"]},{"value":["2018-09-25","3.650000"]},{"value":["2018-09-26","3.680000"]},{"value":["2018-09-27","3.680000"]},{"value":["2018-09-28","3.720000"]},{"value":["2018-10-08","3.620000"]},{"value":["2018-10-09","3.590000"]},{"value":["2018-10-10","3.600000"]},{"value":["2018-10-11","3.500000"]},{"value":["2018-10-12","3.550000"]},{"value":["2018-10-15","3.500000"]},{"value":["2018-10-16","3.510000"]},{"value":["2018-10-17","3.540000"]},{"value":["2018-10-18","3.520000"]},{"value":["2018-10-19","3.610000"]},{"value":["2018-10-22","3.700000"]},{"value":["2018-10-23","3.620000"]},{"value":["2018-10-24","3.680000"]},{"value":["2018-10-25","3.710000"]},{"value":["2018-10-26","3.710000"]},{"value":["2018-10-29","3.700000"]},{"value":["2018-10-30","3.740000"]},{"value":["2018-10-31","3.720000"]},{"value":["2018-11-01","3.670000"]},{"value":["2018-11-02","3.710000"]},{"value":["2018-11-05","3.680000"]},{"value":["2018-11-06","3.660000"]},{"value":["2018-11-07","3.660000"]},{"value":["2018-11-08","3.670000"]},{"value":["2018-11-09","3.590000"]},{"value":["2018-11-12","3.610000"]},{"value":["2018-11-13","3.620000"]},{"value":["2018-11-14","3.580000"]},{"value":["2018-11-15","3.600000"]},{"value":["2018-11-16","3.600000"]},{"value":["2018-11-19","3.640000"]},{"value":["2018-11-20","3.600000"]},{"value":["2018-11-21","3.610000"]},{"value":["2018-11-22","3.610000"]},{"value":["2018-11-23","3.590000"]},{"value":["2018-11-26","3.600000"]},{"value":["2018-11-27","3.580000"]},{"value":["2018-11-28","3.600000"]},{"value":["2018-11-29","3.590000"]},{"value":["2018-11-30","3.610000"]},{"value":["2018-12-03","3.650000"]},{"value":["2018-12-04","3.660000"]},{"value":["2018-12-05","3.640000"]},{"value":["2018-12-06","3.610000"]},{"value":["2018-12-07","3.600000"]},{"value":["2018-12-10","3.570000"]},{"value":["2018-12-11","3.570000"]},{"value":["2018-12-12","3.590000"]},{"value":["2018-12-13","3.600000"]},{"value":["2018-12-14","3.580000"]},{"value":["2018-12-17","3.610000"]},{"value":["2018-12-18","3.590000"]},{"value":["2018-12-19","3.580000"]},{"value":["2018-12-20","3.560000"]},{"value":["2018-12-21","3.570000"]},{"value":["2018-12-24","3.590000"]},{"value":["2018-12-25","3.610000"]},{"value":["2018-12-26","3.630000"]},{"value":["2018-12-27","3.630000"]},{"value":["2018-12-28","3.610000"]}],"yAxisIndex":0,"xAxisIndex":0,"name":"close_adj","type":"line","coordinateSystem":"cartesian2d","symbol":"none","markLine":{"symbol":"none","lineStyle":{"type":"dashed","color":"grey"},"data":[{"yAxis":3.61}]}},{"data":[{"value":["2013-12-31",null]},{"value":["2014-01-02",null]},{"value":["2014-01-03",null]},{"value":["2014-01-06",null]},{"value":["2014-01-07",null]},{"value":["2014-01-08",null]},{"value":["2014-01-09",null]},{"value":["2014-01-10",null]},{"value":["2014-01-13",null]},{"value":["2014-01-14",null]},{"value":["2014-01-15",null]},{"value":["2014-01-16",null]},{"value":["2014-01-17",null]},{"value":["2014-01-20",null]},{"value":["2014-01-21",null]},{"value":["2014-01-22",null]},{"value":["2014-01-23",null]},{"value":["2014-01-24",null]},{"value":["2014-01-27",null]},{"value":["2014-01-28",null]},{"value":["2014-01-29",null]},{"value":["2014-01-30",null]},{"value":["2014-02-07",null]},{"value":["2014-02-10",null]},{"value":["2014-02-11",null]},{"value":["2014-02-12",null]},{"value":["2014-02-13",null]},{"value":["2014-02-14",null]},{"value":["2014-02-17",null]},{"value":["2014-02-18",null]},{"value":["2014-02-19",null]},{"value":["2014-02-20",null]},{"value":["2014-02-21",null]},{"value":["2014-02-24",null]},{"value":["2014-02-25",null]},{"value":["2014-02-26",null]},{"value":["2014-02-27",null]},{"value":["2014-02-28",null]},{"value":["2014-03-03",null]},{"value":["2014-03-04",null]},{"value":["2014-03-05",null]},{"value":["2014-03-06",null]},{"value":["2014-03-07",null]},{"value":["2014-03-10",null]},{"value":["2014-03-11",null]},{"value":["2014-03-12",null]},{"value":["2014-03-13",null]},{"value":["2014-03-14",null]},{"value":["2014-03-17",null]},{"value":["2014-03-18",null]},{"value":["2014-03-19",null]},{"value":["2014-03-20",null]},{"value":["2014-03-21",null]},{"value":["2014-03-24",null]},{"value":["2014-03-25",null]},{"value":["2014-03-26",null]},{"value":["2014-03-27",null]},{"value":["2014-03-28",null]},{"value":["2014-03-31",null]},{"value":["2014-04-01",null]},{"value":["2014-04-02",null]},{"value":["2014-04-03",null]},{"value":["2014-04-04",null]},{"value":["2014-04-08",null]},{"value":["2014-04-09",null]},{"value":["2014-04-10",null]},{"value":["2014-04-11",null]},{"value":["2014-04-14",null]},{"value":["2014-04-15",null]},{"value":["2014-04-16",null]},{"value":["2014-04-17",null]},{"value":["2014-04-18",null]},{"value":["2014-04-21",null]},{"value":["2014-04-22",null]},{"value":["2014-04-23",null]},{"value":["2014-04-24",null]},{"value":["2014-04-25",null]},{"value":["2014-04-28",null]},{"value":["2014-04-29",null]},{"value":["2014-04-30",null]},{"value":["2014-05-05",null]},{"value":["2014-05-06",null]},{"value":["2014-05-07",null]},{"value":["2014-05-08",null]},{"value":["2014-05-09",null]},{"value":["2014-05-12",null]},{"value":["2014-05-13",null]},{"value":["2014-05-14",null]},{"value":["2014-05-15",null]},{"value":["2014-05-16",null]},{"value":["2014-05-19",null]},{"value":["2014-05-20",null]},{"value":["2014-05-21",null]},{"value":["2014-05-22",null]},{"value":["2014-05-23",null]},{"value":["2014-05-26",null]},{"value":["2014-05-27",null]},{"value":["2014-05-28",null]},{"value":["2014-05-29",null]},{"value":["2014-05-30",null]},{"value":["2014-06-03",null]},{"value":["2014-06-04",null]},{"value":["2014-06-05",null]},{"value":["2014-06-06",null]},{"value":["2014-06-09",null]},{"value":["2014-06-10",null]},{"value":["2014-06-11",null]},{"value":["2014-06-12",null]},{"value":["2014-06-13",null]},{"value":["2014-06-16",null]},{"value":["2014-06-17",null]},{"value":["2014-06-18",null]},{"value":["2014-06-19",null]},{"value":["2014-06-20",null]},{"value":["2014-06-23",null]},{"value":["2014-06-24",null]},{"value":["2014-06-25",null]},{"value":["2014-06-26",null]},{"value":["2014-06-27",null]},{"value":["2014-06-30",null]},{"value":["2014-07-01",null]},{"value":["2014-07-02",null]},{"value":["2014-07-03",null]},{"value":["2014-07-04",null]},{"value":["2014-07-07",null]},{"value":["2014-07-08",null]},{"value":["2014-07-09",null]},{"value":["2014-07-10",null]},{"value":["2014-07-11",null]},{"value":["2014-07-14",null]},{"value":["2014-07-15",null]},{"value":["2014-07-16",null]},{"value":["2014-07-17",null]},{"value":["2014-07-18",null]},{"value":["2014-07-21",null]},{"value":["2014-07-22",null]},{"value":["2014-07-23",null]},{"value":["2014-07-24",null]},{"value":["2014-07-25",null]},{"value":["2014-07-28",null]},{"value":["2014-07-29",null]},{"value":["2014-07-30",null]},{"value":["2014-07-31",null]},{"value":["2014-08-01",null]},{"value":["2014-08-04",null]},{"value":["2014-08-05",null]},{"value":["2014-08-06",null]},{"value":["2014-08-07",null]},{"value":["2014-08-08",null]},{"value":["2014-08-11",null]},{"value":["2014-08-12",null]},{"value":["2014-08-13",null]},{"value":["2014-08-14",null]},{"value":["2014-08-15",null]},{"value":["2014-08-18",null]},{"value":["2014-08-19",null]},{"value":["2014-08-20",null]},{"value":["2014-08-21",null]},{"value":["2014-08-22",null]},{"value":["2014-08-25",null]},{"value":["2014-08-26",null]},{"value":["2014-08-27",null]},{"value":["2014-08-28",null]},{"value":["2014-08-29",null]},{"value":["2014-09-01",null]},{"value":["2014-09-02",null]},{"value":["2014-09-03",null]},{"value":["2014-09-04",null]},{"value":["2014-09-05",null]},{"value":["2014-09-09",null]},{"value":["2014-09-10",null]},{"value":["2014-09-11",null]},{"value":["2014-09-12",null]},{"value":["2014-09-15",null]},{"value":["2014-09-16",null]},{"value":["2014-09-17",null]},{"value":["2014-09-18",null]},{"value":["2014-09-19",null]},{"value":["2014-09-22",null]},{"value":["2014-09-23",null]},{"value":["2014-09-24",null]},{"value":["2014-09-25",null]},{"value":["2014-09-26",null]},{"value":["2014-09-29",null]},{"value":["2014-09-30",null]},{"value":["2014-10-08",null]},{"value":["2014-10-09",null]},{"value":["2014-10-10",null]},{"value":["2014-10-13",null]},{"value":["2014-10-14",null]},{"value":["2014-10-15",null]},{"value":["2014-10-16",null]},{"value":["2014-10-17",null]},{"value":["2014-10-20",null]},{"value":["2014-10-21",null]},{"value":["2014-10-22",null]},{"value":["2014-10-23",null]},{"value":["2014-10-24",null]},{"value":["2014-10-27",null]},{"value":["2014-10-28",null]},{"value":["2014-10-29",null]},{"value":["2014-10-30",null]},{"value":["2014-10-31",null]},{"value":["2014-11-03",null]},{"value":["2014-11-04",null]},{"value":["2014-11-05",null]},{"value":["2014-11-06",null]},{"value":["2014-11-07",null]},{"value":["2014-11-10",null]},{"value":["2014-11-11",null]},{"value":["2014-11-12",null]},{"value":["2014-11-13",null]},{"value":["2014-11-14",null]},{"value":["2014-11-17",null]},{"value":["2014-11-18",null]},{"value":["2014-11-19",null]},{"value":["2014-11-20",null]},{"value":["2014-11-21",null]},{"value":["2014-11-24",null]},{"value":["2014-11-25",null]},{"value":["2014-11-26",null]},{"value":["2014-11-27",null]},{"value":["2014-11-28",null]},{"value":["2014-12-01",null]},{"value":["2014-12-02",null]},{"value":["2014-12-03",null]},{"value":["2014-12-04",null]},{"value":["2014-12-05",null]},{"value":["2014-12-08",null]},{"value":["2014-12-09",null]},{"value":["2014-12-10",null]},{"value":["2014-12-11",null]},{"value":["2014-12-12",null]},{"value":["2014-12-15",null]},{"value":["2014-12-16",null]},{"value":["2014-12-17",null]},{"value":["2014-12-18",null]},{"value":["2014-12-19",null]},{"value":["2014-12-22",null]},{"value":["2014-12-23",null]},{"value":["2014-12-24",null]},{"value":["2014-12-25",null]},{"value":["2014-12-26",null]},{"value":["2014-12-29",null]},{"value":["2014-12-30",null]},{"value":["2014-12-31",null]},{"value":["2015-01-05",null]},{"value":["2015-01-06",null]},{"value":["2015-01-07",null]},{"value":["2015-01-08",null]},{"value":["2015-01-09",null]},{"value":["2015-01-12",null]},{"value":["2015-01-13",null]},{"value":["2015-01-14",null]},{"value":["2015-01-15",null]},{"value":["2015-01-16",null]},{"value":["2015-01-19",null]},{"value":["2015-01-20",null]},{"value":["2015-01-21",null]},{"value":["2015-01-22",null]},{"value":["2015-01-23",null]},{"value":["2015-01-26",null]},{"value":["2015-01-27",null]},{"value":["2015-01-28",null]},{"value":["2015-01-29",null]},{"value":["2015-01-30",null]},{"value":["2015-02-02",null]},{"value":["2015-02-03",null]},{"value":["2015-02-04",null]},{"value":["2015-02-05",null]},{"value":["2015-02-06",null]},{"value":["2015-02-09",null]},{"value":["2015-02-10",null]},{"value":["2015-02-11",null]},{"value":["2015-02-12",null]},{"value":["2015-02-13",null]},{"value":["2015-02-16",null]},{"value":["2015-02-17",null]},{"value":["2015-02-25",null]},{"value":["2015-02-26",null]},{"value":["2015-02-27",null]},{"value":["2015-03-02",null]},{"value":["2015-03-03",null]},{"value":["2015-03-04",null]},{"value":["2015-03-05",null]},{"value":["2015-03-06",null]},{"value":["2015-03-09",null]},{"value":["2015-03-10",null]},{"value":["2015-03-11",null]},{"value":["2015-03-12",null]},{"value":["2015-03-13",null]},{"value":["2015-03-16",null]},{"value":["2015-03-17",null]},{"value":["2015-03-18",null]},{"value":["2015-03-19",null]},{"value":["2015-03-20",null]},{"value":["2015-03-23",null]},{"value":["2015-03-24",null]},{"value":["2015-03-25",null]},{"value":["2015-03-26",null]},{"value":["2015-03-27",null]},{"value":["2015-03-30",null]},{"value":["2015-03-31",null]},{"value":["2015-04-01",null]},{"value":["2015-04-02",null]},{"value":["2015-04-03",null]},{"value":["2015-04-07",null]},{"value":["2015-04-08",null]},{"value":["2015-04-09",null]},{"value":["2015-04-10",null]},{"value":["2015-04-13",null]},{"value":["2015-04-14",null]},{"value":["2015-04-15",null]},{"value":["2015-04-16",null]},{"value":["2015-04-17",null]},{"value":["2015-04-20",null]},{"value":["2015-04-21",null]},{"value":["2015-04-22",null]},{"value":["2015-04-23",null]},{"value":["2015-04-24",null]},{"value":["2015-04-27",null]},{"value":["2015-04-28",null]},{"value":["2015-04-29",null]},{"value":["2015-04-30",null]},{"value":["2015-05-04",null]},{"value":["2015-05-05",null]},{"value":["2015-05-06",null]},{"value":["2015-05-07",null]},{"value":["2015-05-08",null]},{"value":["2015-05-11",null]},{"value":["2015-05-12",null]},{"value":["2015-05-13",null]},{"value":["2015-05-14",null]},{"value":["2015-05-15",null]},{"value":["2015-05-18",null]},{"value":["2015-05-19",null]},{"value":["2015-05-20",null]},{"value":["2015-05-21",null]},{"value":["2015-05-22",null]},{"value":["2015-05-25",null]},{"value":["2015-05-26",null]},{"value":["2015-05-27",null]},{"value":["2015-05-28",null]},{"value":["2015-05-29",null]},{"value":["2015-06-01",null]},{"value":["2015-06-02",null]},{"value":["2015-06-03",null]},{"value":["2015-06-04",null]},{"value":["2015-06-05",null]},{"value":["2015-06-08",null]},{"value":["2015-06-09",null]},{"value":["2015-06-10",null]},{"value":["2015-06-11",null]},{"value":["2015-06-12",null]},{"value":["2015-06-15",null]},{"value":["2015-06-16",null]},{"value":["2015-06-17",null]},{"value":["2015-06-18",null]},{"value":["2015-06-19",null]},{"value":["2015-06-23",null]},{"value":["2015-06-24",null]},{"value":["2015-06-25",null]},{"value":["2015-06-26",null]},{"value":["2015-06-29",null]},{"value":["2015-06-30",null]},{"value":["2015-07-01",null]},{"value":["2015-07-02",null]},{"value":["2015-07-03",null]},{"value":["2015-07-06",null]},{"value":["2015-07-07",null]},{"value":["2015-07-08",null]},{"value":["2015-07-09",null]},{"value":["2015-07-10",null]},{"value":["2015-07-13",null]},{"value":["2015-07-14",null]},{"value":["2015-07-15",null]},{"value":["2015-07-16",null]},{"value":["2015-07-17",null]},{"value":["2015-07-20",null]},{"value":["2015-07-21",null]},{"value":["2015-07-22",null]},{"value":["2015-07-23",null]},{"value":["2015-07-24",null]},{"value":["2015-07-27",null]},{"value":["2015-07-28",null]},{"value":["2015-07-29",null]},{"value":["2015-07-30",null]},{"value":["2015-07-31",null]},{"value":["2015-08-03",null]},{"value":["2015-08-04",null]},{"value":["2015-08-05",null]},{"value":["2015-08-06",null]},{"value":["2015-08-07",null]},{"value":["2015-08-10",null]},{"value":["2015-08-11",null]},{"value":["2015-08-12",null]},{"value":["2015-08-13",null]},{"value":["2015-08-14",null]},{"value":["2015-08-17",null]},{"value":["2015-08-18",null]},{"value":["2015-08-19",null]},{"value":["2015-08-20",null]},{"value":["2015-08-21",null]},{"value":["2015-08-24",null]},{"value":["2015-08-25",null]},{"value":["2015-08-26",null]},{"value":["2015-08-27",null]},{"value":["2015-08-28",null]},{"value":["2015-08-31",null]},{"value":["2015-09-01",null]},{"value":["2015-09-02",null]},{"value":["2015-09-07",null]},{"value":["2015-09-08",null]},{"value":["2015-09-09",null]},{"value":["2015-09-10",null]},{"value":["2015-09-11",null]},{"value":["2015-09-14",null]},{"value":["2015-09-15",null]},{"value":["2015-09-16",null]},{"value":["2015-09-17",null]},{"value":["2015-09-18",null]},{"value":["2015-09-21",null]},{"value":["2015-09-22",null]},{"value":["2015-09-23",null]},{"value":["2015-09-24",null]},{"value":["2015-09-25",null]},{"value":["2015-09-28",null]},{"value":["2015-09-29",null]},{"value":["2015-09-30",null]},{"value":["2015-10-08",null]},{"value":["2015-10-09",null]},{"value":["2015-10-12",null]},{"value":["2015-10-13",null]},{"value":["2015-10-14",null]},{"value":["2015-10-15",null]},{"value":["2015-10-16",null]},{"value":["2015-10-19",null]},{"value":["2015-10-20",null]},{"value":["2015-10-21",null]},{"value":["2015-10-22",null]},{"value":["2015-10-23",null]},{"value":["2015-10-26",null]},{"value":["2015-10-27",null]},{"value":["2015-10-28",null]},{"value":["2015-10-29",null]},{"value":["2015-10-30",null]},{"value":["2015-11-02",null]},{"value":["2015-11-03",null]},{"value":["2015-11-04",null]},{"value":["2015-11-05",null]},{"value":["2015-11-06",null]},{"value":["2015-11-09",null]},{"value":["2015-11-10",null]},{"value":["2015-11-11",null]},{"value":["2015-11-12",null]},{"value":["2015-11-13",null]},{"value":["2015-11-16",null]},{"value":["2015-11-17",null]},{"value":["2015-11-18",null]},{"value":["2015-11-19",null]},{"value":["2015-11-20",null]},{"value":["2015-11-23",null]},{"value":["2015-11-24",null]},{"value":["2015-11-25",null]},{"value":["2015-11-26",null]},{"value":["2015-11-27",null]},{"value":["2015-11-30",null]},{"value":["2015-12-01",null]},{"value":["2015-12-02",null]},{"value":["2015-12-03",null]},{"value":["2015-12-04",null]},{"value":["2015-12-07",null]},{"value":["2015-12-08",null]},{"value":["2015-12-09",null]},{"value":["2015-12-10",null]},{"value":["2015-12-11",null]},{"value":["2015-12-14",null]},{"value":["2015-12-15",null]},{"value":["2015-12-16",null]},{"value":["2015-12-17",null]},{"value":["2015-12-18",null]},{"value":["2015-12-21",null]},{"value":["2015-12-22",null]},{"value":["2015-12-23",null]},{"value":["2015-12-24",null]},{"value":["2015-12-25",null]},{"value":["2015-12-28",null]},{"value":["2015-12-29",null]},{"value":["2015-12-30",null]},{"value":["2015-12-31",null]},{"value":["2016-01-04",null]},{"value":["2016-01-05",null]},{"value":["2016-01-06",null]},{"value":["2016-01-07",null]},{"value":["2016-01-08",null]},{"value":["2016-01-11",null]},{"value":["2016-01-12",null]},{"value":["2016-01-13",null]},{"value":["2016-01-14",null]},{"value":["2016-01-15",null]},{"value":["2016-01-18",null]},{"value":["2016-01-19",null]},{"value":["2016-01-20",null]},{"value":["2016-01-21",null]},{"value":["2016-01-22",null]},{"value":["2016-01-25",null]},{"value":["2016-01-26",null]},{"value":["2016-01-27",null]},{"value":["2016-01-28",null]},{"value":["2016-01-29",null]},{"value":["2016-02-01",null]},{"value":["2016-02-02",null]},{"value":["2016-02-03",null]},{"value":["2016-02-04",null]},{"value":["2016-02-05",null]},{"value":["2016-02-15",null]},{"value":["2016-02-16",null]},{"value":["2016-02-17",null]},{"value":["2016-02-18",null]},{"value":["2016-02-19",null]},{"value":["2016-02-22",null]},{"value":["2016-02-23",null]},{"value":["2016-02-24",null]},{"value":["2016-02-25",null]},{"value":["2016-02-26",null]},{"value":["2016-02-29",null]},{"value":["2016-03-01",null]},{"value":["2016-03-02",null]},{"value":["2016-03-03",null]},{"value":["2016-03-04",null]},{"value":["2016-03-07",null]},{"value":["2016-03-08",null]},{"value":["2016-03-09",null]},{"value":["2016-03-10",null]},{"value":["2016-03-11",null]},{"value":["2016-03-14",null]},{"value":["2016-03-15",null]},{"value":["2016-03-16",null]},{"value":["2016-03-17",null]},{"value":["2016-03-18",null]},{"value":["2016-03-21",null]},{"value":["2016-03-22",null]},{"value":["2016-03-23",null]},{"value":["2016-03-24",null]},{"value":["2016-03-25",null]},{"value":["2016-03-28",null]},{"value":["2016-03-29",null]},{"value":["2016-03-30",null]},{"value":["2016-03-31",null]},{"value":["2016-04-01",null]},{"value":["2016-04-05",null]},{"value":["2016-04-06",null]},{"value":["2016-04-07",null]},{"value":["2016-04-08",null]},{"value":["2016-04-11",null]},{"value":["2016-04-12",null]},{"value":["2016-04-13",null]},{"value":["2016-04-14",null]},{"value":["2016-04-15",null]},{"value":["2016-04-18",null]},{"value":["2016-04-19",null]},{"value":["2016-04-20",null]},{"value":["2016-04-21",null]},{"value":["2016-04-22",null]},{"value":["2016-04-25",null]},{"value":["2016-04-26",null]},{"value":["2016-04-27",null]},{"value":["2016-04-28",null]},{"value":["2016-04-29",null]},{"value":["2016-05-03",null]},{"value":["2016-05-04",null]},{"value":["2016-05-05",null]},{"value":["2016-05-06",null]},{"value":["2016-05-09",null]},{"value":["2016-05-10",null]},{"value":["2016-05-11",null]},{"value":["2016-05-12",null]},{"value":["2016-05-13",null]},{"value":["2016-05-16",null]},{"value":["2016-05-17",null]},{"value":["2016-05-18",null]},{"value":["2016-05-19",null]},{"value":["2016-05-20",null]},{"value":["2016-05-23",null]},{"value":["2016-05-24",null]},{"value":["2016-05-25",null]},{"value":["2016-05-26",null]},{"value":["2016-05-27",null]},{"value":["2016-05-30",null]},{"value":["2016-05-31",null]},{"value":["2016-06-01",null]},{"value":["2016-06-02",null]},{"value":["2016-06-03",null]},{"value":["2016-06-06",null]},{"value":["2016-06-07",null]},{"value":["2016-06-08",null]},{"value":["2016-06-13",null]},{"value":["2016-06-14",null]},{"value":["2016-06-15",null]},{"value":["2016-06-16",null]},{"value":["2016-06-17",null]},{"value":["2016-06-20",null]},{"value":["2016-06-21",null]},{"value":["2016-06-22",null]},{"value":["2016-06-23",null]},{"value":["2016-06-24",null]},{"value":["2016-06-27",null]},{"value":["2016-06-28",null]},{"value":["2016-06-29",null]},{"value":["2016-06-30",null]},{"value":["2016-07-01",null]},{"value":["2016-07-04",null]},{"value":["2016-07-05",null]},{"value":["2016-07-06",null]},{"value":["2016-07-07",null]},{"value":["2016-07-08",null]},{"value":["2016-07-11",null]},{"value":["2016-07-12",null]},{"value":["2016-07-13",null]},{"value":["2016-07-14",null]},{"value":["2016-07-15",null]},{"value":["2016-07-18",null]},{"value":["2016-07-19",null]},{"value":["2016-07-20",null]},{"value":["2016-07-21",null]},{"value":["2016-07-22",null]},{"value":["2016-07-25",null]},{"value":["2016-07-26",null]},{"value":["2016-07-27",null]},{"value":["2016-07-28",null]},{"value":["2016-07-29",null]},{"value":["2016-08-01",null]},{"value":["2016-08-02",null]},{"value":["2016-08-03",null]},{"value":["2016-08-04",null]},{"value":["2016-08-05",null]},{"value":["2016-08-08",null]},{"value":["2016-08-09",null]},{"value":["2016-08-10",null]},{"value":["2016-08-11",null]},{"value":["2016-08-12",null]},{"value":["2016-08-15",null]},{"value":["2016-08-16",null]},{"value":["2016-08-17",null]},{"value":["2016-08-18",null]},{"value":["2016-08-19",null]},{"value":["2016-08-22",null]},{"value":["2016-08-23",null]},{"value":["2016-08-24",null]},{"value":["2016-08-25",null]},{"value":["2016-08-26",null]},{"value":["2016-08-29",null]},{"value":["2016-08-30",null]},{"value":["2016-08-31",null]},{"value":["2016-09-01",null]},{"value":["2016-09-02",null]},{"value":["2016-09-05","3.157722"]},{"value":["2016-09-06",null]},{"value":["2016-09-07",null]},{"value":["2016-09-08",null]},{"value":["2016-09-09",null]},{"value":["2016-09-12",null]},{"value":["2016-09-13",null]},{"value":["2016-09-14",null]},{"value":["2016-09-19",null]},{"value":["2016-09-20",null]},{"value":["2016-09-21",null]},{"value":["2016-09-22",null]},{"value":["2016-09-23",null]},{"value":["2016-09-26",null]},{"value":["2016-09-27",null]},{"value":["2016-09-28",null]},{"value":["2016-09-29",null]},{"value":["2016-09-30",null]},{"value":["2016-10-10",null]},{"value":["2016-10-11",null]},{"value":["2016-10-12",null]},{"value":["2016-10-13",null]},{"value":["2016-10-14",null]},{"value":["2016-10-17",null]},{"value":["2016-10-18",null]},{"value":["2016-10-19",null]},{"value":["2016-10-20",null]},{"value":["2016-10-21",null]},{"value":["2016-10-24",null]},{"value":["2016-10-25",null]},{"value":["2016-10-26",null]},{"value":["2016-10-27",null]},{"value":["2016-10-28",null]},{"value":["2016-10-31",null]},{"value":["2016-11-01",null]},{"value":["2016-11-02",null]},{"value":["2016-11-03",null]},{"value":["2016-11-04",null]},{"value":["2016-11-07",null]},{"value":["2016-11-08",null]},{"value":["2016-11-09",null]},{"value":["2016-11-10",null]},{"value":["2016-11-11",null]},{"value":["2016-11-14",null]},{"value":["2016-11-15",null]},{"value":["2016-11-16",null]},{"value":["2016-11-17",null]},{"value":["2016-11-18",null]},{"value":["2016-11-21",null]},{"value":["2016-11-22",null]},{"value":["2016-11-23",null]},{"value":["2016-11-24",null]},{"value":["2016-11-25",null]},{"value":["2016-11-28",null]},{"value":["2016-11-29",null]},{"value":["2016-11-30",null]},{"value":["2016-12-01",null]},{"value":["2016-12-02",null]},{"value":["2016-12-05",null]},{"value":["2016-12-06",null]},{"value":["2016-12-07",null]},{"value":["2016-12-08",null]},{"value":["2016-12-09",null]},{"value":["2016-12-12",null]},{"value":["2016-12-13",null]},{"value":["2016-12-14",null]},{"value":["2016-12-15",null]},{"value":["2016-12-16",null]},{"value":["2016-12-19",null]},{"value":["2016-12-20",null]},{"value":["2016-12-21",null]},{"value":["2016-12-22",null]},{"value":["2016-12-23",null]},{"value":["2016-12-26",null]},{"value":["2016-12-27",null]},{"value":["2016-12-28",null]},{"value":["2016-12-29",null]},{"value":["2016-12-30",null]},{"value":["2017-01-03",null]},{"value":["2017-01-04",null]},{"value":["2017-01-05",null]},{"value":["2017-01-06",null]},{"value":["2017-01-09",null]},{"value":["2017-01-10",null]},{"value":["2017-01-11",null]},{"value":["2017-01-12",null]},{"value":["2017-01-13",null]},{"value":["2017-01-16",null]},{"value":["2017-01-17",null]},{"value":["2017-01-18",null]},{"value":["2017-01-19",null]},{"value":["2017-01-20",null]},{"value":["2017-01-23",null]},{"value":["2017-01-24",null]},{"value":["2017-01-25",null]},{"value":["2017-01-26",null]},{"value":["2017-02-03",null]},{"value":["2017-02-06",null]},{"value":["2017-02-07",null]},{"value":["2017-02-08",null]},{"value":["2017-02-09",null]},{"value":["2017-02-10",null]},{"value":["2017-02-13",null]},{"value":["2017-02-14",null]},{"value":["2017-02-15",null]},{"value":["2017-02-16",null]},{"value":["2017-02-17",null]},{"value":["2017-02-20",null]},{"value":["2017-02-21",null]},{"value":["2017-02-22",null]},{"value":["2017-02-23",null]},{"value":["2017-02-24",null]},{"value":["2017-02-27",null]},{"value":["2017-02-28",null]},{"value":["2017-03-01",null]},{"value":["2017-03-02",null]},{"value":["2017-03-03",null]},{"value":["2017-03-06",null]},{"value":["2017-03-07",null]},{"value":["2017-03-08",null]},{"value":["2017-03-09",null]},{"value":["2017-03-10",null]},{"value":["2017-03-13",null]},{"value":["2017-03-14",null]},{"value":["2017-03-15",null]},{"value":["2017-03-16",null]},{"value":["2017-03-17",null]},{"value":["2017-03-20",null]},{"value":["2017-03-21",null]},{"value":["2017-03-22",null]},{"value":["2017-03-23",null]},{"value":["2017-03-24",null]},{"value":["2017-03-27",null]},{"value":["2017-03-28",null]},{"value":["2017-03-29",null]},{"value":["2017-03-30",null]},{"value":["2017-03-31",null]},{"value":["2017-04-05",null]},{"value":["2017-04-06",null]},{"value":["2017-04-07",null]},{"value":["2017-04-10",null]},{"value":["2017-04-11",null]},{"value":["2017-04-12",null]},{"value":["2017-04-13",null]},{"value":["2017-04-14",null]},{"value":["2017-04-17",null]},{"value":["2017-04-18",null]},{"value":["2017-04-19",null]},{"value":["2017-04-20",null]},{"value":["2017-04-21",null]},{"value":["2017-04-24",null]},{"value":["2017-04-25",null]},{"value":["2017-04-26",null]},{"value":["2017-04-27",null]},{"value":["2017-04-28",null]},{"value":["2017-05-02",null]},{"value":["2017-05-03",null]},{"value":["2017-05-04",null]},{"value":["2017-05-05",null]},{"value":["2017-05-08",null]},{"value":["2017-05-09",null]},{"value":["2017-05-10",null]},{"value":["2017-05-11",null]},{"value":["2017-05-12",null]},{"value":["2017-05-15",null]},{"value":["2017-05-16",null]},{"value":["2017-05-17",null]},{"value":["2017-05-18",null]},{"value":["2017-05-19",null]},{"value":["2017-05-22",null]},{"value":["2017-05-23",null]},{"value":["2017-05-24",null]},{"value":["2017-05-25",null]},{"value":["2017-05-26",null]},{"value":["2017-05-31",null]},{"value":["2017-06-01",null]},{"value":["2017-06-02",null]},{"value":["2017-06-05",null]},{"value":["2017-06-06",null]},{"value":["2017-06-07",null]},{"value":["2017-06-08",null]},{"value":["2017-06-09",null]},{"value":["2017-06-12",null]},{"value":["2017-06-13",null]},{"value":["2017-06-14",null]},{"value":["2017-06-15",null]},{"value":["2017-06-16",null]},{"value":["2017-06-19",null]},{"value":["2017-06-20",null]},{"value":["2017-06-21",null]},{"value":["2017-06-22",null]},{"value":["2017-06-23",null]},{"value":["2017-06-26",null]},{"value":["2017-06-27",null]},{"value":["2017-06-28",null]},{"value":["2017-06-29",null]},{"value":["2017-06-30",null]},{"value":["2017-07-03",null]},{"value":["2017-07-04",null]},{"value":["2017-07-05",null]},{"value":["2017-07-06",null]},{"value":["2017-07-07",null]},{"value":["2017-07-10",null]},{"value":["2017-07-11",null]},{"value":["2017-07-12",null]},{"value":["2017-07-13",null]},{"value":["2017-07-14",null]},{"value":["2017-07-17",null]},{"value":["2017-07-18",null]},{"value":["2017-07-19",null]},{"value":["2017-07-20",null]},{"value":["2017-07-21",null]},{"value":["2017-07-24",null]},{"value":["2017-07-25",null]},{"value":["2017-07-26",null]},{"value":["2017-07-27",null]},{"value":["2017-07-28",null]},{"value":["2017-07-31",null]},{"value":["2017-08-01",null]},{"value":["2017-08-02",null]},{"value":["2017-08-03",null]},{"value":["2017-08-04",null]},{"value":["2017-08-07",null]},{"value":["2017-08-08",null]},{"value":["2017-08-09",null]},{"value":["2017-08-10",null]},{"value":["2017-08-11",null]},{"value":["2017-08-14",null]},{"value":["2017-08-15",null]},{"value":["2017-08-16",null]},{"value":["2017-08-17",null]},{"value":["2017-08-18",null]},{"value":["2017-08-21",null]},{"value":["2017-08-22",null]},{"value":["2017-08-23",null]},{"value":["2017-08-24",null]},{"value":["2017-08-25",null]},{"value":["2017-08-28",null]},{"value":["2017-08-29",null]},{"value":["2017-08-30",null]},{"value":["2017-08-31",null]},{"value":["2017-09-01",null]},{"value":["2017-09-04",null]},{"value":["2017-09-05",null]},{"value":["2017-09-06",null]},{"value":["2017-09-07",null]},{"value":["2017-09-08",null]},{"value":["2017-09-11",null]},{"value":["2017-09-12",null]},{"value":["2017-09-13",null]},{"value":["2017-09-14",null]},{"value":["2017-09-15",null]},{"value":["2017-09-18",null]},{"value":["2017-09-19",null]},{"value":["2017-09-20",null]},{"value":["2017-09-21",null]},{"value":["2017-09-22",null]},{"value":["2017-09-25",null]},{"value":["2017-09-26",null]},{"value":["2017-09-27",null]},{"value":["2017-09-28",null]},{"value":["2017-09-29",null]},{"value":["2017-10-09",null]},{"value":["2017-10-10",null]},{"value":["2017-10-11",null]},{"value":["2017-10-12",null]},{"value":["2017-10-13",null]},{"value":["2017-10-16",null]},{"value":["2017-10-17",null]},{"value":["2017-10-18",null]},{"value":["2017-10-19",null]},{"value":["2017-10-20",null]},{"value":["2017-10-23",null]},{"value":["2017-10-24",null]},{"value":["2017-10-25",null]},{"value":["2017-10-26",null]},{"value":["2017-10-27",null]},{"value":["2017-10-30",null]},{"value":["2017-10-31",null]},{"value":["2017-11-01",null]},{"value":["2017-11-02",null]},{"value":["2017-11-03",null]},{"value":["2017-11-06",null]},{"value":["2017-11-07",null]},{"value":["2017-11-08",null]},{"value":["2017-11-09",null]},{"value":["2017-11-10",null]},{"value":["2017-11-13",null]},{"value":["2017-11-14",null]},{"value":["2017-11-15",null]},{"value":["2017-11-16",null]},{"value":["2017-11-17",null]},{"value":["2017-11-20",null]},{"value":["2017-11-21",null]},{"value":["2017-11-22",null]},{"value":["2017-11-23",null]},{"value":["2017-11-24",null]},{"value":["2017-11-27",null]},{"value":["2017-11-28",null]},{"value":["2017-11-29",null]},{"value":["2017-11-30",null]},{"value":["2017-12-01",null]},{"value":["2017-12-04",null]},{"value":["2017-12-05",null]},{"value":["2017-12-06",null]},{"value":["2017-12-07",null]},{"value":["2017-12-08",null]},{"value":["2017-12-11",null]},{"value":["2017-12-12",null]},{"value":["2017-12-13",null]},{"value":["2017-12-14",null]},{"value":["2017-12-15",null]},{"value":["2017-12-18",null]},{"value":["2017-12-19",null]},{"value":["2017-12-20",null]},{"value":["2017-12-21",null]},{"value":["2017-12-22",null]},{"value":["2017-12-25",null]},{"value":["2017-12-26",null]},{"value":["2017-12-27",null]},{"value":["2017-12-28",null]},{"value":["2017-12-29",null]},{"value":["2018-01-02",null]},{"value":["2018-01-03",null]},{"value":["2018-01-04",null]},{"value":["2018-01-05",null]},{"value":["2018-01-08",null]},{"value":["2018-01-09",null]},{"value":["2018-01-10",null]},{"value":["2018-01-11",null]},{"value":["2018-01-12",null]},{"value":["2018-01-15",null]},{"value":["2018-01-16",null]},{"value":["2018-01-17",null]},{"value":["2018-01-18",null]},{"value":["2018-01-19",null]},{"value":["2018-01-22",null]},{"value":["2018-01-23",null]},{"value":["2018-01-24",null]},{"value":["2018-01-25",null]},{"value":["2018-01-26",null]},{"value":["2018-01-29",null]},{"value":["2018-01-30",null]},{"value":["2018-01-31",null]},{"value":["2018-02-01",null]},{"value":["2018-02-02",null]},{"value":["2018-02-05",null]},{"value":["2018-02-06",null]},{"value":["2018-02-07",null]},{"value":["2018-02-08",null]},{"value":["2018-02-09",null]},{"value":["2018-02-12",null]},{"value":["2018-02-13",null]},{"value":["2018-02-14",null]},{"value":["2018-02-22",null]},{"value":["2018-02-23",null]},{"value":["2018-02-26",null]},{"value":["2018-02-27",null]},{"value":["2018-02-28",null]},{"value":["2018-03-01",null]},{"value":["2018-03-02",null]},{"value":["2018-03-05",null]},{"value":["2018-03-06",null]},{"value":["2018-03-07",null]},{"value":["2018-03-08",null]},{"value":["2018-03-09",null]},{"value":["2018-03-12",null]},{"value":["2018-03-13",null]},{"value":["2018-03-14",null]},{"value":["2018-03-15",null]},{"value":["2018-03-16",null]},{"value":["2018-03-19",null]},{"value":["2018-03-20",null]},{"value":["2018-03-21",null]},{"value":["2018-03-22",null]},{"value":["2018-03-23",null]},{"value":["2018-03-26",null]},{"value":["2018-03-27",null]},{"value":["2018-03-28",null]},{"value":["2018-03-29",null]},{"value":["2018-03-30",null]},{"value":["2018-04-02",null]},{"value":["2018-04-03",null]},{"value":["2018-04-04",null]},{"value":["2018-04-09",null]},{"value":["2018-04-10",null]},{"value":["2018-04-11",null]},{"value":["2018-04-12",null]},{"value":["2018-04-13",null]},{"value":["2018-04-16",null]},{"value":["2018-04-17",null]},{"value":["2018-04-18",null]},{"value":["2018-04-19",null]},{"value":["2018-04-20",null]},{"value":["2018-04-23",null]},{"value":["2018-04-24",null]},{"value":["2018-04-25",null]},{"value":["2018-04-26",null]},{"value":["2018-04-27",null]},{"value":["2018-05-02",null]},{"value":["2018-05-03",null]},{"value":["2018-05-04",null]},{"value":["2018-05-07",null]},{"value":["2018-05-08",null]},{"value":["2018-05-09",null]},{"value":["2018-05-10",null]},{"value":["2018-05-11",null]},{"value":["2018-05-14",null]},{"value":["2018-05-15",null]},{"value":["2018-05-16",null]},{"value":["2018-05-17",null]},{"value":["2018-05-18",null]},{"value":["2018-05-21",null]},{"value":["2018-05-22",null]},{"value":["2018-05-23",null]},{"value":["2018-05-24",null]},{"value":["2018-05-25",null]},{"value":["2018-05-28",null]},{"value":["2018-05-29",null]},{"value":["2018-05-30",null]},{"value":["2018-05-31",null]},{"value":["2018-06-01",null]},{"value":["2018-06-04",null]},{"value":["2018-06-05",null]},{"value":["2018-06-06",null]},{"value":["2018-06-07",null]},{"value":["2018-06-08",null]},{"value":["2018-06-11",null]},{"value":["2018-06-12",null]},{"value":["2018-06-13",null]},{"value":["2018-06-14",null]},{"value":["2018-06-15",null]},{"value":["2018-06-19",null]},{"value":["2018-06-20",null]},{"value":["2018-06-21",null]},{"value":["2018-06-22",null]},{"value":["2018-06-25",null]},{"value":["2018-06-26",null]},{"value":["2018-06-27",null]},{"value":["2018-06-28",null]},{"value":["2018-06-29",null]},{"value":["2018-07-02",null]},{"value":["2018-07-03",null]},{"value":["2018-07-04",null]},{"value":["2018-07-05",null]},{"value":["2018-07-06",null]},{"value":["2018-07-09",null]},{"value":["2018-07-10",null]},{"value":["2018-07-11",null]},{"value":["2018-07-12",null]},{"value":["2018-07-13",null]},{"value":["2018-07-16",null]},{"value":["2018-07-17",null]},{"value":["2018-07-18",null]},{"value":["2018-07-19",null]},{"value":["2018-07-20",null]},{"value":["2018-07-23",null]},{"value":["2018-07-24",null]},{"value":["2018-07-25",null]},{"value":["2018-07-26",null]},{"value":["2018-07-27",null]},{"value":["2018-07-30",null]},{"value":["2018-07-31",null]},{"value":["2018-08-01",null]},{"value":["2018-08-02",null]},{"value":["2018-08-03",null]},{"value":["2018-08-06",null]},{"value":["2018-08-07",null]},{"value":["2018-08-08",null]},{"value":["2018-08-09",null]},{"value":["2018-08-10",null]},{"value":["2018-08-13",null]},{"value":["2018-08-14",null]},{"value":["2018-08-15",null]},{"value":["2018-08-16",null]},{"value":["2018-08-17",null]},{"value":["2018-08-20",null]},{"value":["2018-08-21",null]},{"value":["2018-08-22",null]},{"value":["2018-08-23",null]},{"value":["2018-08-24",null]},{"value":["2018-08-27",null]},{"value":["2018-08-28",null]},{"value":["2018-08-29",null]},{"value":["2018-08-30",null]},{"value":["2018-08-31",null]},{"value":["2018-09-03",null]},{"value":["2018-09-04",null]},{"value":["2018-09-05",null]},{"value":["2018-09-06",null]},{"value":["2018-09-07",null]},{"value":["2018-09-10",null]},{"value":["2018-09-11",null]},{"value":["2018-09-12",null]},{"value":["2018-09-13",null]},{"value":["2018-09-14",null]},{"value":["2018-09-17",null]},{"value":["2018-09-18",null]},{"value":["2018-09-19",null]},{"value":["2018-09-20",null]},{"value":["2018-09-21",null]},{"value":["2018-09-25",null]},{"value":["2018-09-26",null]},{"value":["2018-09-27",null]},{"value":["2018-09-28",null]},{"value":["2018-10-08",null]},{"value":["2018-10-09",null]},{"value":["2018-10-10",null]},{"value":["2018-10-11",null]},{"value":["2018-10-12",null]},{"value":["2018-10-15",null]},{"value":["2018-10-16",null]},{"value":["2018-10-17",null]},{"value":["2018-10-18",null]},{"value":["2018-10-19",null]},{"value":["2018-10-22",null]},{"value":["2018-10-23",null]},{"value":["2018-10-24",null]},{"value":["2018-10-25",null]},{"value":["2018-10-26",null]},{"value":["2018-10-29",null]},{"value":["2018-10-30",null]},{"value":["2018-10-31",null]},{"value":["2018-11-01",null]},{"value":["2018-11-02",null]},{"value":["2018-11-05",null]},{"value":["2018-11-06",null]},{"value":["2018-11-07",null]},{"value":["2018-11-08",null]},{"value":["2018-11-09",null]},{"value":["2018-11-12",null]},{"value":["2018-11-13",null]},{"value":["2018-11-14",null]},{"value":["2018-11-15",null]},{"value":["2018-11-16",null]},{"value":["2018-11-19",null]},{"value":["2018-11-20",null]},{"value":["2018-11-21",null]},{"value":["2018-11-22",null]},{"value":["2018-11-23",null]},{"value":["2018-11-26",null]},{"value":["2018-11-27",null]},{"value":["2018-11-28",null]},{"value":["2018-11-29",null]},{"value":["2018-11-30",null]},{"value":["2018-12-03",null]},{"value":["2018-12-04",null]},{"value":["2018-12-05",null]},{"value":["2018-12-06",null]},{"value":["2018-12-07",null]},{"value":["2018-12-10",null]},{"value":["2018-12-11",null]},{"value":["2018-12-12",null]},{"value":["2018-12-13",null]},{"value":["2018-12-14",null]},{"value":["2018-12-17",null]},{"value":["2018-12-18",null]},{"value":["2018-12-19",null]},{"value":["2018-12-20",null]},{"value":["2018-12-21",null]},{"value":["2018-12-24",null]},{"value":["2018-12-25",null]},{"value":["2018-12-26","3.610000"]},{"value":["2018-12-27",null]},{"value":["2018-12-28",null]}],"name":"buy","type":"scatter","symbol":"triangle","symbolSize":12,"color":"#CF002F","coordinateSystem":"cartesian2d","symbolSize.1":3,"color.1":"#CF002F","yAxisIndex":0,"xAxisIndex":0,"markLine":{"symbol":"none","lineStyle":{"type":"dashed","color":"grey"},"data":[{"yAxis":3.61}]}},{"data":[{"value":["2013-12-31",null]},{"value":["2014-01-02",null]},{"value":["2014-01-03",null]},{"value":["2014-01-06",null]},{"value":["2014-01-07",null]},{"value":["2014-01-08",null]},{"value":["2014-01-09",null]},{"value":["2014-01-10",null]},{"value":["2014-01-13",null]},{"value":["2014-01-14",null]},{"value":["2014-01-15",null]},{"value":["2014-01-16",null]},{"value":["2014-01-17",null]},{"value":["2014-01-20",null]},{"value":["2014-01-21",null]},{"value":["2014-01-22",null]},{"value":["2014-01-23",null]},{"value":["2014-01-24",null]},{"value":["2014-01-27",null]},{"value":["2014-01-28",null]},{"value":["2014-01-29",null]},{"value":["2014-01-30",null]},{"value":["2014-02-07",null]},{"value":["2014-02-10",null]},{"value":["2014-02-11",null]},{"value":["2014-02-12",null]},{"value":["2014-02-13",null]},{"value":["2014-02-14",null]},{"value":["2014-02-17",null]},{"value":["2014-02-18",null]},{"value":["2014-02-19",null]},{"value":["2014-02-20",null]},{"value":["2014-02-21",null]},{"value":["2014-02-24",null]},{"value":["2014-02-25",null]},{"value":["2014-02-26",null]},{"value":["2014-02-27",null]},{"value":["2014-02-28",null]},{"value":["2014-03-03",null]},{"value":["2014-03-04",null]},{"value":["2014-03-05",null]},{"value":["2014-03-06",null]},{"value":["2014-03-07",null]},{"value":["2014-03-10",null]},{"value":["2014-03-11",null]},{"value":["2014-03-12",null]},{"value":["2014-03-13",null]},{"value":["2014-03-14",null]},{"value":["2014-03-17",null]},{"value":["2014-03-18",null]},{"value":["2014-03-19",null]},{"value":["2014-03-20",null]},{"value":["2014-03-21",null]},{"value":["2014-03-24",null]},{"value":["2014-03-25",null]},{"value":["2014-03-26",null]},{"value":["2014-03-27",null]},{"value":["2014-03-28",null]},{"value":["2014-03-31",null]},{"value":["2014-04-01",null]},{"value":["2014-04-02",null]},{"value":["2014-04-03",null]},{"value":["2014-04-04",null]},{"value":["2014-04-08",null]},{"value":["2014-04-09",null]},{"value":["2014-04-10",null]},{"value":["2014-04-11",null]},{"value":["2014-04-14",null]},{"value":["2014-04-15",null]},{"value":["2014-04-16",null]},{"value":["2014-04-17",null]},{"value":["2014-04-18",null]},{"value":["2014-04-21",null]},{"value":["2014-04-22",null]},{"value":["2014-04-23",null]},{"value":["2014-04-24",null]},{"value":["2014-04-25",null]},{"value":["2014-04-28",null]},{"value":["2014-04-29",null]},{"value":["2014-04-30",null]},{"value":["2014-05-05",null]},{"value":["2014-05-06",null]},{"value":["2014-05-07",null]},{"value":["2014-05-08",null]},{"value":["2014-05-09",null]},{"value":["2014-05-12",null]},{"value":["2014-05-13",null]},{"value":["2014-05-14",null]},{"value":["2014-05-15",null]},{"value":["2014-05-16",null]},{"value":["2014-05-19",null]},{"value":["2014-05-20",null]},{"value":["2014-05-21",null]},{"value":["2014-05-22",null]},{"value":["2014-05-23",null]},{"value":["2014-05-26",null]},{"value":["2014-05-27",null]},{"value":["2014-05-28",null]},{"value":["2014-05-29",null]},{"value":["2014-05-30",null]},{"value":["2014-06-03",null]},{"value":["2014-06-04",null]},{"value":["2014-06-05",null]},{"value":["2014-06-06",null]},{"value":["2014-06-09",null]},{"value":["2014-06-10",null]},{"value":["2014-06-11",null]},{"value":["2014-06-12",null]},{"value":["2014-06-13",null]},{"value":["2014-06-16",null]},{"value":["2014-06-17",null]},{"value":["2014-06-18",null]},{"value":["2014-06-19",null]},{"value":["2014-06-20",null]},{"value":["2014-06-23",null]},{"value":["2014-06-24",null]},{"value":["2014-06-25",null]},{"value":["2014-06-26",null]},{"value":["2014-06-27",null]},{"value":["2014-06-30",null]},{"value":["2014-07-01",null]},{"value":["2014-07-02",null]},{"value":["2014-07-03",null]},{"value":["2014-07-04",null]},{"value":["2014-07-07",null]},{"value":["2014-07-08",null]},{"value":["2014-07-09",null]},{"value":["2014-07-10",null]},{"value":["2014-07-11",null]},{"value":["2014-07-14",null]},{"value":["2014-07-15",null]},{"value":["2014-07-16",null]},{"value":["2014-07-17",null]},{"value":["2014-07-18",null]},{"value":["2014-07-21",null]},{"value":["2014-07-22",null]},{"value":["2014-07-23",null]},{"value":["2014-07-24",null]},{"value":["2014-07-25",null]},{"value":["2014-07-28",null]},{"value":["2014-07-29",null]},{"value":["2014-07-30",null]},{"value":["2014-07-31",null]},{"value":["2014-08-01",null]},{"value":["2014-08-04",null]},{"value":["2014-08-05",null]},{"value":["2014-08-06",null]},{"value":["2014-08-07",null]},{"value":["2014-08-08",null]},{"value":["2014-08-11",null]},{"value":["2014-08-12",null]},{"value":["2014-08-13",null]},{"value":["2014-08-14",null]},{"value":["2014-08-15",null]},{"value":["2014-08-18",null]},{"value":["2014-08-19",null]},{"value":["2014-08-20",null]},{"value":["2014-08-21",null]},{"value":["2014-08-22",null]},{"value":["2014-08-25",null]},{"value":["2014-08-26",null]},{"value":["2014-08-27",null]},{"value":["2014-08-28",null]},{"value":["2014-08-29",null]},{"value":["2014-09-01",null]},{"value":["2014-09-02",null]},{"value":["2014-09-03",null]},{"value":["2014-09-04",null]},{"value":["2014-09-05",null]},{"value":["2014-09-09",null]},{"value":["2014-09-10",null]},{"value":["2014-09-11",null]},{"value":["2014-09-12",null]},{"value":["2014-09-15",null]},{"value":["2014-09-16",null]},{"value":["2014-09-17",null]},{"value":["2014-09-18",null]},{"value":["2014-09-19",null]},{"value":["2014-09-22",null]},{"value":["2014-09-23",null]},{"value":["2014-09-24",null]},{"value":["2014-09-25",null]},{"value":["2014-09-26",null]},{"value":["2014-09-29",null]},{"value":["2014-09-30",null]},{"value":["2014-10-08",null]},{"value":["2014-10-09",null]},{"value":["2014-10-10",null]},{"value":["2014-10-13",null]},{"value":["2014-10-14",null]},{"value":["2014-10-15",null]},{"value":["2014-10-16",null]},{"value":["2014-10-17",null]},{"value":["2014-10-20",null]},{"value":["2014-10-21",null]},{"value":["2014-10-22",null]},{"value":["2014-10-23",null]},{"value":["2014-10-24",null]},{"value":["2014-10-27",null]},{"value":["2014-10-28",null]},{"value":["2014-10-29",null]},{"value":["2014-10-30",null]},{"value":["2014-10-31",null]},{"value":["2014-11-03",null]},{"value":["2014-11-04",null]},{"value":["2014-11-05",null]},{"value":["2014-11-06",null]},{"value":["2014-11-07",null]},{"value":["2014-11-10",null]},{"value":["2014-11-11",null]},{"value":["2014-11-12",null]},{"value":["2014-11-13",null]},{"value":["2014-11-14",null]},{"value":["2014-11-17",null]},{"value":["2014-11-18",null]},{"value":["2014-11-19",null]},{"value":["2014-11-20",null]},{"value":["2014-11-21",null]},{"value":["2014-11-24",null]},{"value":["2014-11-25",null]},{"value":["2014-11-26",null]},{"value":["2014-11-27",null]},{"value":["2014-11-28",null]},{"value":["2014-12-01",null]},{"value":["2014-12-02",null]},{"value":["2014-12-03",null]},{"value":["2014-12-04",null]},{"value":["2014-12-05",null]},{"value":["2014-12-08",null]},{"value":["2014-12-09",null]},{"value":["2014-12-10",null]},{"value":["2014-12-11",null]},{"value":["2014-12-12",null]},{"value":["2014-12-15",null]},{"value":["2014-12-16",null]},{"value":["2014-12-17",null]},{"value":["2014-12-18",null]},{"value":["2014-12-19",null]},{"value":["2014-12-22",null]},{"value":["2014-12-23",null]},{"value":["2014-12-24",null]},{"value":["2014-12-25",null]},{"value":["2014-12-26",null]},{"value":["2014-12-29",null]},{"value":["2014-12-30",null]},{"value":["2014-12-31",null]},{"value":["2015-01-05",null]},{"value":["2015-01-06",null]},{"value":["2015-01-07",null]},{"value":["2015-01-08",null]},{"value":["2015-01-09",null]},{"value":["2015-01-12",null]},{"value":["2015-01-13",null]},{"value":["2015-01-14",null]},{"value":["2015-01-15",null]},{"value":["2015-01-16",null]},{"value":["2015-01-19",null]},{"value":["2015-01-20",null]},{"value":["2015-01-21",null]},{"value":["2015-01-22",null]},{"value":["2015-01-23",null]},{"value":["2015-01-26",null]},{"value":["2015-01-27",null]},{"value":["2015-01-28",null]},{"value":["2015-01-29",null]},{"value":["2015-01-30",null]},{"value":["2015-02-02",null]},{"value":["2015-02-03",null]},{"value":["2015-02-04",null]},{"value":["2015-02-05",null]},{"value":["2015-02-06",null]},{"value":["2015-02-09",null]},{"value":["2015-02-10",null]},{"value":["2015-02-11",null]},{"value":["2015-02-12",null]},{"value":["2015-02-13",null]},{"value":["2015-02-16",null]},{"value":["2015-02-17",null]},{"value":["2015-02-25",null]},{"value":["2015-02-26",null]},{"value":["2015-02-27",null]},{"value":["2015-03-02",null]},{"value":["2015-03-03",null]},{"value":["2015-03-04",null]},{"value":["2015-03-05",null]},{"value":["2015-03-06",null]},{"value":["2015-03-09",null]},{"value":["2015-03-10",null]},{"value":["2015-03-11",null]},{"value":["2015-03-12",null]},{"value":["2015-03-13",null]},{"value":["2015-03-16",null]},{"value":["2015-03-17",null]},{"value":["2015-03-18",null]},{"value":["2015-03-19",null]},{"value":["2015-03-20",null]},{"value":["2015-03-23",null]},{"value":["2015-03-24",null]},{"value":["2015-03-25",null]},{"value":["2015-03-26",null]},{"value":["2015-03-27",null]},{"value":["2015-03-30",null]},{"value":["2015-03-31",null]},{"value":["2015-04-01",null]},{"value":["2015-04-02",null]},{"value":["2015-04-03",null]},{"value":["2015-04-07",null]},{"value":["2015-04-08",null]},{"value":["2015-04-09",null]},{"value":["2015-04-10",null]},{"value":["2015-04-13",null]},{"value":["2015-04-14",null]},{"value":["2015-04-15",null]},{"value":["2015-04-16",null]},{"value":["2015-04-17",null]},{"value":["2015-04-20",null]},{"value":["2015-04-21",null]},{"value":["2015-04-22",null]},{"value":["2015-04-23",null]},{"value":["2015-04-24",null]},{"value":["2015-04-27",null]},{"value":["2015-04-28",null]},{"value":["2015-04-29",null]},{"value":["2015-04-30",null]},{"value":["2015-05-04",null]},{"value":["2015-05-05",null]},{"value":["2015-05-06",null]},{"value":["2015-05-07",null]},{"value":["2015-05-08",null]},{"value":["2015-05-11",null]},{"value":["2015-05-12",null]},{"value":["2015-05-13",null]},{"value":["2015-05-14",null]},{"value":["2015-05-15",null]},{"value":["2015-05-18",null]},{"value":["2015-05-19",null]},{"value":["2015-05-20",null]},{"value":["2015-05-21",null]},{"value":["2015-05-22",null]},{"value":["2015-05-25",null]},{"value":["2015-05-26",null]},{"value":["2015-05-27",null]},{"value":["2015-05-28",null]},{"value":["2015-05-29",null]},{"value":["2015-06-01",null]},{"value":["2015-06-02",null]},{"value":["2015-06-03",null]},{"value":["2015-06-04",null]},{"value":["2015-06-05",null]},{"value":["2015-06-08",null]},{"value":["2015-06-09",null]},{"value":["2015-06-10",null]},{"value":["2015-06-11",null]},{"value":["2015-06-12",null]},{"value":["2015-06-15",null]},{"value":["2015-06-16",null]},{"value":["2015-06-17",null]},{"value":["2015-06-18",null]},{"value":["2015-06-19",null]},{"value":["2015-06-23",null]},{"value":["2015-06-24",null]},{"value":["2015-06-25",null]},{"value":["2015-06-26",null]},{"value":["2015-06-29",null]},{"value":["2015-06-30",null]},{"value":["2015-07-01",null]},{"value":["2015-07-02",null]},{"value":["2015-07-03",null]},{"value":["2015-07-06",null]},{"value":["2015-07-07",null]},{"value":["2015-07-08",null]},{"value":["2015-07-09",null]},{"value":["2015-07-10",null]},{"value":["2015-07-13",null]},{"value":["2015-07-14",null]},{"value":["2015-07-15",null]},{"value":["2015-07-16",null]},{"value":["2015-07-17",null]},{"value":["2015-07-20",null]},{"value":["2015-07-21",null]},{"value":["2015-07-22",null]},{"value":["2015-07-23",null]},{"value":["2015-07-24",null]},{"value":["2015-07-27",null]},{"value":["2015-07-28",null]},{"value":["2015-07-29",null]},{"value":["2015-07-30",null]},{"value":["2015-07-31",null]},{"value":["2015-08-03",null]},{"value":["2015-08-04",null]},{"value":["2015-08-05",null]},{"value":["2015-08-06",null]},{"value":["2015-08-07",null]},{"value":["2015-08-10",null]},{"value":["2015-08-11",null]},{"value":["2015-08-12",null]},{"value":["2015-08-13",null]},{"value":["2015-08-14",null]},{"value":["2015-08-17",null]},{"value":["2015-08-18",null]},{"value":["2015-08-19",null]},{"value":["2015-08-20",null]},{"value":["2015-08-21",null]},{"value":["2015-08-24",null]},{"value":["2015-08-25",null]},{"value":["2015-08-26",null]},{"value":["2015-08-27",null]},{"value":["2015-08-28",null]},{"value":["2015-08-31",null]},{"value":["2015-09-01",null]},{"value":["2015-09-02",null]},{"value":["2015-09-07",null]},{"value":["2015-09-08",null]},{"value":["2015-09-09",null]},{"value":["2015-09-10",null]},{"value":["2015-09-11",null]},{"value":["2015-09-14",null]},{"value":["2015-09-15",null]},{"value":["2015-09-16",null]},{"value":["2015-09-17",null]},{"value":["2015-09-18",null]},{"value":["2015-09-21",null]},{"value":["2015-09-22",null]},{"value":["2015-09-23",null]},{"value":["2015-09-24",null]},{"value":["2015-09-25",null]},{"value":["2015-09-28","3.309507"]},{"value":["2015-09-29",null]},{"value":["2015-09-30",null]},{"value":["2015-10-08",null]},{"value":["2015-10-09",null]},{"value":["2015-10-12",null]},{"value":["2015-10-13",null]},{"value":["2015-10-14",null]},{"value":["2015-10-15",null]},{"value":["2015-10-16",null]},{"value":["2015-10-19",null]},{"value":["2015-10-20",null]},{"value":["2015-10-21",null]},{"value":["2015-10-22",null]},{"value":["2015-10-23",null]},{"value":["2015-10-26",null]},{"value":["2015-10-27",null]},{"value":["2015-10-28",null]},{"value":["2015-10-29",null]},{"value":["2015-10-30",null]},{"value":["2015-11-02",null]},{"value":["2015-11-03",null]},{"value":["2015-11-04",null]},{"value":["2015-11-05",null]},{"value":["2015-11-06",null]},{"value":["2015-11-09",null]},{"value":["2015-11-10",null]},{"value":["2015-11-11",null]},{"value":["2015-11-12",null]},{"value":["2015-11-13",null]},{"value":["2015-11-16",null]},{"value":["2015-11-17",null]},{"value":["2015-11-18",null]},{"value":["2015-11-19",null]},{"value":["2015-11-20",null]},{"value":["2015-11-23",null]},{"value":["2015-11-24",null]},{"value":["2015-11-25",null]},{"value":["2015-11-26",null]},{"value":["2015-11-27",null]},{"value":["2015-11-30",null]},{"value":["2015-12-01",null]},{"value":["2015-12-02",null]},{"value":["2015-12-03",null]},{"value":["2015-12-04",null]},{"value":["2015-12-07",null]},{"value":["2015-12-08",null]},{"value":["2015-12-09",null]},{"value":["2015-12-10",null]},{"value":["2015-12-11",null]},{"value":["2015-12-14",null]},{"value":["2015-12-15",null]},{"value":["2015-12-16",null]},{"value":["2015-12-17",null]},{"value":["2015-12-18",null]},{"value":["2015-12-21",null]},{"value":["2015-12-22",null]},{"value":["2015-12-23",null]},{"value":["2015-12-24",null]},{"value":["2015-12-25",null]},{"value":["2015-12-28",null]},{"value":["2015-12-29",null]},{"value":["2015-12-30",null]},{"value":["2015-12-31",null]},{"value":["2016-01-04",null]},{"value":["2016-01-05",null]},{"value":["2016-01-06",null]},{"value":["2016-01-07",null]},{"value":["2016-01-08",null]},{"value":["2016-01-11",null]},{"value":["2016-01-12",null]},{"value":["2016-01-13",null]},{"value":["2016-01-14",null]},{"value":["2016-01-15",null]},{"value":["2016-01-18",null]},{"value":["2016-01-19",null]},{"value":["2016-01-20",null]},{"value":["2016-01-21",null]},{"value":["2016-01-22",null]},{"value":["2016-01-25",null]},{"value":["2016-01-26",null]},{"value":["2016-01-27",null]},{"value":["2016-01-28",null]},{"value":["2016-01-29",null]},{"value":["2016-02-01",null]},{"value":["2016-02-02",null]},{"value":["2016-02-03",null]},{"value":["2016-02-04",null]},{"value":["2016-02-05",null]},{"value":["2016-02-15",null]},{"value":["2016-02-16",null]},{"value":["2016-02-17",null]},{"value":["2016-02-18",null]},{"value":["2016-02-19",null]},{"value":["2016-02-22",null]},{"value":["2016-02-23",null]},{"value":["2016-02-24",null]},{"value":["2016-02-25",null]},{"value":["2016-02-26",null]},{"value":["2016-02-29",null]},{"value":["2016-03-01",null]},{"value":["2016-03-02",null]},{"value":["2016-03-03",null]},{"value":["2016-03-04",null]},{"value":["2016-03-07",null]},{"value":["2016-03-08",null]},{"value":["2016-03-09",null]},{"value":["2016-03-10",null]},{"value":["2016-03-11",null]},{"value":["2016-03-14",null]},{"value":["2016-03-15",null]},{"value":["2016-03-16",null]},{"value":["2016-03-17",null]},{"value":["2016-03-18",null]},{"value":["2016-03-21",null]},{"value":["2016-03-22",null]},{"value":["2016-03-23",null]},{"value":["2016-03-24",null]},{"value":["2016-03-25",null]},{"value":["2016-03-28",null]},{"value":["2016-03-29",null]},{"value":["2016-03-30",null]},{"value":["2016-03-31",null]},{"value":["2016-04-01",null]},{"value":["2016-04-05",null]},{"value":["2016-04-06",null]},{"value":["2016-04-07",null]},{"value":["2016-04-08",null]},{"value":["2016-04-11",null]},{"value":["2016-04-12",null]},{"value":["2016-04-13",null]},{"value":["2016-04-14",null]},{"value":["2016-04-15",null]},{"value":["2016-04-18",null]},{"value":["2016-04-19",null]},{"value":["2016-04-20",null]},{"value":["2016-04-21",null]},{"value":["2016-04-22",null]},{"value":["2016-04-25",null]},{"value":["2016-04-26",null]},{"value":["2016-04-27",null]},{"value":["2016-04-28",null]},{"value":["2016-04-29",null]},{"value":["2016-05-03",null]},{"value":["2016-05-04",null]},{"value":["2016-05-05",null]},{"value":["2016-05-06",null]},{"value":["2016-05-09",null]},{"value":["2016-05-10",null]},{"value":["2016-05-11",null]},{"value":["2016-05-12",null]},{"value":["2016-05-13",null]},{"value":["2016-05-16",null]},{"value":["2016-05-17",null]},{"value":["2016-05-18",null]},{"value":["2016-05-19",null]},{"value":["2016-05-20",null]},{"value":["2016-05-23",null]},{"value":["2016-05-24",null]},{"value":["2016-05-25",null]},{"value":["2016-05-26",null]},{"value":["2016-05-27",null]},{"value":["2016-05-30",null]},{"value":["2016-05-31",null]},{"value":["2016-06-01",null]},{"value":["2016-06-02",null]},{"value":["2016-06-03",null]},{"value":["2016-06-06",null]},{"value":["2016-06-07",null]},{"value":["2016-06-08",null]},{"value":["2016-06-13",null]},{"value":["2016-06-14",null]},{"value":["2016-06-15",null]},{"value":["2016-06-16",null]},{"value":["2016-06-17",null]},{"value":["2016-06-20",null]},{"value":["2016-06-21",null]},{"value":["2016-06-22",null]},{"value":["2016-06-23",null]},{"value":["2016-06-24",null]},{"value":["2016-06-27",null]},{"value":["2016-06-28",null]},{"value":["2016-06-29",null]},{"value":["2016-06-30",null]},{"value":["2016-07-01",null]},{"value":["2016-07-04",null]},{"value":["2016-07-05",null]},{"value":["2016-07-06",null]},{"value":["2016-07-07",null]},{"value":["2016-07-08",null]},{"value":["2016-07-11",null]},{"value":["2016-07-12",null]},{"value":["2016-07-13",null]},{"value":["2016-07-14",null]},{"value":["2016-07-15",null]},{"value":["2016-07-18",null]},{"value":["2016-07-19",null]},{"value":["2016-07-20",null]},{"value":["2016-07-21",null]},{"value":["2016-07-22",null]},{"value":["2016-07-25",null]},{"value":["2016-07-26",null]},{"value":["2016-07-27",null]},{"value":["2016-07-28",null]},{"value":["2016-07-29",null]},{"value":["2016-08-01",null]},{"value":["2016-08-02",null]},{"value":["2016-08-03",null]},{"value":["2016-08-04",null]},{"value":["2016-08-05",null]},{"value":["2016-08-08",null]},{"value":["2016-08-09",null]},{"value":["2016-08-10",null]},{"value":["2016-08-11",null]},{"value":["2016-08-12",null]},{"value":["2016-08-15",null]},{"value":["2016-08-16",null]},{"value":["2016-08-17",null]},{"value":["2016-08-18",null]},{"value":["2016-08-19",null]},{"value":["2016-08-22",null]},{"value":["2016-08-23",null]},{"value":["2016-08-24",null]},{"value":["2016-08-25",null]},{"value":["2016-08-26",null]},{"value":["2016-08-29",null]},{"value":["2016-08-30",null]},{"value":["2016-08-31",null]},{"value":["2016-09-01",null]},{"value":["2016-09-02",null]},{"value":["2016-09-05",null]},{"value":["2016-09-06",null]},{"value":["2016-09-07",null]},{"value":["2016-09-08",null]},{"value":["2016-09-09",null]},{"value":["2016-09-12",null]},{"value":["2016-09-13",null]},{"value":["2016-09-14",null]},{"value":["2016-09-19",null]},{"value":["2016-09-20",null]},{"value":["2016-09-21",null]},{"value":["2016-09-22",null]},{"value":["2016-09-23",null]},{"value":["2016-09-26",null]},{"value":["2016-09-27",null]},{"value":["2016-09-28",null]},{"value":["2016-09-29",null]},{"value":["2016-09-30",null]},{"value":["2016-10-10",null]},{"value":["2016-10-11",null]},{"value":["2016-10-12",null]},{"value":["2016-10-13",null]},{"value":["2016-10-14",null]},{"value":["2016-10-17",null]},{"value":["2016-10-18",null]},{"value":["2016-10-19",null]},{"value":["2016-10-20",null]},{"value":["2016-10-21",null]},{"value":["2016-10-24",null]},{"value":["2016-10-25",null]},{"value":["2016-10-26",null]},{"value":["2016-10-27",null]},{"value":["2016-10-28",null]},{"value":["2016-10-31",null]},{"value":["2016-11-01",null]},{"value":["2016-11-02",null]},{"value":["2016-11-03",null]},{"value":["2016-11-04",null]},{"value":["2016-11-07",null]},{"value":["2016-11-08",null]},{"value":["2016-11-09",null]},{"value":["2016-11-10",null]},{"value":["2016-11-11",null]},{"value":["2016-11-14",null]},{"value":["2016-11-15",null]},{"value":["2016-11-16",null]},{"value":["2016-11-17",null]},{"value":["2016-11-18",null]},{"value":["2016-11-21",null]},{"value":["2016-11-22",null]},{"value":["2016-11-23",null]},{"value":["2016-11-24",null]},{"value":["2016-11-25",null]},{"value":["2016-11-28",null]},{"value":["2016-11-29",null]},{"value":["2016-11-30",null]},{"value":["2016-12-01",null]},{"value":["2016-12-02",null]},{"value":["2016-12-05",null]},{"value":["2016-12-06",null]},{"value":["2016-12-07",null]},{"value":["2016-12-08",null]},{"value":["2016-12-09",null]},{"value":["2016-12-12",null]},{"value":["2016-12-13",null]},{"value":["2016-12-14",null]},{"value":["2016-12-15",null]},{"value":["2016-12-16",null]},{"value":["2016-12-19",null]},{"value":["2016-12-20",null]},{"value":["2016-12-21",null]},{"value":["2016-12-22",null]},{"value":["2016-12-23",null]},{"value":["2016-12-26",null]},{"value":["2016-12-27",null]},{"value":["2016-12-28",null]},{"value":["2016-12-29",null]},{"value":["2016-12-30",null]},{"value":["2017-01-03",null]},{"value":["2017-01-04",null]},{"value":["2017-01-05",null]},{"value":["2017-01-06",null]},{"value":["2017-01-09",null]},{"value":["2017-01-10",null]},{"value":["2017-01-11",null]},{"value":["2017-01-12",null]},{"value":["2017-01-13",null]},{"value":["2017-01-16",null]},{"value":["2017-01-17",null]},{"value":["2017-01-18",null]},{"value":["2017-01-19",null]},{"value":["2017-01-20",null]},{"value":["2017-01-23",null]},{"value":["2017-01-24",null]},{"value":["2017-01-25",null]},{"value":["2017-01-26",null]},{"value":["2017-02-03",null]},{"value":["2017-02-06",null]},{"value":["2017-02-07",null]},{"value":["2017-02-08",null]},{"value":["2017-02-09",null]},{"value":["2017-02-10",null]},{"value":["2017-02-13",null]},{"value":["2017-02-14",null]},{"value":["2017-02-15",null]},{"value":["2017-02-16",null]},{"value":["2017-02-17",null]},{"value":["2017-02-20",null]},{"value":["2017-02-21",null]},{"value":["2017-02-22",null]},{"value":["2017-02-23",null]},{"value":["2017-02-24",null]},{"value":["2017-02-27",null]},{"value":["2017-02-28",null]},{"value":["2017-03-01",null]},{"value":["2017-03-02",null]},{"value":["2017-03-03",null]},{"value":["2017-03-06",null]},{"value":["2017-03-07",null]},{"value":["2017-03-08",null]},{"value":["2017-03-09",null]},{"value":["2017-03-10",null]},{"value":["2017-03-13",null]},{"value":["2017-03-14",null]},{"value":["2017-03-15",null]},{"value":["2017-03-16",null]},{"value":["2017-03-17",null]},{"value":["2017-03-20",null]},{"value":["2017-03-21",null]},{"value":["2017-03-22",null]},{"value":["2017-03-23",null]},{"value":["2017-03-24",null]},{"value":["2017-03-27",null]},{"value":["2017-03-28",null]},{"value":["2017-03-29",null]},{"value":["2017-03-30",null]},{"value":["2017-03-31",null]},{"value":["2017-04-05",null]},{"value":["2017-04-06",null]},{"value":["2017-04-07",null]},{"value":["2017-04-10",null]},{"value":["2017-04-11",null]},{"value":["2017-04-12",null]},{"value":["2017-04-13",null]},{"value":["2017-04-14",null]},{"value":["2017-04-17",null]},{"value":["2017-04-18",null]},{"value":["2017-04-19",null]},{"value":["2017-04-20",null]},{"value":["2017-04-21",null]},{"value":["2017-04-24",null]},{"value":["2017-04-25",null]},{"value":["2017-04-26",null]},{"value":["2017-04-27",null]},{"value":["2017-04-28",null]},{"value":["2017-05-02",null]},{"value":["2017-05-03",null]},{"value":["2017-05-04",null]},{"value":["2017-05-05",null]},{"value":["2017-05-08",null]},{"value":["2017-05-09",null]},{"value":["2017-05-10",null]},{"value":["2017-05-11",null]},{"value":["2017-05-12",null]},{"value":["2017-05-15",null]},{"value":["2017-05-16",null]},{"value":["2017-05-17",null]},{"value":["2017-05-18",null]},{"value":["2017-05-19",null]},{"value":["2017-05-22",null]},{"value":["2017-05-23",null]},{"value":["2017-05-24",null]},{"value":["2017-05-25",null]},{"value":["2017-05-26",null]},{"value":["2017-05-31",null]},{"value":["2017-06-01",null]},{"value":["2017-06-02",null]},{"value":["2017-06-05",null]},{"value":["2017-06-06",null]},{"value":["2017-06-07",null]},{"value":["2017-06-08",null]},{"value":["2017-06-09",null]},{"value":["2017-06-12",null]},{"value":["2017-06-13",null]},{"value":["2017-06-14",null]},{"value":["2017-06-15",null]},{"value":["2017-06-16",null]},{"value":["2017-06-19",null]},{"value":["2017-06-20",null]},{"value":["2017-06-21",null]},{"value":["2017-06-22",null]},{"value":["2017-06-23",null]},{"value":["2017-06-26",null]},{"value":["2017-06-27",null]},{"value":["2017-06-28",null]},{"value":["2017-06-29",null]},{"value":["2017-06-30",null]},{"value":["2017-07-03",null]},{"value":["2017-07-04",null]},{"value":["2017-07-05",null]},{"value":["2017-07-06",null]},{"value":["2017-07-07",null]},{"value":["2017-07-10",null]},{"value":["2017-07-11",null]},{"value":["2017-07-12",null]},{"value":["2017-07-13",null]},{"value":["2017-07-14",null]},{"value":["2017-07-17",null]},{"value":["2017-07-18",null]},{"value":["2017-07-19",null]},{"value":["2017-07-20",null]},{"value":["2017-07-21",null]},{"value":["2017-07-24",null]},{"value":["2017-07-25",null]},{"value":["2017-07-26",null]},{"value":["2017-07-27",null]},{"value":["2017-07-28",null]},{"value":["2017-07-31",null]},{"value":["2017-08-01",null]},{"value":["2017-08-02",null]},{"value":["2017-08-03",null]},{"value":["2017-08-04",null]},{"value":["2017-08-07",null]},{"value":["2017-08-08",null]},{"value":["2017-08-09",null]},{"value":["2017-08-10",null]},{"value":["2017-08-11",null]},{"value":["2017-08-14",null]},{"value":["2017-08-15",null]},{"value":["2017-08-16",null]},{"value":["2017-08-17",null]},{"value":["2017-08-18",null]},{"value":["2017-08-21",null]},{"value":["2017-08-22",null]},{"value":["2017-08-23",null]},{"value":["2017-08-24",null]},{"value":["2017-08-25",null]},{"value":["2017-08-28",null]},{"value":["2017-08-29",null]},{"value":["2017-08-30",null]},{"value":["2017-08-31",null]},{"value":["2017-09-01",null]},{"value":["2017-09-04",null]},{"value":["2017-09-05",null]},{"value":["2017-09-06",null]},{"value":["2017-09-07",null]},{"value":["2017-09-08",null]},{"value":["2017-09-11",null]},{"value":["2017-09-12",null]},{"value":["2017-09-13",null]},{"value":["2017-09-14",null]},{"value":["2017-09-15",null]},{"value":["2017-09-18",null]},{"value":["2017-09-19",null]},{"value":["2017-09-20",null]},{"value":["2017-09-21",null]},{"value":["2017-09-22",null]},{"value":["2017-09-25",null]},{"value":["2017-09-26",null]},{"value":["2017-09-27",null]},{"value":["2017-09-28",null]},{"value":["2017-09-29",null]},{"value":["2017-10-09",null]},{"value":["2017-10-10",null]},{"value":["2017-10-11",null]},{"value":["2017-10-12",null]},{"value":["2017-10-13",null]},{"value":["2017-10-16",null]},{"value":["2017-10-17",null]},{"value":["2017-10-18",null]},{"value":["2017-10-19",null]},{"value":["2017-10-20",null]},{"value":["2017-10-23",null]},{"value":["2017-10-24",null]},{"value":["2017-10-25",null]},{"value":["2017-10-26",null]},{"value":["2017-10-27",null]},{"value":["2017-10-30",null]},{"value":["2017-10-31",null]},{"value":["2017-11-01",null]},{"value":["2017-11-02",null]},{"value":["2017-11-03",null]},{"value":["2017-11-06",null]},{"value":["2017-11-07",null]},{"value":["2017-11-08",null]},{"value":["2017-11-09",null]},{"value":["2017-11-10",null]},{"value":["2017-11-13",null]},{"value":["2017-11-14",null]},{"value":["2017-11-15",null]},{"value":["2017-11-16",null]},{"value":["2017-11-17",null]},{"value":["2017-11-20",null]},{"value":["2017-11-21",null]},{"value":["2017-11-22",null]},{"value":["2017-11-23",null]},{"value":["2017-11-24",null]},{"value":["2017-11-27",null]},{"value":["2017-11-28",null]},{"value":["2017-11-29",null]},{"value":["2017-11-30",null]},{"value":["2017-12-01",null]},{"value":["2017-12-04",null]},{"value":["2017-12-05",null]},{"value":["2017-12-06",null]},{"value":["2017-12-07",null]},{"value":["2017-12-08",null]},{"value":["2017-12-11",null]},{"value":["2017-12-12",null]},{"value":["2017-12-13",null]},{"value":["2017-12-14",null]},{"value":["2017-12-15",null]},{"value":["2017-12-18",null]},{"value":["2017-12-19",null]},{"value":["2017-12-20",null]},{"value":["2017-12-21",null]},{"value":["2017-12-22",null]},{"value":["2017-12-25",null]},{"value":["2017-12-26",null]},{"value":["2017-12-27",null]},{"value":["2017-12-28",null]},{"value":["2017-12-29",null]},{"value":["2018-01-02",null]},{"value":["2018-01-03",null]},{"value":["2018-01-04",null]},{"value":["2018-01-05",null]},{"value":["2018-01-08",null]},{"value":["2018-01-09",null]},{"value":["2018-01-10",null]},{"value":["2018-01-11",null]},{"value":["2018-01-12",null]},{"value":["2018-01-15",null]},{"value":["2018-01-16",null]},{"value":["2018-01-17",null]},{"value":["2018-01-18",null]},{"value":["2018-01-19",null]},{"value":["2018-01-22",null]},{"value":["2018-01-23",null]},{"value":["2018-01-24",null]},{"value":["2018-01-25",null]},{"value":["2018-01-26",null]},{"value":["2018-01-29",null]},{"value":["2018-01-30",null]},{"value":["2018-01-31",null]},{"value":["2018-02-01",null]},{"value":["2018-02-02",null]},{"value":["2018-02-05",null]},{"value":["2018-02-06",null]},{"value":["2018-02-07",null]},{"value":["2018-02-08",null]},{"value":["2018-02-09",null]},{"value":["2018-02-12",null]},{"value":["2018-02-13",null]},{"value":["2018-02-14",null]},{"value":["2018-02-22",null]},{"value":["2018-02-23",null]},{"value":["2018-02-26",null]},{"value":["2018-02-27",null]},{"value":["2018-02-28",null]},{"value":["2018-03-01",null]},{"value":["2018-03-02",null]},{"value":["2018-03-05",null]},{"value":["2018-03-06",null]},{"value":["2018-03-07",null]},{"value":["2018-03-08",null]},{"value":["2018-03-09",null]},{"value":["2018-03-12",null]},{"value":["2018-03-13",null]},{"value":["2018-03-14",null]},{"value":["2018-03-15",null]},{"value":["2018-03-16",null]},{"value":["2018-03-19",null]},{"value":["2018-03-20",null]},{"value":["2018-03-21",null]},{"value":["2018-03-22",null]},{"value":["2018-03-23",null]},{"value":["2018-03-26",null]},{"value":["2018-03-27",null]},{"value":["2018-03-28",null]},{"value":["2018-03-29",null]},{"value":["2018-03-30",null]},{"value":["2018-04-02",null]},{"value":["2018-04-03",null]},{"value":["2018-04-04",null]},{"value":["2018-04-09",null]},{"value":["2018-04-10",null]},{"value":["2018-04-11",null]},{"value":["2018-04-12",null]},{"value":["2018-04-13",null]},{"value":["2018-04-16",null]},{"value":["2018-04-17",null]},{"value":["2018-04-18",null]},{"value":["2018-04-19",null]},{"value":["2018-04-20",null]},{"value":["2018-04-23",null]},{"value":["2018-04-24",null]},{"value":["2018-04-25",null]},{"value":["2018-04-26",null]},{"value":["2018-04-27",null]},{"value":["2018-05-02",null]},{"value":["2018-05-03","3.629050"]},{"value":["2018-05-04",null]},{"value":["2018-05-07",null]},{"value":["2018-05-08",null]},{"value":["2018-05-09",null]},{"value":["2018-05-10",null]},{"value":["2018-05-11",null]},{"value":["2018-05-14",null]},{"value":["2018-05-15",null]},{"value":["2018-05-16",null]},{"value":["2018-05-17",null]},{"value":["2018-05-18",null]},{"value":["2018-05-21",null]},{"value":["2018-05-22",null]},{"value":["2018-05-23",null]},{"value":["2018-05-24",null]},{"value":["2018-05-25",null]},{"value":["2018-05-28",null]},{"value":["2018-05-29",null]},{"value":["2018-05-30",null]},{"value":["2018-05-31",null]},{"value":["2018-06-01",null]},{"value":["2018-06-04",null]},{"value":["2018-06-05",null]},{"value":["2018-06-06",null]},{"value":["2018-06-07",null]},{"value":["2018-06-08",null]},{"value":["2018-06-11",null]},{"value":["2018-06-12",null]},{"value":["2018-06-13",null]},{"value":["2018-06-14",null]},{"value":["2018-06-15",null]},{"value":["2018-06-19",null]},{"value":["2018-06-20",null]},{"value":["2018-06-21",null]},{"value":["2018-06-22",null]},{"value":["2018-06-25",null]},{"value":["2018-06-26",null]},{"value":["2018-06-27",null]},{"value":["2018-06-28",null]},{"value":["2018-06-29",null]},{"value":["2018-07-02",null]},{"value":["2018-07-03",null]},{"value":["2018-07-04",null]},{"value":["2018-07-05",null]},{"value":["2018-07-06",null]},{"value":["2018-07-09",null]},{"value":["2018-07-10",null]},{"value":["2018-07-11",null]},{"value":["2018-07-12",null]},{"value":["2018-07-13",null]},{"value":["2018-07-16",null]},{"value":["2018-07-17",null]},{"value":["2018-07-18",null]},{"value":["2018-07-19",null]},{"value":["2018-07-20",null]},{"value":["2018-07-23",null]},{"value":["2018-07-24",null]},{"value":["2018-07-25",null]},{"value":["2018-07-26",null]},{"value":["2018-07-27",null]},{"value":["2018-07-30",null]},{"value":["2018-07-31",null]},{"value":["2018-08-01",null]},{"value":["2018-08-02",null]},{"value":["2018-08-03",null]},{"value":["2018-08-06",null]},{"value":["2018-08-07",null]},{"value":["2018-08-08",null]},{"value":["2018-08-09",null]},{"value":["2018-08-10",null]},{"value":["2018-08-13",null]},{"value":["2018-08-14",null]},{"value":["2018-08-15",null]},{"value":["2018-08-16",null]},{"value":["2018-08-17",null]},{"value":["2018-08-20",null]},{"value":["2018-08-21",null]},{"value":["2018-08-22",null]},{"value":["2018-08-23",null]},{"value":["2018-08-24",null]},{"value":["2018-08-27",null]},{"value":["2018-08-28",null]},{"value":["2018-08-29",null]},{"value":["2018-08-30",null]},{"value":["2018-08-31",null]},{"value":["2018-09-03",null]},{"value":["2018-09-04",null]},{"value":["2018-09-05",null]},{"value":["2018-09-06",null]},{"value":["2018-09-07",null]},{"value":["2018-09-10",null]},{"value":["2018-09-11",null]},{"value":["2018-09-12",null]},{"value":["2018-09-13",null]},{"value":["2018-09-14",null]},{"value":["2018-09-17",null]},{"value":["2018-09-18",null]},{"value":["2018-09-19",null]},{"value":["2018-09-20",null]},{"value":["2018-09-21",null]},{"value":["2018-09-25",null]},{"value":["2018-09-26",null]},{"value":["2018-09-27",null]},{"value":["2018-09-28",null]},{"value":["2018-10-08",null]},{"value":["2018-10-09",null]},{"value":["2018-10-10",null]},{"value":["2018-10-11",null]},{"value":["2018-10-12",null]},{"value":["2018-10-15",null]},{"value":["2018-10-16",null]},{"value":["2018-10-17",null]},{"value":["2018-10-18",null]},{"value":["2018-10-19",null]},{"value":["2018-10-22",null]},{"value":["2018-10-23",null]},{"value":["2018-10-24",null]},{"value":["2018-10-25",null]},{"value":["2018-10-26",null]},{"value":["2018-10-29",null]},{"value":["2018-10-30",null]},{"value":["2018-10-31",null]},{"value":["2018-11-01",null]},{"value":["2018-11-02",null]},{"value":["2018-11-05",null]},{"value":["2018-11-06",null]},{"value":["2018-11-07",null]},{"value":["2018-11-08",null]},{"value":["2018-11-09",null]},{"value":["2018-11-12",null]},{"value":["2018-11-13",null]},{"value":["2018-11-14",null]},{"value":["2018-11-15",null]},{"value":["2018-11-16",null]},{"value":["2018-11-19",null]},{"value":["2018-11-20",null]},{"value":["2018-11-21",null]},{"value":["2018-11-22",null]},{"value":["2018-11-23",null]},{"value":["2018-11-26",null]},{"value":["2018-11-27",null]},{"value":["2018-11-28",null]},{"value":["2018-11-29",null]},{"value":["2018-11-30",null]},{"value":["2018-12-03",null]},{"value":["2018-12-04",null]},{"value":["2018-12-05",null]},{"value":["2018-12-06",null]},{"value":["2018-12-07",null]},{"value":["2018-12-10",null]},{"value":["2018-12-11",null]},{"value":["2018-12-12",null]},{"value":["2018-12-13",null]},{"value":["2018-12-14",null]},{"value":["2018-12-17",null]},{"value":["2018-12-18",null]},{"value":["2018-12-19",null]},{"value":["2018-12-20",null]},{"value":["2018-12-21",null]},{"value":["2018-12-24",null]},{"value":["2018-12-25",null]},{"value":["2018-12-26",null]},{"value":["2018-12-27",null]},{"value":["2018-12-28",null]}],"name":"sell","type":"scatter","symbol":"triangle","symbolSize":12,"symbolRotate":180,"color":"#000000","coordinateSystem":"cartesian2d","symbolSize.1":3,"symbolRotate.1":180,"color.1":"#000000","yAxisIndex":0,"xAxisIndex":0,"markLine":{"symbol":"none","lineStyle":{"type":"dashed","color":"grey"},"data":[{"yAxis":3.61}]}},{"data":[{"value":["2013-12-31",null]},{"value":["2014-01-02",null]},{"value":["2014-01-03",null]},{"value":["2014-01-06",null]},{"value":["2014-01-07",null]},{"value":["2014-01-08",null]},{"value":["2014-01-09",null]},{"value":["2014-01-10",null]},{"value":["2014-01-13",null]},{"value":["2014-01-14",null]},{"value":["2014-01-15",null]},{"value":["2014-01-16",null]},{"value":["2014-01-17",null]},{"value":["2014-01-20",null]},{"value":["2014-01-21",null]},{"value":["2014-01-22",null]},{"value":["2014-01-23",null]},{"value":["2014-01-24",null]},{"value":["2014-01-27",null]},{"value":["2014-01-28",null]},{"value":["2014-01-29",null]},{"value":["2014-01-30",null]},{"value":["2014-02-07",null]},{"value":["2014-02-10",null]},{"value":["2014-02-11",null]},{"value":["2014-02-12",null]},{"value":["2014-02-13",null]},{"value":["2014-02-14",null]},{"value":["2014-02-17",null]},{"value":["2014-02-18",null]},{"value":["2014-02-19",null]},{"value":["2014-02-20",null]},{"value":["2014-02-21",null]},{"value":["2014-02-24",null]},{"value":["2014-02-25",null]},{"value":["2014-02-26",null]},{"value":["2014-02-27",null]},{"value":["2014-02-28",null]},{"value":["2014-03-03",null]},{"value":["2014-03-04",null]},{"value":["2014-03-05",null]},{"value":["2014-03-06",null]},{"value":["2014-03-07",null]},{"value":["2014-03-10",null]},{"value":["2014-03-11",null]},{"value":["2014-03-12",null]},{"value":["2014-03-13",null]},{"value":["2014-03-14",null]},{"value":["2014-03-17",null]},{"value":["2014-03-18",null]},{"value":["2014-03-19",null]},{"value":["2014-03-20",null]},{"value":["2014-03-21",null]},{"value":["2014-03-24",null]},{"value":["2014-03-25",null]},{"value":["2014-03-26",null]},{"value":["2014-03-27",null]},{"value":["2014-03-28",null]},{"value":["2014-03-31",null]},{"value":["2014-04-01",null]},{"value":["2014-04-02",null]},{"value":["2014-04-03",null]},{"value":["2014-04-04",null]},{"value":["2014-04-08",null]},{"value":["2014-04-09",null]},{"value":["2014-04-10",null]},{"value":["2014-04-11",null]},{"value":["2014-04-14",null]},{"value":["2014-04-15",null]},{"value":["2014-04-16",null]},{"value":["2014-04-17",null]},{"value":["2014-04-18",null]},{"value":["2014-04-21",null]},{"value":["2014-04-22",null]},{"value":["2014-04-23",null]},{"value":["2014-04-24",null]},{"value":["2014-04-25",null]},{"value":["2014-04-28",null]},{"value":["2014-04-29",null]},{"value":["2014-04-30",null]},{"value":["2014-05-05",null]},{"value":["2014-05-06",null]},{"value":["2014-05-07",null]},{"value":["2014-05-08",null]},{"value":["2014-05-09",null]},{"value":["2014-05-12",null]},{"value":["2014-05-13",null]},{"value":["2014-05-14",null]},{"value":["2014-05-15",null]},{"value":["2014-05-16",null]},{"value":["2014-05-19",null]},{"value":["2014-05-20",null]},{"value":["2014-05-21",null]},{"value":["2014-05-22",null]},{"value":["2014-05-23",null]},{"value":["2014-05-26",null]},{"value":["2014-05-27",null]},{"value":["2014-05-28",null]},{"value":["2014-05-29",null]},{"value":["2014-05-30",null]},{"value":["2014-06-03",null]},{"value":["2014-06-04",null]},{"value":["2014-06-05",null]},{"value":["2014-06-06",null]},{"value":["2014-06-09",null]},{"value":["2014-06-10",null]},{"value":["2014-06-11",null]},{"value":["2014-06-12",null]},{"value":["2014-06-13",null]},{"value":["2014-06-16",null]},{"value":["2014-06-17",null]},{"value":["2014-06-18",null]},{"value":["2014-06-19",null]},{"value":["2014-06-20",null]},{"value":["2014-06-23",null]},{"value":["2014-06-24",null]},{"value":["2014-06-25",null]},{"value":["2014-06-26",null]},{"value":["2014-06-27",null]},{"value":["2014-06-30",null]},{"value":["2014-07-01",null]},{"value":["2014-07-02",null]},{"value":["2014-07-03",null]},{"value":["2014-07-04",null]},{"value":["2014-07-07",null]},{"value":["2014-07-08",null]},{"value":["2014-07-09",null]},{"value":["2014-07-10",null]},{"value":["2014-07-11",null]},{"value":["2014-07-14",null]},{"value":["2014-07-15",null]},{"value":["2014-07-16",null]},{"value":["2014-07-17",null]},{"value":["2014-07-18",null]},{"value":["2014-07-21",null]},{"value":["2014-07-22",null]},{"value":["2014-07-23",null]},{"value":["2014-07-24",null]},{"value":["2014-07-25",null]},{"value":["2014-07-28",null]},{"value":["2014-07-29",null]},{"value":["2014-07-30",null]},{"value":["2014-07-31",null]},{"value":["2014-08-01",null]},{"value":["2014-08-04",null]},{"value":["2014-08-05",null]},{"value":["2014-08-06",null]},{"value":["2014-08-07",null]},{"value":["2014-08-08",null]},{"value":["2014-08-11",null]},{"value":["2014-08-12",null]},{"value":["2014-08-13",null]},{"value":["2014-08-14",null]},{"value":["2014-08-15",null]},{"value":["2014-08-18",null]},{"value":["2014-08-19",null]},{"value":["2014-08-20",null]},{"value":["2014-08-21",null]},{"value":["2014-08-22",null]},{"value":["2014-08-25",null]},{"value":["2014-08-26",null]},{"value":["2014-08-27",null]},{"value":["2014-08-28",null]},{"value":["2014-08-29",null]},{"value":["2014-09-01",null]},{"value":["2014-09-02",null]},{"value":["2014-09-03",null]},{"value":["2014-09-04",null]},{"value":["2014-09-05",null]},{"value":["2014-09-09",null]},{"value":["2014-09-10",null]},{"value":["2014-09-11",null]},{"value":["2014-09-12",null]},{"value":["2014-09-15",null]},{"value":["2014-09-16",null]},{"value":["2014-09-17",null]},{"value":["2014-09-18",null]},{"value":["2014-09-19",null]},{"value":["2014-09-22",null]},{"value":["2014-09-23",null]},{"value":["2014-09-24",null]},{"value":["2014-09-25",null]},{"value":["2014-09-26",null]},{"value":["2014-09-29",null]},{"value":["2014-09-30",null]},{"value":["2014-10-08",null]},{"value":["2014-10-09",null]},{"value":["2014-10-10",null]},{"value":["2014-10-13",null]},{"value":["2014-10-14",null]},{"value":["2014-10-15",null]},{"value":["2014-10-16",null]},{"value":["2014-10-17",null]},{"value":["2014-10-20",null]},{"value":["2014-10-21",null]},{"value":["2014-10-22",null]},{"value":["2014-10-23",null]},{"value":["2014-10-24",null]},{"value":["2014-10-27",null]},{"value":["2014-10-28","2.087185"]},{"value":["2014-10-29","2.088262"]},{"value":["2014-10-30","2.089461"]},{"value":["2014-10-31","2.091227"]},{"value":["2014-11-03","2.093025"]},{"value":["2014-11-04","2.094904"]},{"value":["2014-11-05","2.096696"]},{"value":["2014-11-06","2.098486"]},{"value":["2014-11-07","2.100397"]},{"value":["2014-11-10","2.102567"]},{"value":["2014-11-11","2.105980"]},{"value":["2014-11-12","2.109103"]},{"value":["2014-11-13","2.112347"]},{"value":["2014-11-14","2.115627"]},{"value":["2014-11-17","2.118654"]},{"value":["2014-11-18","2.121308"]},{"value":["2014-11-19","2.123765"]},{"value":["2014-11-20","2.126339"]},{"value":["2014-11-21","2.128998"]},{"value":["2014-11-24","2.131654"]},{"value":["2014-11-25","2.134397"]},{"value":["2014-11-26","2.137317"]},{"value":["2014-11-27","2.140440"]},{"value":["2014-11-28","2.144679"]},{"value":["2014-12-01","2.149049"]},{"value":["2014-12-02","2.154012"]},{"value":["2014-12-03","2.158391"]},{"value":["2014-12-04","2.163229"]},{"value":["2014-12-05","2.168772"]},{"value":["2014-12-08","2.174151"]},{"value":["2014-12-09","2.178197"]},{"value":["2014-12-10","2.182146"]},{"value":["2014-12-11","2.185963"]},{"value":["2014-12-12","2.189897"]},{"value":["2014-12-15","2.193815"]},{"value":["2014-12-16","2.198152"]},{"value":["2014-12-17","2.203187"]},{"value":["2014-12-18","2.207812"]},{"value":["2014-12-19","2.212846"]},{"value":["2014-12-22","2.219374"]},{"value":["2014-12-23","2.225196"]},{"value":["2014-12-24","2.230432"]},{"value":["2014-12-25","2.236254"]},{"value":["2014-12-26","2.242529"]},{"value":["2014-12-29","2.249045"]},{"value":["2014-12-30","2.255935"]},{"value":["2014-12-31","2.263737"]},{"value":["2015-01-05","2.272507"]},{"value":["2015-01-06","2.281935"]},{"value":["2015-01-07","2.291242"]},{"value":["2015-01-08","2.299715"]},{"value":["2015-01-09","2.308770"]},{"value":["2015-01-12","2.317572"]},{"value":["2015-01-13","2.326270"]},{"value":["2015-01-14","2.335554"]},{"value":["2015-01-15","2.346001"]},{"value":["2015-01-16","2.356906"]},{"value":["2015-01-19","2.365655"]},{"value":["2015-01-20","2.374202"]},{"value":["2015-01-21","2.384540"]},{"value":["2015-01-22","2.394669"]},{"value":["2015-01-23","2.404761"]},{"value":["2015-01-26","2.414680"]},{"value":["2015-01-27","2.423898"]},{"value":["2015-01-28","2.432629"]},{"value":["2015-01-29","2.440776"]},{"value":["2015-01-30","2.448768"]},{"value":["2015-02-02","2.455600"]},{"value":["2015-02-03","2.463093"]},{"value":["2015-02-04","2.470156"]},{"value":["2015-02-05","2.477180"]},{"value":["2015-02-06","2.483706"]},{"value":["2015-02-09","2.490315"]},{"value":["2015-02-10","2.497330"]},{"value":["2015-02-11","2.504105"]},{"value":["2015-02-12","2.510915"]},{"value":["2015-02-13","2.517972"]},{"value":["2015-02-16","2.524907"]},{"value":["2015-02-17","2.531845"]},{"value":["2015-02-25","2.538614"]},{"value":["2015-02-26","2.545796"]},{"value":["2015-02-27","2.552853"]},{"value":["2015-03-02","2.559948"]},{"value":["2015-03-03","2.566423"]},{"value":["2015-03-04","2.572610"]},{"value":["2015-03-05","2.578340"]},{"value":["2015-03-06","2.584078"]},{"value":["2015-03-09","2.590610"]},{"value":["2015-03-10","2.596726"]},{"value":["2015-03-11","2.603174"]},{"value":["2015-03-12","2.610329"]},{"value":["2015-03-13","2.617359"]},{"value":["2015-03-16","2.624514"]},{"value":["2015-03-17","2.631756"]},{"value":["2015-03-18","2.639457"]},{"value":["2015-03-19","2.646870"]},{"value":["2015-03-20","2.654702"]},{"value":["2015-03-23","2.662781"]},{"value":["2015-03-24","2.670862"]},{"value":["2015-03-25","2.678362"]},{"value":["2015-03-26","2.685980"]},{"value":["2015-03-27","2.693447"]},{"value":["2015-03-30","2.701496"]},{"value":["2015-03-31","2.709174"]},{"value":["2015-04-01","2.717092"]},{"value":["2015-04-02","2.724725"]},{"value":["2015-04-03","2.732323"]},{"value":["2015-04-07","2.740336"]},{"value":["2015-04-08","2.748522"]},{"value":["2015-04-09","2.756387"]},{"value":["2015-04-10","2.764504"]},{"value":["2015-04-13","2.773236"]},{"value":["2015-04-14","2.781838"]},{"value":["2015-04-15","2.790926"]},{"value":["2015-04-16","2.800272"]},{"value":["2015-04-17","2.810236"]},{"value":["2015-04-20","2.819333"]},{"value":["2015-04-21","2.828519"]},{"value":["2015-04-22","2.838038"]},{"value":["2015-04-23","2.847432"]},{"value":["2015-04-24","2.856203"]},{"value":["2015-04-27","2.865555"]},{"value":["2015-04-28","2.875864"]},{"value":["2015-04-29","2.885756"]},{"value":["2015-04-30","2.895109"]},{"value":["2015-05-04","2.904461"]},{"value":["2015-05-05","2.912816"]},{"value":["2015-05-06","2.920922"]},{"value":["2015-05-07","2.928778"]},{"value":["2015-05-08","2.936842"]},{"value":["2015-05-11","2.945155"]},{"value":["2015-05-12","2.953510"]},{"value":["2015-05-13","2.961782"]},{"value":["2015-05-14","2.970012"]},{"value":["2015-05-15","2.977785"]},{"value":["2015-05-18","2.985101"]},{"value":["2015-05-19","2.992957"]},{"value":["2015-05-20","3.000771"]},{"value":["2015-05-21","3.008461"]},{"value":["2015-05-22","3.016608"]},{"value":["2015-05-25","3.024797"]},{"value":["2015-05-26","3.032944"]},{"value":["2015-05-27","3.041132"]},{"value":["2015-05-28","3.048074"]},{"value":["2015-05-29","3.054808"]},{"value":["2015-06-01","3.062165"]},{"value":["2015-06-02","3.069481"]},{"value":["2015-06-03","3.076838"]},{"value":["2015-06-04","3.085276"]},{"value":["2015-06-05","3.093631"]},{"value":["2015-06-08","3.103856"]},{"value":["2015-06-09","3.114622"]},{"value":["2015-06-10","3.124806"]},{"value":["2015-06-11","3.134449"]},{"value":["2015-06-12","3.144425"]},{"value":["2015-06-15","3.153819"]},{"value":["2015-06-16","3.162714"]},{"value":["2015-06-17","3.172067"]},{"value":["2015-06-18","3.180546"]},{"value":["2015-06-19","3.188111"]},{"value":["2015-06-23","3.196508"]},{"value":["2015-06-24","3.205320"]},{"value":["2015-06-25","3.213924"]},{"value":["2015-06-26","3.221822"]},{"value":["2015-06-29","3.229636"]},{"value":["2015-06-30","3.238906"]},{"value":["2015-07-01","3.247468"]},{"value":["2015-07-02","3.257029"]},{"value":["2015-07-03","3.264992"]},{"value":["2015-07-06","3.274773"]},{"value":["2015-07-07","3.286713"]},{"value":["2015-07-08","3.299600"]},{"value":["2015-07-09","3.312572"]},{"value":["2015-07-10","3.325158"]},{"value":["2015-07-13","3.336404"]},{"value":["2015-07-14","3.347694"]},{"value":["2015-07-15","3.359672"]},{"value":["2015-07-16","3.370962"]},{"value":["2015-07-17","3.381993"]},{"value":["2015-07-20","3.393189"]},{"value":["2015-07-21","3.403997"]},{"value":["2015-07-22","3.414549"]},{"value":["2015-07-23","3.425188"]},{"value":["2015-07-24","3.435652"]},{"value":["2015-07-27","3.444517"]},{"value":["2015-07-28","3.453470"]},{"value":["2015-07-29","3.462293"]},{"value":["2015-07-30","3.470599"]},{"value":["2015-07-31","3.478643"]},{"value":["2015-08-03","3.486771"]},{"value":["2015-08-04","3.495071"]},{"value":["2015-08-05","3.502939"]},{"value":["2015-08-06","3.510679"]},{"value":["2015-08-07","3.518547"]},{"value":["2015-08-10","3.526807"]},{"value":["2015-08-11","3.534934"]},{"value":["2015-08-12","3.542718"]},{"value":["2015-08-13","3.550587"]},{"value":["2015-08-14","3.558411"]},{"value":["2015-08-17","3.566270"]},{"value":["2015-08-18","3.573702"]},{"value":["2015-08-19","3.580667"]},{"value":["2015-08-20","3.587462"]},{"value":["2015-08-21","3.593753"]},{"value":["2015-08-24","3.598616"]},{"value":["2015-08-25","3.602097"]},{"value":["2015-08-26","3.606438"]},{"value":["2015-08-27","3.611814"]},{"value":["2015-08-28","3.617107"]},{"value":["2015-08-31","3.621896"]},{"value":["2015-09-01","3.626300"]},{"value":["2015-09-02","3.631817"]},{"value":["2015-09-07","3.636040"]},{"value":["2015-09-08","3.640392"]},{"value":["2015-09-09","3.645121"]},{"value":["2015-09-10","3.650224"]},{"value":["2015-09-11","3.655066"]},{"value":["2015-09-14","3.659908"]},{"value":["2015-09-15","3.664539"]},{"value":["2015-09-16","3.669341"]},{"value":["2015-09-17","3.673803"]},{"value":["2015-09-18","3.677802"]},{"value":["2015-09-21","3.681634"]},{"value":["2015-09-22","3.684512"]},{"value":["2015-09-23","3.686881"]},{"value":["2015-09-24","3.688458"]},{"value":["2015-09-25","3.690831"]},{"value":["2015-09-28","3.692576"]},{"value":["2015-09-29","3.693310"]},{"value":["2015-09-30","3.694169"]},{"value":["2015-10-08","3.696659"]},{"value":["2015-10-09","3.698986"]},{"value":["2015-10-12","3.701823"]},{"value":["2015-10-13","3.704531"]},{"value":["2015-10-14","3.707489"]},{"value":["2015-10-15","3.710163"]},{"value":["2015-10-16","3.712385"]},{"value":["2015-10-19","3.714896"]},{"value":["2015-10-20","3.717075"]},{"value":["2015-10-21","3.718317"]},{"value":["2015-10-22","3.719877"]},{"value":["2015-10-23","3.722104"]},{"value":["2015-10-26","3.723664"]},{"value":["2015-10-27","3.724722"]},{"value":["2015-10-28","3.725612"]},{"value":["2015-10-29","3.726042"]},{"value":["2015-10-30","3.725555"]},{"value":["2015-11-02","3.723817"]},{"value":["2015-11-03","3.721323"]},{"value":["2015-11-04","3.719432"]},{"value":["2015-11-05","3.718715"]},{"value":["2015-11-06","3.717460"]},{"value":["2015-11-09","3.717274"]},{"value":["2015-11-10","3.716749"]},{"value":["2015-11-11","3.715384"]},{"value":["2015-11-12","3.712639"]},{"value":["2015-11-13","3.709307"]},{"value":["2015-11-16","3.708183"]},{"value":["2015-11-17","3.707228"]},{"value":["2015-11-18","3.704486"]},{"value":["2015-11-19","3.701995"]},{"value":["2015-11-20","3.699375"]},{"value":["2015-11-23","3.696875"]},{"value":["2015-11-24","3.695084"]},{"value":["2015-11-25","3.693799"]},{"value":["2015-11-26","3.693094"]},{"value":["2015-11-27","3.691871"]},{"value":["2015-11-30","3.692461"]},{"value":["2015-12-01","3.692039"]},{"value":["2015-12-02","3.692932"]},{"value":["2015-12-03","3.694516"]},{"value":["2015-12-04","3.695864"]},{"value":["2015-12-07","3.696914"]},{"value":["2015-12-08","3.697413"]},{"value":["2015-12-09","3.698252"]},{"value":["2015-12-10","3.698961"]},{"value":["2015-12-11","3.699332"]},{"value":["2015-12-14","3.700045"]},{"value":["2015-12-15","3.700458"]},{"value":["2015-12-16","3.701078"]},{"value":["2015-12-17","3.701454"]},{"value":["2015-12-18","3.701998"]},{"value":["2015-12-21","3.702714"]},{"value":["2015-12-22","3.703883"]},{"value":["2015-12-23","3.705259"]},{"value":["2015-12-24","3.707005"]},{"value":["2015-12-25","3.708626"]},{"value":["2015-12-28","3.709243"]},{"value":["2015-12-29","3.710362"]},{"value":["2015-12-30","3.711106"]},{"value":["2015-12-31","3.711056"]},{"value":["2016-01-04","3.710526"]},{"value":["2016-01-05","3.709829"]},{"value":["2016-01-06","3.709136"]},{"value":["2016-01-07","3.707512"]},{"value":["2016-01-08","3.706181"]},{"value":["2016-01-11","3.703745"]},{"value":["2016-01-12","3.701014"]},{"value":["2016-01-13","3.698026"]},{"value":["2016-01-14","3.695663"]},{"value":["2016-01-15","3.692783"]},{"value":["2016-01-18","3.689776"]},{"value":["2016-01-19","3.686488"]},{"value":["2016-01-20","3.683274"]},{"value":["2016-01-21","3.679763"]},{"value":["2016-01-22","3.676546"]},{"value":["2016-01-25","3.673244"]},{"value":["2016-01-26","3.668662"]},{"value":["2016-01-27","3.663744"]},{"value":["2016-01-28","3.658734"]},{"value":["2016-01-29","3.653691"]},{"value":["2016-02-01","3.647934"]},{"value":["2016-02-02","3.642514"]},{"value":["2016-02-03","3.636633"]},{"value":["2016-02-04","3.630463"]},{"value":["2016-02-05","3.623710"]},{"value":["2016-02-15","3.617531"]},{"value":["2016-02-16","3.611617"]},{"value":["2016-02-17","3.605414"]},{"value":["2016-02-18","3.599417"]},{"value":["2016-02-19","3.593959"]},{"value":["2016-02-22","3.588180"]},{"value":["2016-02-23","3.581146"]},{"value":["2016-02-24","3.574486"]},{"value":["2016-02-25","3.567848"]},{"value":["2016-02-26","3.561381"]},{"value":["2016-02-29","3.555998"]},{"value":["2016-03-01","3.551077"]},{"value":["2016-03-02","3.546748"]},{"value":["2016-03-03","3.542256"]},{"value":["2016-03-04","3.537991"]},{"value":["2016-03-07","3.533600"]},{"value":["2016-03-08","3.529335"]},{"value":["2016-03-09","3.525242"]},{"value":["2016-03-10","3.521260"]},{"value":["2016-03-11","3.517951"]},{"value":["2016-03-14","3.514103"]},{"value":["2016-03-15","3.510250"]},{"value":["2016-03-16","3.506657"]},{"value":["2016-03-17","3.502481"]},{"value":["2016-03-18","3.497888"]},{"value":["2016-03-21","3.493255"]},{"value":["2016-03-22","3.488616"]},{"value":["2016-03-23","3.485056"]},{"value":["2016-03-24","3.481532"]},{"value":["2016-03-25","3.477344"]},{"value":["2016-03-28","3.473196"]},{"value":["2016-03-29","3.469004"]},{"value":["2016-03-30","3.464116"]},{"value":["2016-03-31","3.459270"]},{"value":["2016-04-01","3.452513"]},{"value":["2016-04-05","3.445343"]},{"value":["2016-04-06","3.438713"]},{"value":["2016-04-07","3.432491"]},{"value":["2016-04-08","3.425809"]},{"value":["2016-04-11","3.419838"]},{"value":["2016-04-12","3.414408"]},{"value":["2016-04-13","3.408609"]},{"value":["2016-04-14","3.403724"]},{"value":["2016-04-15","3.399671"]},{"value":["2016-04-18","3.394698"]},{"value":["2016-04-19","3.389349"]},{"value":["2016-04-20","3.384164"]},{"value":["2016-04-21","3.379764"]},{"value":["2016-04-22","3.375284"]},{"value":["2016-04-25","3.369431"]},{"value":["2016-04-26","3.364204"]},{"value":["2016-04-27","3.357853"]},{"value":["2016-04-28","3.353144"]},{"value":["2016-04-29","3.346447"]},{"value":["2016-05-03","3.337719"]},{"value":["2016-05-04","3.328128"]},{"value":["2016-05-05","3.318493"]},{"value":["2016-05-06","3.308988"]},{"value":["2016-05-09","3.300650"]},{"value":["2016-05-10","3.292181"]},{"value":["2016-05-11","3.283108"]},{"value":["2016-05-12","3.274554"]},{"value":["2016-05-13","3.266258"]},{"value":["2016-05-16","3.258006"]},{"value":["2016-05-17","3.250056"]},{"value":["2016-05-18","3.242280"]},{"value":["2016-05-19","3.234373"]},{"value":["2016-05-20","3.226726"]},{"value":["2016-05-23","3.220807"]},{"value":["2016-05-24","3.214628"]},{"value":["2016-05-25","3.208580"]},{"value":["2016-05-26","3.203006"]},{"value":["2016-05-27","3.197778"]},{"value":["2016-05-30","3.192680"]},{"value":["2016-05-31","3.187841"]},{"value":["2016-06-01","3.183305"]},{"value":["2016-06-02","3.178855"]},{"value":["2016-06-03","3.174318"]},{"value":["2016-06-06","3.169349"]},{"value":["2016-06-07","3.164597"]},{"value":["2016-06-08","3.160190"]},{"value":["2016-06-13","3.155481"]},{"value":["2016-06-14","3.150944"]},{"value":["2016-06-15","3.146580"]},{"value":["2016-06-16","3.142433"]},{"value":["2016-06-17","3.138587"]},{"value":["2016-06-20","3.134872"]},{"value":["2016-06-21","3.131372"]},{"value":["2016-06-22","3.129471"]},{"value":["2016-06-23","3.128953"]},{"value":["2016-06-24","3.127518"]},{"value":["2016-06-27","3.125180"]},{"value":["2016-06-28","3.122797"]},{"value":["2016-06-29","3.120634"]},{"value":["2016-06-30","3.117742"]},{"value":["2016-07-01","3.114074"]},{"value":["2016-07-04","3.111753"]},{"value":["2016-07-05","3.109345"]},{"value":["2016-07-06","3.106850"]},{"value":["2016-07-07","3.104401"]},{"value":["2016-07-08","3.102118"]},{"value":["2016-07-11","3.099926"]},{"value":["2016-07-12","3.098048"]},{"value":["2016-07-13","3.096222"]},{"value":["2016-07-14","3.094522"]},{"value":["2016-07-15","3.092996"]},{"value":["2016-07-18","3.091559"]},{"value":["2016-07-19","3.089945"]},{"value":["2016-07-20","3.088542"]},{"value":["2016-07-21","3.087183"]},{"value":["2016-07-22","3.085650"]},{"value":["2016-07-25","3.084384"]},{"value":["2016-07-26","3.083600"]},{"value":["2016-07-27","3.082725"]},{"value":["2016-07-28","3.081545"]},{"value":["2016-07-29","3.080324"]},{"value":["2016-08-01","3.078846"]},{"value":["2016-08-02","3.077544"]},{"value":["2016-08-03","3.076107"]},{"value":["2016-08-04","3.074406"]},{"value":["2016-08-05","3.072621"]},{"value":["2016-08-08","3.070925"]},{"value":["2016-08-09","3.069320"]},{"value":["2016-08-10","3.067151"]},{"value":["2016-08-11","3.065598"]},{"value":["2016-08-12","3.064458"]},{"value":["2016-08-15","3.063633"]},{"value":["2016-08-16","3.062394"]},{"value":["2016-08-17","3.061153"]},{"value":["2016-08-18","3.059907"]},{"value":["2016-08-19","3.058750"]},{"value":["2016-08-22","3.057814"]},{"value":["2016-08-23","3.057096"]},{"value":["2016-08-24","3.055860"]},{"value":["2016-08-25","3.054321"]},{"value":["2016-08-26","3.052693"]},{"value":["2016-08-29","3.050197"]},{"value":["2016-08-30","3.047964"]},{"value":["2016-08-31","3.046039"]},{"value":["2016-09-01","3.044375"]},{"value":["2016-09-02","3.042795"]},{"value":["2016-09-05","3.041040"]},{"value":["2016-09-06","3.039242"]},{"value":["2016-09-07","3.037355"]},{"value":["2016-09-08","3.035425"]},{"value":["2016-09-09","3.033579"]},{"value":["2016-09-12","3.031590"]},{"value":["2016-09-13","3.029514"]},{"value":["2016-09-14","3.027084"]},{"value":["2016-09-19","3.024742"]},{"value":["2016-09-20","3.022873"]},{"value":["2016-09-21","3.020495"]},{"value":["2016-09-22","3.018551"]},{"value":["2016-09-23","3.015961"]},{"value":["2016-09-26","3.012544"]},{"value":["2016-09-27","3.009906"]},{"value":["2016-09-28","3.007394"]},{"value":["2016-09-29","3.005186"]},{"value":["2016-09-30","3.002763"]},{"value":["2016-10-10","3.000648"]},{"value":["2016-10-11","2.998664"]},{"value":["2016-10-12","2.996374"]},{"value":["2016-10-13","2.994299"]},{"value":["2016-10-14","2.992223"]},{"value":["2016-10-17","2.989926"]},{"value":["2016-10-18","2.987723"]},{"value":["2016-10-19","2.985302"]},{"value":["2016-10-20","2.983056"]},{"value":["2016-10-21","2.980898"]},{"value":["2016-10-24","2.978964"]},{"value":["2016-10-25","2.977074"]},{"value":["2016-10-26","2.975222"]},{"value":["2016-10-27","2.973191"]},{"value":["2016-10-28","2.971295"]},{"value":["2016-10-31","2.969395"]},{"value":["2016-11-01","2.968144"]},{"value":["2016-11-02","2.966846"]},{"value":["2016-11-03","2.965419"]},{"value":["2016-11-04","2.964378"]},{"value":["2016-11-07","2.963248"]},{"value":["2016-11-08","2.962857"]},{"value":["2016-11-09","2.962462"]},{"value":["2016-11-10","2.962373"]},{"value":["2016-11-11","2.962242"]},{"value":["2016-11-14","2.962677"]},{"value":["2016-11-15","2.963152"]},{"value":["2016-11-16","2.963326"]},{"value":["2016-11-17","2.963758"]},{"value":["2016-11-18","2.964318"]},{"value":["2016-11-21","2.964928"]},{"value":["2016-11-22","2.965672"]},{"value":["2016-11-23","2.967371"]},{"value":["2016-11-24","2.969157"]},{"value":["2016-11-25","2.971383"]},{"value":["2016-11-28","2.973578"]},{"value":["2016-11-29","2.976081"]},{"value":["2016-11-30","2.978226"]},{"value":["2016-12-01","2.980546"]},{"value":["2016-12-02","2.982734"]},{"value":["2016-12-05","2.984965"]},{"value":["2016-12-06","2.987367"]},{"value":["2016-12-07","2.989337"]},{"value":["2016-12-08","2.991355"]},{"value":["2016-12-09","2.993734"]},{"value":["2016-12-12","2.996384"]},{"value":["2016-12-13","2.998459"]},{"value":["2016-12-14","3.000795"]},{"value":["2016-12-15","3.002631"]},{"value":["2016-12-16","3.004894"]},{"value":["2016-12-19","3.007027"]},{"value":["2016-12-20","3.008892"]},{"value":["2016-12-21","3.010810"]},{"value":["2016-12-22","3.012423"]},{"value":["2016-12-23","3.013858"]},{"value":["2016-12-26","3.014820"]},{"value":["2016-12-27","3.015690"]},{"value":["2016-12-28","3.016561"]},{"value":["2016-12-29","3.017303"]},{"value":["2016-12-30","3.018481"]},{"value":["2017-01-03","3.019580"]},{"value":["2017-01-04","3.020679"]},{"value":["2017-01-05","3.021819"]},{"value":["2017-01-06","3.022608"]},{"value":["2017-01-09","3.023441"]},{"value":["2017-01-10","3.024228"]},{"value":["2017-01-11","3.025016"]},{"value":["2017-01-12","3.025933"]},{"value":["2017-01-13","3.027029"]},{"value":["2017-01-16","3.028481"]},{"value":["2017-01-17","3.029889"]},{"value":["2017-01-18","3.031386"]},{"value":["2017-01-19","3.032971"]},{"value":["2017-01-20","3.034298"]},{"value":["2017-01-23","3.035579"]},{"value":["2017-01-24","3.036953"]},{"value":["2017-01-25","3.038241"]},{"value":["2017-01-26","3.039754"]},{"value":["2017-02-03","3.041303"]},{"value":["2017-02-06","3.042939"]},{"value":["2017-02-07","3.044399"]},{"value":["2017-02-08","3.045905"]},{"value":["2017-02-09","3.047373"]},{"value":["2017-02-10","3.048977"]},{"value":["2017-02-13","3.050717"]},{"value":["2017-02-14","3.052542"]},{"value":["2017-02-15","3.054637"]},{"value":["2017-02-16","3.056910"]},{"value":["2017-02-17","3.058996"]},{"value":["2017-02-20","3.061312"]},{"value":["2017-02-21","3.063719"]},{"value":["2017-02-22","3.066128"]},{"value":["2017-02-23","3.068626"]},{"value":["2017-02-24","3.070992"]},{"value":["2017-02-27","3.073308"]},{"value":["2017-02-28","3.075492"]},{"value":["2017-03-01","3.077631"]},{"value":["2017-03-02","3.079769"]},{"value":["2017-03-03","3.081989"]},{"value":["2017-03-06","3.084335"]},{"value":["2017-03-07","3.086769"]},{"value":["2017-03-08","3.089203"]},{"value":["2017-03-09","3.091723"]},{"value":["2017-03-10","3.094198"]},{"value":["2017-03-13","3.096855"]},{"value":["2017-03-14","3.099510"]},{"value":["2017-03-15","3.102165"]},{"value":["2017-03-16","3.104954"]},{"value":["2017-03-17","3.107563"]},{"value":["2017-03-20","3.110176"]},{"value":["2017-03-21","3.112786"]},{"value":["2017-03-22","3.115211"]},{"value":["2017-03-23","3.117635"]},{"value":["2017-03-24","3.120154"]},{"value":["2017-03-27","3.122633"]},{"value":["2017-03-28","3.124678"]},{"value":["2017-03-29","3.126856"]},{"value":["2017-03-30","3.129121"]},{"value":["2017-03-31","3.131526"]},{"value":["2017-04-05","3.133840"]},{"value":["2017-04-06","3.136019"]},{"value":["2017-04-07","3.138064"]},{"value":["2017-04-10","3.140323"]},{"value":["2017-04-11","3.142450"]},{"value":["2017-04-12","3.144577"]},{"value":["2017-04-13","3.146836"]},{"value":["2017-04-14","3.148961"]},{"value":["2017-04-17","3.150997"]},{"value":["2017-04-18","3.152810"]},{"value":["2017-04-19","3.154355"]},{"value":["2017-04-20","3.155765"]},{"value":["2017-04-21","3.157494"]},{"value":["2017-04-24","3.159269"]},{"value":["2017-04-25","3.161089"]},{"value":["2017-04-26","3.162863"]},{"value":["2017-04-27","3.164592"]},{"value":["2017-04-28","3.166276"]},{"value":["2017-05-02","3.167686"]},{"value":["2017-05-03","3.168960"]},{"value":["2017-05-04","3.170234"]},{"value":["2017-05-05","3.171508"]},{"value":["2017-05-08","3.173055"]},{"value":["2017-05-09","3.174511"]},{"value":["2017-05-10","3.175785"]},{"value":["2017-05-11","3.177060"]},{"value":["2017-05-12","3.178652"]},{"value":["2017-05-15","3.180154"]},{"value":["2017-05-16","3.181473"]},{"value":["2017-05-17","3.182838"]},{"value":["2017-05-18","3.184249"]},{"value":["2017-05-19","3.185659"]},{"value":["2017-05-22","3.187161"]},{"value":["2017-05-23","3.188753"]},{"value":["2017-05-24","3.190118"]},{"value":["2017-05-25","3.191756"]},{"value":["2017-05-26","3.193440"]},{"value":["2017-05-31","3.195214"]},{"value":["2017-06-01","3.196989"]},{"value":["2017-06-02","3.198627"]},{"value":["2017-06-05","3.199946"]},{"value":["2017-06-06","3.201402"]},{"value":["2017-06-07","3.202949"]},{"value":["2017-06-08","3.204451"]},{"value":["2017-06-09","3.205816"]},{"value":["2017-06-12","3.207135"]},{"value":["2017-06-13","3.208227"]},{"value":["2017-06-14","3.208819"]},{"value":["2017-06-15","3.209046"]},{"value":["2017-06-16","3.209820"]},{"value":["2017-06-19","3.210593"]},{"value":["2017-06-20","3.211503"]},{"value":["2017-06-21","3.212368"]},{"value":["2017-06-22","3.213278"]},{"value":["2017-06-23","3.214233"]},{"value":["2017-06-26","3.215143"]},{"value":["2017-06-27","3.216053"]},{"value":["2017-06-28","3.217054"]},{"value":["2017-06-29","3.218192"]},{"value":["2017-06-30","3.219329"]},{"value":["2017-07-03","3.220330"]},{"value":["2017-07-04","3.221240"]},{"value":["2017-07-05","3.222196"]},{"value":["2017-07-06","3.223151"]},{"value":["2017-07-07","3.224198"]},{"value":["2017-07-10","3.225290"]},{"value":["2017-07-11","3.226518"]},{"value":["2017-07-12","3.227929"]},{"value":["2017-07-13","3.229885"]},{"value":["2017-07-14","3.232032"]},{"value":["2017-07-17","3.234885"]},{"value":["2017-07-18","3.237550"]},{"value":["2017-07-19","3.240260"]},{"value":["2017-07-20","3.242931"]},{"value":["2017-07-21","3.245508"]},{"value":["2017-07-24","3.248136"]},{"value":["2017-07-25","3.251138"]},{"value":["2017-07-26","3.254333"]},{"value":["2017-07-27","3.257476"]},{"value":["2017-07-28","3.260621"]},{"value":["2017-07-31","3.263956"]},{"value":["2017-08-01","3.267346"]},{"value":["2017-08-02","3.271164"]},{"value":["2017-08-03","3.275075"]},{"value":["2017-08-04","3.278889"]},{"value":["2017-08-07","3.282608"]},{"value":["2017-08-08","3.286703"]},{"value":["2017-08-09","3.290234"]},{"value":["2017-08-10","3.293809"]},{"value":["2017-08-11","3.297197"]},{"value":["2017-08-14","3.300348"]},{"value":["2017-08-15","3.303982"]},{"value":["2017-08-16","3.307523"]},{"value":["2017-08-17","3.311153"]},{"value":["2017-08-18","3.314778"]},{"value":["2017-08-21","3.318265"]},{"value":["2017-08-22","3.321891"]},{"value":["2017-08-23","3.325757"]},{"value":["2017-08-24","3.329713"]},{"value":["2017-08-25","3.334384"]},{"value":["2017-08-28","3.339196"]},{"value":["2017-08-29","3.344482"]},{"value":["2017-08-30","3.349439"]},{"value":["2017-08-31","3.354297"]},{"value":["2017-09-01","3.358920"]},{"value":["2017-09-04","3.363591"]},{"value":["2017-09-05","3.368457"]},{"value":["2017-09-06","3.373321"]},{"value":["2017-09-07","3.377994"]},{"value":["2017-09-08","3.382572"]},{"value":["2017-09-11","3.387196"]},{"value":["2017-09-12","3.391683"]},{"value":["2017-09-13","3.395984"]},{"value":["2017-09-14","3.399908"]},{"value":["2017-09-15","3.403546"]},{"value":["2017-09-18","3.407002"]},{"value":["2017-09-19","3.410326"]},{"value":["2017-09-20","3.413798"]},{"value":["2017-09-21","3.417639"]},{"value":["2017-09-22","3.421578"]},{"value":["2017-09-25","3.425753"]},{"value":["2017-09-26","3.429785"]},{"value":["2017-09-27","3.433624"]},{"value":["2017-09-28","3.437416"]},{"value":["2017-09-29","3.441022"]},{"value":["2017-10-09","3.444309"]},{"value":["2017-10-10","3.447463"]},{"value":["2017-10-11","3.451034"]},{"value":["2017-10-12","3.454512"]},{"value":["2017-10-13","3.458394"]},{"value":["2017-10-16","3.462416"]},{"value":["2017-10-17","3.466390"]},{"value":["2017-10-18","3.470688"]},{"value":["2017-10-19","3.474900"]},{"value":["2017-10-20","3.479062"]},{"value":["2017-10-23","3.483030"]},{"value":["2017-10-24","3.486999"]},{"value":["2017-10-25","3.490867"]},{"value":["2017-10-26","3.494356"]},{"value":["2017-10-27","3.497988"]},{"value":["2017-10-30","3.501624"]},{"value":["2017-10-31","3.504600"]},{"value":["2017-11-01","3.507433"]},{"value":["2017-11-02","3.510216"]},{"value":["2017-11-03","3.513042"]},{"value":["2017-11-06","3.515821"]},{"value":["2017-11-07","3.518836"]},{"value":["2017-11-08","3.522042"]},{"value":["2017-11-09","3.525104"]},{"value":["2017-11-10","3.527792"]},{"value":["2017-11-13","3.530393"]},{"value":["2017-11-14","3.532947"]},{"value":["2017-11-15","3.535407"]},{"value":["2017-11-16","3.537679"]},{"value":["2017-11-17","3.540332"]},{"value":["2017-11-20","3.543126"]},{"value":["2017-11-21","3.545688"]},{"value":["2017-11-22","3.548774"]},{"value":["2017-11-23","3.551535"]},{"value":["2017-11-24","3.554432"]},{"value":["2017-11-27","3.557092"]},{"value":["2017-11-28","3.559654"]},{"value":["2017-11-29","3.562218"]},{"value":["2017-11-30","3.564691"]},{"value":["2017-12-01","3.566980"]},{"value":["2017-12-04","3.569133"]},{"value":["2017-12-05","3.571616"]},{"value":["2017-12-06","3.573777"]},{"value":["2017-12-07","3.575752"]},{"value":["2017-12-08","3.577905"]},{"value":["2017-12-11","3.579832"]},{"value":["2017-12-12","3.581430"]},{"value":["2017-12-13","3.583078"]},{"value":["2017-12-14","3.584632"]},{"value":["2017-12-15","3.586089"]},{"value":["2017-12-18","3.587683"]},{"value":["2017-12-19","3.589465"]},{"value":["2017-12-20","3.591341"]},{"value":["2017-12-21","3.593216"]},{"value":["2017-12-22","3.595182"]},{"value":["2017-12-25","3.597335"]},{"value":["2017-12-26","3.599585"]},{"value":["2017-12-27","3.601692"]},{"value":["2017-12-28","3.603846"]},{"value":["2017-12-29","3.606237"]},{"value":["2018-01-02","3.608493"]},{"value":["2018-01-03","3.610890"]},{"value":["2018-01-04","3.613287"]},{"value":["2018-01-05","3.615498"]},{"value":["2018-01-08","3.617988"]},{"value":["2018-01-09","3.620149"]},{"value":["2018-01-10","3.622734"]},{"value":["2018-01-11","3.625833"]},{"value":["2018-01-12","3.628932"]},{"value":["2018-01-15","3.632323"]},{"value":["2018-01-16","3.635813"]},{"value":["2018-01-17","3.639444"]},{"value":["2018-01-18","3.643601"]},{"value":["2018-01-19","3.647903"]},{"value":["2018-01-22","3.652023"]},{"value":["2018-01-23","3.657567"]},{"value":["2018-01-24","3.663251"]},{"value":["2018-01-25","3.668739"]},{"value":["2018-01-26","3.674368"]},{"value":["2018-01-29","3.679805"]},{"value":["2018-01-30","3.684670"]},{"value":["2018-01-31","3.689775"]},{"value":["2018-02-01","3.695400"]},{"value":["2018-02-02","3.700832"]},{"value":["2018-02-05","3.707258"]},{"value":["2018-02-06","3.713866"]},{"value":["2018-02-07","3.719756"]},{"value":["2018-02-08","3.724699"]},{"value":["2018-02-09","3.728887"]},{"value":["2018-02-12","3.732693"]},{"value":["2018-02-13","3.736788"]},{"value":["2018-02-14","3.740837"]},{"value":["2018-02-22","3.745315"]},{"value":["2018-02-23","3.750024"]},{"value":["2018-02-26","3.754918"]},{"value":["2018-02-27","3.759430"]},{"value":["2018-02-28","3.763468"]},{"value":["2018-03-01","3.767323"]},{"value":["2018-03-02","3.770939"]},{"value":["2018-03-05","3.774462"]},{"value":["2018-03-06","3.777946"]},{"value":["2018-03-07","3.781345"]},{"value":["2018-03-08","3.784695"]},{"value":["2018-03-09","3.788085"]},{"value":["2018-03-12","3.791474"]},{"value":["2018-03-13","3.795003"]},{"value":["2018-03-14","3.798438"]},{"value":["2018-03-15","3.801781"]},{"value":["2018-03-16","3.804706"]},{"value":["2018-03-19","3.807722"]},{"value":["2018-03-20","3.810747"]},{"value":["2018-03-21","3.813771"]},{"value":["2018-03-22","3.816659"]},{"value":["2018-03-23","3.819264"]},{"value":["2018-03-26","3.821340"]},{"value":["2018-03-27","3.823731"]},{"value":["2018-03-28","3.826028"]},{"value":["2018-03-29","3.828332"]},{"value":["2018-03-30","3.830351"]},{"value":["2018-04-02","3.832176"]},{"value":["2018-04-03","3.834093"]},{"value":["2018-04-04","3.835866"]},{"value":["2018-04-09","3.837685"]},{"value":["2018-04-10","3.839976"]},{"value":["2018-04-11","3.842413"]},{"value":["2018-04-12","3.844608"]},{"value":["2018-04-13","3.846568"]},{"value":["2018-04-16","3.848052"]},{"value":["2018-04-17","3.849352"]},{"value":["2018-04-18","3.850655"]},{"value":["2018-04-19","3.852052"]},{"value":["2018-04-20","3.853401"]},{"value":["2018-04-23","3.854705"]},{"value":["2018-04-24","3.856249"]},{"value":["2018-04-25","3.857512"]},{"value":["2018-04-26","3.858772"]},{"value":["2018-04-27","3.860221"]},{"value":["2018-05-02","3.861622"]},{"value":["2018-05-03","3.863116"]},{"value":["2018-05-04","3.864424"]},{"value":["2018-05-07","3.865875"]},{"value":["2018-05-08","3.867380"]},{"value":["2018-05-09","3.868796"]},{"value":["2018-05-10","3.869939"]},{"value":["2018-05-11","3.870987"]},{"value":["2018-05-14","3.871511"]},{"value":["2018-05-15","3.872130"]},{"value":["2018-05-16","3.872558"]},{"value":["2018-05-17","3.872844"]},{"value":["2018-05-18","3.873416"]},{"value":["2018-05-21","3.873940"]},{"value":["2018-05-22","3.874082"]},{"value":["2018-05-23","3.873797"]},{"value":["2018-05-24","3.873511"]},{"value":["2018-05-25","3.873225"]},{"value":["2018-05-28","3.872749"]},{"value":["2018-05-29","3.872130"]},{"value":["2018-05-30","3.870749"]},{"value":["2018-05-31","3.869368"]},{"value":["2018-06-01","3.868082"]},{"value":["2018-06-04","3.867034"]},{"value":["2018-06-05","3.865700"]},{"value":["2018-06-06","3.864700"]},{"value":["2018-06-07","3.863748"]},{"value":["2018-06-08","3.862748"]},{"value":["2018-06-11","3.861938"]},{"value":["2018-06-12","3.860509"]},{"value":["2018-06-13","3.859080"]},{"value":["2018-06-14","3.857699"]},{"value":["2018-06-15","3.856509"]},{"value":["2018-06-19","3.855413"]},{"value":["2018-06-20","3.854127"]},{"value":["2018-06-21","3.852603"]},{"value":["2018-06-22","3.851079"]},{"value":["2018-06-25","3.848651"]},{"value":["2018-06-26","3.845888"]},{"value":["2018-06-27","3.842554"]},{"value":["2018-06-28","3.839173"]},{"value":["2018-06-29","3.836220"]},{"value":["2018-07-02","3.832744"]},{"value":["2018-07-03","3.829410"]},{"value":["2018-07-04","3.825886"]},{"value":["2018-07-05","3.822647"]},{"value":["2018-07-06","3.819694"]},{"value":["2018-07-09","3.817551"]},{"value":["2018-07-10","3.815313"]},{"value":["2018-07-11","3.813027"]},{"value":["2018-07-12","3.811169"]},{"value":["2018-07-13","3.809348"]},{"value":["2018-07-16","3.807512"]},{"value":["2018-07-17","3.805576"]},{"value":["2018-07-18","3.803445"]},{"value":["2018-07-19","3.801026"]},{"value":["2018-07-20","3.798963"]},{"value":["2018-07-23","3.796958"]},{"value":["2018-07-24","3.794813"]},{"value":["2018-07-25","3.792760"]},{"value":["2018-07-26","3.790846"]},{"value":["2018-07-27","3.788929"]},{"value":["2018-07-30","3.787307"]},{"value":["2018-07-31","3.785786"]},{"value":["2018-08-01","3.783869"]},{"value":["2018-08-02","3.781459"]},{"value":["2018-08-03","3.779097"]},{"value":["2018-08-06","3.776930"]},{"value":["2018-08-07","3.774965"]},{"value":["2018-08-08","3.772899"]},{"value":["2018-08-09","3.770839"]},{"value":["2018-08-10","3.768584"]},{"value":["2018-08-13","3.766224"]},{"value":["2018-08-14","3.764200"]},{"value":["2018-08-15","3.761828"]},{"value":["2018-08-16","3.759795"]},{"value":["2018-08-17","3.757995"]},{"value":["2018-08-20","3.756302"]},{"value":["2018-08-21","3.754613"]},{"value":["2018-08-22","3.753349"]},{"value":["2018-08-23","3.752228"]},{"value":["2018-08-24","3.751351"]},{"value":["2018-08-27","3.750623"]},{"value":["2018-08-28","3.749992"]},{"value":["2018-08-29","3.749170"]},{"value":["2018-08-30","3.748008"]},{"value":["2018-08-31","3.747089"]},{"value":["2018-09-03","3.746408"]},{"value":["2018-09-04","3.745782"]},{"value":["2018-09-05","3.744953"]},{"value":["2018-09-06","3.744022"]},{"value":["2018-09-07","3.743284"]},{"value":["2018-09-10","3.742215"]},{"value":["2018-09-11","3.741001"]},{"value":["2018-09-12","3.739781"]},{"value":["2018-09-13","3.738288"]},{"value":["2018-09-14","3.736938"]},{"value":["2018-09-17","3.735488"]},{"value":["2018-09-18","3.734426"]},{"value":["2018-09-19","3.733557"]},{"value":["2018-09-20","3.732690"]},{"value":["2018-09-21","3.732373"]},{"value":["2018-09-25","3.731954"]},{"value":["2018-09-26","3.731685"]},{"value":["2018-09-27","3.731130"]},{"value":["2018-09-28","3.730871"]},{"value":["2018-10-08","3.730206"]},{"value":["2018-10-09","3.729487"]},{"value":["2018-10-10","3.728770"]},{"value":["2018-10-11","3.727792"]},{"value":["2018-10-12","3.726968"]},{"value":["2018-10-15","3.725942"]},{"value":["2018-10-16","3.725108"]},{"value":["2018-10-17","3.724425"]},{"value":["2018-10-18","3.723499"]},{"value":["2018-10-19","3.722975"]},{"value":["2018-10-22","3.722901"]},{"value":["2018-10-23","3.722427"]},{"value":["2018-10-24","3.722158"]},{"value":["2018-10-25","3.721896"]},{"value":["2018-10-26","3.721777"]},{"value":["2018-10-29","3.721560"]},{"value":["2018-10-30","3.721353"]},{"value":["2018-10-31","3.720998"]},{"value":["2018-11-01","3.720298"]},{"value":["2018-11-02","3.719798"]},{"value":["2018-11-05","3.719243"]},{"value":["2018-11-06","3.718445"]},{"value":["2018-11-07","3.717885"]},{"value":["2018-11-08","3.717042"]},{"value":["2018-11-09","3.715514"]},{"value":["2018-11-12","3.714085"]},{"value":["2018-11-13","3.712277"]},{"value":["2018-11-14","3.710080"]},{"value":["2018-11-15","3.707886"]},{"value":["2018-11-16","3.705122"]},{"value":["2018-11-19","3.702367"]},{"value":["2018-11-20","3.699411"]},{"value":["2018-11-21","3.695173"]},{"value":["2018-11-22","3.690887"]},{"value":["2018-11-23","3.686786"]},{"value":["2018-11-26","3.682640"]},{"value":["2018-11-27","3.678633"]},{"value":["2018-11-28","3.675296"]},{"value":["2018-11-29","3.671625"]},{"value":["2018-11-30","3.667624"]},{"value":["2018-12-03","3.664062"]},{"value":["2018-12-04","3.659692"]},{"value":["2018-12-05","3.655222"]},{"value":["2018-12-06","3.651412"]},{"value":["2018-12-07","3.648362"]},{"value":["2018-12-10","3.645781"]},{"value":["2018-12-11","3.643580"]},{"value":["2018-12-12","3.641147"]},{"value":["2018-12-13","3.638763"]},{"value":["2018-12-14","3.635851"]},{"value":["2018-12-17","3.632993"]},{"value":["2018-12-18","3.629988"]},{"value":["2018-12-19","3.627314"]},{"value":["2018-12-20","3.624968"]},{"value":["2018-12-21","3.622721"]},{"value":["2018-12-24","3.620811"]},{"value":["2018-12-25","3.619049"]},{"value":["2018-12-26","3.617244"]},{"value":["2018-12-27","3.615296"]},{"value":["2018-12-28","3.613343"]}],"yAxisIndex":0,"xAxisIndex":0,"name":"sma_200","type":"line","coordinateSystem":"cartesian2d","symbol":"none","lineStyle":{"width":1}},{"data":[{"value":["2013-12-31",null]},{"value":["2014-01-02",null]},{"value":["2014-01-03",null]},{"value":["2014-01-06",null]},{"value":["2014-01-07",null]},{"value":["2014-01-08",null]},{"value":["2014-01-09",null]},{"value":["2014-01-10",null]},{"value":["2014-01-13",null]},{"value":["2014-01-14",null]},{"value":["2014-01-15",null]},{"value":["2014-01-16",null]},{"value":["2014-01-17",null]},{"value":["2014-01-20",null]},{"value":["2014-01-21",null]},{"value":["2014-01-22",null]},{"value":["2014-01-23",null]},{"value":["2014-01-24",null]},{"value":["2014-01-27",null]},{"value":["2014-01-28",null]},{"value":["2014-01-29",null]},{"value":["2014-01-30",null]},{"value":["2014-02-07",null]},{"value":["2014-02-10",null]},{"value":["2014-02-11",null]},{"value":["2014-02-12",null]},{"value":["2014-02-13",null]},{"value":["2014-02-14",null]},{"value":["2014-02-17",null]},{"value":["2014-02-18",null]},{"value":["2014-02-19",null]},{"value":["2014-02-20",null]},{"value":["2014-02-21",null]},{"value":["2014-02-24",null]},{"value":["2014-02-25",null]},{"value":["2014-02-26",null]},{"value":["2014-02-27",null]},{"value":["2014-02-28",null]},{"value":["2014-03-03",null]},{"value":["2014-03-04",null]},{"value":["2014-03-05",null]},{"value":["2014-03-06",null]},{"value":["2014-03-07",null]},{"value":["2014-03-10",null]},{"value":["2014-03-11",null]},{"value":["2014-03-12",null]},{"value":["2014-03-13",null]},{"value":["2014-03-14",null]},{"value":["2014-03-17",null]},{"value":["2014-03-18","1.939991"]},{"value":["2014-03-19","1.937677"]},{"value":["2014-03-20","1.935363"]},{"value":["2014-03-21","1.934900"]},{"value":["2014-03-24","1.935054"]},{"value":["2014-03-25","1.935363"]},{"value":["2014-03-26","1.935826"]},{"value":["2014-03-27","1.936751"]},{"value":["2014-03-28","1.938140"]},{"value":["2014-03-31","1.939219"]},{"value":["2014-04-01","1.940454"]},{"value":["2014-04-02","1.941842"]},{"value":["2014-04-03","1.943076"]},{"value":["2014-04-04","1.944927"]},{"value":["2014-04-08","1.947550"]},{"value":["2014-04-09","1.950018"]},{"value":["2014-04-10","1.952486"]},{"value":["2014-04-11","1.955571"]},{"value":["2014-04-14","1.958348"]},{"value":["2014-04-15","1.960508"]},{"value":["2014-04-16","1.962667"]},{"value":["2014-04-17","1.964210"]},{"value":["2014-04-18","1.966061"]},{"value":["2014-04-21","1.967758"]},{"value":["2014-04-22","1.969609"]},{"value":["2014-04-23","1.970535"]},{"value":["2014-04-24","1.971460"]},{"value":["2014-04-25","1.972386"]},{"value":["2014-04-28","1.973620"]},{"value":["2014-04-29","1.974545"]},{"value":["2014-04-30","1.975471"]},{"value":["2014-05-05","1.975008"]},{"value":["2014-05-06","1.974700"]},{"value":["2014-05-07","1.975008"]},{"value":["2014-05-08","1.976242"]},{"value":["2014-05-09","1.977322"]},{"value":["2014-05-12","1.979327"]},{"value":["2014-05-13","1.981179"]},{"value":["2014-05-14","1.983338"]},{"value":["2014-05-15","1.985652"]},{"value":["2014-05-16","1.987966"]},{"value":["2014-05-19","1.990434"]},{"value":["2014-05-20","1.992748"]},{"value":["2014-05-21","1.995371"]},{"value":["2014-05-22","1.998610"]},{"value":["2014-05-23","2.002004"]},{"value":["2014-05-26","2.005706"]},{"value":["2014-05-27","2.008637"]},{"value":["2014-05-28","2.012031"]},{"value":["2014-05-29","2.015270"]},{"value":["2014-05-30","2.018356"]},{"value":["2014-06-03","2.022212"]},{"value":["2014-06-04","2.026223"]},{"value":["2014-06-05","2.029308"]},{"value":["2014-06-06","2.031776"]},{"value":["2014-06-09","2.034553"]},{"value":["2014-06-10","2.037638"]},{"value":["2014-06-11","2.040415"]},{"value":["2014-06-12","2.043192"]},{"value":["2014-06-13","2.046431"]},{"value":["2014-06-16","2.049825"]},{"value":["2014-06-17","2.052602"]},{"value":["2014-06-18","2.055378"]},{"value":["2014-06-19","2.057229"]},{"value":["2014-06-20","2.058926"]},{"value":["2014-06-23","2.060469"]},{"value":["2014-06-24","2.061703"]},{"value":["2014-06-25","2.063091"]},{"value":["2014-06-26","2.064634"]},{"value":["2014-06-27","2.066449"]},{"value":["2014-06-30","2.068276"]},{"value":["2014-07-01","2.070270"]},{"value":["2014-07-02","2.072596"]},{"value":["2014-07-03","2.075551"]},{"value":["2014-07-04","2.078043"]},{"value":["2014-07-07","2.080677"]},{"value":["2014-07-08","2.083466"]},{"value":["2014-07-09","2.085768"]},{"value":["2014-07-10","2.087583"]},{"value":["2014-07-11","2.089719"]},{"value":["2014-07-14","2.092175"]},{"value":["2014-07-15","2.094797"]},{"value":["2014-07-16","2.097574"]},{"value":["2014-07-17","2.100196"]},{"value":["2014-07-18","2.102664"]},{"value":["2014-07-21","2.105133"]},{"value":["2014-07-22","2.107138"]},{"value":["2014-07-23","2.108823"]},{"value":["2014-07-24","2.111007"]},{"value":["2014-07-25","2.113523"]},{"value":["2014-07-28","2.117535"]},{"value":["2014-07-29","2.121881"]},{"value":["2014-07-30","2.125561"]},{"value":["2014-07-31","2.129585"]},{"value":["2014-08-01","2.133456"]},{"value":["2014-08-04","2.137504"]},{"value":["2014-08-05","2.141232"]},{"value":["2014-08-06","2.144948"]},{"value":["2014-08-07","2.148011"]},{"value":["2014-08-08","2.151241"]},{"value":["2014-08-11","2.155111"]},{"value":["2014-08-12","2.158044"]},{"value":["2014-08-13","2.160976"]},{"value":["2014-08-14","2.163589"]},{"value":["2014-08-15","2.166830"]},{"value":["2014-08-18","2.169763"]},{"value":["2014-08-19","2.172221"]},{"value":["2014-08-20","2.174845"]},{"value":["2014-08-21","2.176994"]},{"value":["2014-08-22","2.178859"]},{"value":["2014-08-25","2.180404"]},{"value":["2014-08-26","2.182078"]},{"value":["2014-08-27","2.183895"]},{"value":["2014-08-28","2.185664"]},{"value":["2014-08-29","2.187802"]},{"value":["2014-09-01","2.189915"]},{"value":["2014-09-02","2.192219"]},{"value":["2014-09-03","2.194547"]},{"value":["2014-09-04","2.197041"]},{"value":["2014-09-05","2.200366"]},{"value":["2014-09-09","2.203359"]},{"value":["2014-09-10","2.205853"]},{"value":["2014-09-11","2.207848"]},{"value":["2014-09-12","2.210009"]},{"value":["2014-09-15","2.212005"]},{"value":["2014-09-16","2.214332"]},{"value":["2014-09-17","2.216327"]},{"value":["2014-09-18","2.218988"]},{"value":["2014-09-19","2.221981"]},{"value":["2014-09-22","2.223976"]},{"value":["2014-09-23","2.225971"]},{"value":["2014-09-24","2.228132"]},{"value":["2014-09-25","2.230294"]},{"value":["2014-09-26","2.232289"]},{"value":["2014-09-29","2.234118"]},{"value":["2014-09-30","2.236279"]},{"value":["2014-10-08","2.238275"]},{"value":["2014-10-09","2.240602"]},{"value":["2014-10-10","2.242265"]},{"value":["2014-10-13","2.243263"]},{"value":["2014-10-14","2.242598"]},{"value":["2014-10-15","2.241766"]},{"value":["2014-10-16","2.241600"]},{"value":["2014-10-17","2.240935"]},{"value":["2014-10-20","2.240436"]},{"value":["2014-10-21","2.239438"]},{"value":["2014-10-22","2.238773"]},{"value":["2014-10-23","2.238108"]},{"value":["2014-10-24","2.237776"]},{"value":["2014-10-27","2.236446"]},{"value":["2014-10-28","2.235282"]},{"value":["2014-10-29","2.235116"]},{"value":["2014-10-30","2.235282"]},{"value":["2014-10-31","2.237111"]},{"value":["2014-11-03","2.238441"]},{"value":["2014-11-04","2.239937"]},{"value":["2014-11-05","2.241101"]},{"value":["2014-11-06","2.241932"]},{"value":["2014-11-07","2.243263"]},{"value":["2014-11-10","2.245757"]},{"value":["2014-11-11","2.253239"]},{"value":["2014-11-12","2.259889"]},{"value":["2014-11-13","2.267039"]},{"value":["2014-11-14","2.274687"]},{"value":["2014-11-17","2.280340"]},{"value":["2014-11-18","2.284829"]},{"value":["2014-11-19","2.288653"]},{"value":["2014-11-20","2.292145"]},{"value":["2014-11-21","2.295969"]},{"value":["2014-11-24","2.299294"]},{"value":["2014-11-25","2.303285"]},{"value":["2014-11-26","2.308938"]},{"value":["2014-11-27","2.315256"]},{"value":["2014-11-28","2.325398"]},{"value":["2014-12-01","2.336538"]},{"value":["2014-12-02","2.350504"]},{"value":["2014-12-03","2.362309"]},{"value":["2014-12-04","2.375777"]},{"value":["2014-12-05","2.391905"]},{"value":["2014-12-08","2.408365"]},{"value":["2014-12-09","2.419172"]},{"value":["2014-12-10","2.430645"]},{"value":["2014-12-11","2.441286"]},{"value":["2014-12-12","2.452093"]},{"value":["2014-12-15","2.462235"]},{"value":["2014-12-16","2.473874"]},{"value":["2014-12-17","2.488006"]},{"value":["2014-12-18","2.500643"]},{"value":["2014-12-19","2.514775"]},{"value":["2014-12-22","2.535060"]},{"value":["2014-12-23","2.552684"]},{"value":["2014-12-24","2.567648"]},{"value":["2014-12-25","2.585106"]},{"value":["2014-12-26","2.604226"]},{"value":["2014-12-29","2.623680"]},{"value":["2014-12-30","2.644795"]},{"value":["2014-12-31","2.669236"]},{"value":["2015-01-05","2.698333"]},{"value":["2015-01-06","2.729924"]},{"value":["2015-01-07","2.762013"]},{"value":["2015-01-08","2.790112"]},{"value":["2015-01-09","2.819707"]},{"value":["2015-01-12","2.847806"]},{"value":["2015-01-13","2.875074"]},{"value":["2015-01-14","2.905168"]},{"value":["2015-01-15","2.939751"]},{"value":["2015-01-16","2.976663"]},{"value":["2015-01-19","3.005427"]},{"value":["2015-01-20","3.033359"]},{"value":["2015-01-21","3.067111"]},{"value":["2015-01-22","3.095210"]},{"value":["2015-01-23","3.124473"]},{"value":["2015-01-26","3.152406"]},{"value":["2015-01-27","3.178011"]},{"value":["2015-01-28","3.203450"]},{"value":["2015-01-29","3.227891"]},{"value":["2015-01-30","3.252498"]},{"value":["2015-02-02","3.272616"]},{"value":["2015-02-03","3.294730"]},{"value":["2015-02-04","3.314515"]},{"value":["2015-02-05","3.333802"]},{"value":["2015-02-06","3.349764"]},{"value":["2015-02-09","3.365559"]},{"value":["2015-02-10","3.378361"]},{"value":["2015-02-11","3.389834"]},{"value":["2015-02-12","3.398147"]},{"value":["2015-02-13","3.409786"]},{"value":["2015-02-16","3.419097"]},{"value":["2015-02-17","3.425914"]},{"value":["2015-02-25","3.432398"]},{"value":["2015-02-26","3.445866"]},{"value":["2015-02-27","3.457837"]},{"value":["2015-03-02","3.470639"]},{"value":["2015-03-03","3.481114"]},{"value":["2015-03-04","3.491422"]},{"value":["2015-03-05","3.498073"]},{"value":["2015-03-06","3.502895"]},{"value":["2015-03-09","3.512372"]},{"value":["2015-03-10","3.518856"]},{"value":["2015-03-11","3.520852"]},{"value":["2015-03-12","3.528500"]},{"value":["2015-03-13","3.538143"]},{"value":["2015-03-16","3.545791"]},{"value":["2015-03-17","3.552276"]},{"value":["2015-03-18","3.560257"]},{"value":["2015-03-19","3.565743"]},{"value":["2015-03-20","3.569568"]},{"value":["2015-03-23","3.569734"]},{"value":["2015-03-24","3.567739"]},{"value":["2015-03-25","3.563748"]},{"value":["2015-03-26","3.563416"]},{"value":["2015-03-27","3.560922"]},{"value":["2015-03-30","3.561919"]},{"value":["2015-03-31","3.560922"]},{"value":["2015-04-01","3.557929"]},{"value":["2015-04-02","3.549449"]},{"value":["2015-04-03","3.539307"]},{"value":["2015-04-07","3.539141"]},{"value":["2015-04-08","3.540471"]},{"value":["2015-04-09","3.533820"]},{"value":["2015-04-10","3.529165"]},{"value":["2015-04-13","3.526505"]},{"value":["2015-04-14","3.524011"]},{"value":["2015-04-15","3.525341"]},{"value":["2015-04-16","3.529497"]},{"value":["2015-04-17","3.538309"]},{"value":["2015-04-20","3.543963"]},{"value":["2015-04-21","3.554770"]},{"value":["2015-04-22","3.564413"]},{"value":["2015-04-23","3.575553"]},{"value":["2015-04-24","3.584365"]},{"value":["2015-04-27","3.597666"]},{"value":["2015-04-28","3.614792"]},{"value":["2015-04-29","3.629257"]},{"value":["2015-04-30","3.642060"]},{"value":["2015-05-04","3.654862"]},{"value":["2015-05-05","3.662843"]},{"value":["2015-05-06","3.669826"]},{"value":["2015-05-07","3.675313"]},{"value":["2015-05-08","3.682628"]},{"value":["2015-05-11","3.689612"]},{"value":["2015-05-12","3.697426"]},{"value":["2015-05-13","3.704908"]},{"value":["2015-05-14","3.714552"]},{"value":["2015-05-15","3.723364"]},{"value":["2015-05-18","3.732176"]},{"value":["2015-05-19","3.742651"]},{"value":["2015-05-20","3.749467"]},{"value":["2015-05-21","3.757947"]},{"value":["2015-05-22","3.767258"]},{"value":["2015-05-25","3.775405"]},{"value":["2015-05-26","3.784217"]},{"value":["2015-05-27","3.792032"]},{"value":["2015-05-28","3.794858"]},{"value":["2015-05-29","3.794858"]},{"value":["2015-06-01","3.798682"]},{"value":["2015-06-02","3.800345"]},{"value":["2015-06-03","3.801176"]},{"value":["2015-06-04","3.805665"]},{"value":["2015-06-05","3.812316"]},{"value":["2015-06-08","3.826615"]},{"value":["2015-06-09","3.842743"]},{"value":["2015-06-10","3.854215"]},{"value":["2015-06-11","3.864690"]},{"value":["2015-06-12","3.876162"]},{"value":["2015-06-15","3.886138"]},{"value":["2015-06-16","3.893786"]},{"value":["2015-06-17","3.901767"]},{"value":["2015-06-18","3.905093"]},{"value":["2015-06-19","3.905758"]},{"value":["2015-06-23","3.908418"]},{"value":["2015-06-24","3.910413"]},{"value":["2015-06-25","3.912242"]},{"value":["2015-06-26","3.909249"]},{"value":["2015-06-29","3.905259"]},{"value":["2015-06-30","3.904594"]},{"value":["2015-07-01","3.904760"]},{"value":["2015-07-02","3.908584"]},{"value":["2015-07-03","3.904857"]},{"value":["2015-07-06","3.909731"]},{"value":["2015-07-07","3.925401"]},{"value":["2015-07-08","3.942033"]},{"value":["2015-07-09","3.954682"]},{"value":["2015-07-10","3.967617"]},{"value":["2015-07-13","3.977184"]},{"value":["2015-07-14","3.987262"]},{"value":["2015-07-15","4.003750"]},{"value":["2015-07-16","4.019149"]},{"value":["2015-07-17","4.034842"]},{"value":["2015-07-20","4.049363"]},{"value":["2015-07-21","4.061339"]},{"value":["2015-07-22","4.072290"]},{"value":["2015-07-23","4.083920"]},{"value":["2015-07-24","4.094852"]},{"value":["2015-07-27","4.101045"]},{"value":["2015-07-28","4.109759"]},{"value":["2015-07-29","4.115620"]},{"value":["2015-07-30","4.119913"]},{"value":["2015-07-31","4.122995"]},{"value":["2015-08-03","4.123917"]},{"value":["2015-08-04","4.123694"]},{"value":["2015-08-05","4.121749"]},{"value":["2015-08-06","4.119785"]},{"value":["2015-08-07","4.122829"]},{"value":["2015-08-10","4.128432"]},{"value":["2015-08-11","4.130517"]},{"value":["2015-08-12","4.131725"]},{"value":["2015-08-13","4.133106"]},{"value":["2015-08-14","4.130317"]},{"value":["2015-08-17","4.127003"]},{"value":["2015-08-18","4.114665"]},{"value":["2015-08-19","4.099295"]},{"value":["2015-08-20","4.085906"]},{"value":["2015-08-21","4.074327"]},{"value":["2015-08-24","4.055203"]},{"value":["2015-08-25","4.033050"]},{"value":["2015-08-26","4.015995"]},{"value":["2015-08-27","4.000921"]},{"value":["2015-08-28","3.989504"]},{"value":["2015-08-31","3.980894"]},{"value":["2015-09-01","3.972408"]},{"value":["2015-09-02","3.965876"]},{"value":["2015-09-07","3.955503"]},{"value":["2015-09-08","3.948967"]},{"value":["2015-09-09","3.942278"]},{"value":["2015-09-10","3.930101"]},{"value":["2015-09-11","3.919043"]},{"value":["2015-09-14","3.903662"]},{"value":["2015-09-15","3.894157"]},{"value":["2015-09-16","3.877567"]},{"value":["2015-09-17","3.851644"]},{"value":["2015-09-18","3.821746"]},{"value":["2015-09-21","3.791502"]},{"value":["2015-09-22","3.762814"]},{"value":["2015-09-23","3.738446"]},{"value":["2015-09-24","3.713560"]},{"value":["2015-09-25","3.686946"]},{"value":["2015-09-28","3.662233"]},{"value":["2015-09-29","3.637174"]},{"value":["2015-09-30","3.612288"]},{"value":["2015-10-08","3.589821"]},{"value":["2015-10-09","3.568391"]},{"value":["2015-10-12","3.547826"]},{"value":["2015-10-13","3.527606"]},{"value":["2015-10-14","3.514126"]},{"value":["2015-10-15","3.500646"]},{"value":["2015-10-16","3.488375"]},{"value":["2015-10-19","3.477833"]},{"value":["2015-10-20","3.468501"]},{"value":["2015-10-21","3.461243"]},{"value":["2015-10-22","3.451910"]},{"value":["2015-10-23","3.444306"]},{"value":["2015-10-26","3.437048"]},{"value":["2015-10-27","3.428925"]},{"value":["2015-10-28","3.418902"]},{"value":["2015-10-29","3.409224"]},{"value":["2015-10-30","3.400583"]},{"value":["2015-11-02","3.391251"]},{"value":["2015-11-03","3.381573"]},{"value":["2015-11-04","3.374660"]},{"value":["2015-11-05","3.370166"]},{"value":["2015-11-06","3.366883"]},{"value":["2015-11-09","3.367056"]},{"value":["2015-11-10","3.367056"]},{"value":["2015-11-11","3.372240"]},{"value":["2015-11-12","3.381918"]},{"value":["2015-11-13","3.388140"]},{"value":["2015-11-16","3.390905"]},{"value":["2015-11-17","3.393843"]},{"value":["2015-11-18","3.397472"]},{"value":["2015-11-19","3.397991"]},{"value":["2015-11-20","3.394707"]},{"value":["2015-11-23","3.395744"]},{"value":["2015-11-24","3.396781"]},{"value":["2015-11-25","3.398163"]},{"value":["2015-11-26","3.399373"]},{"value":["2015-11-27","3.399719"]},{"value":["2015-11-30","3.402830"]},{"value":["2015-12-01","3.404731"]},{"value":["2015-12-02","3.408878"]},{"value":["2015-12-03","3.416655"]},{"value":["2015-12-04","3.422013"]},{"value":["2015-12-07","3.426679"]},{"value":["2015-12-08","3.429962"]},{"value":["2015-12-09","3.435320"]},{"value":["2015-12-10","3.440159"]},{"value":["2015-12-11","3.443788"]},{"value":["2015-12-14","3.448972"]},{"value":["2015-12-15","3.454503"]},{"value":["2015-12-16","3.460033"]},{"value":["2015-12-17","3.465045"]},{"value":["2015-12-18","3.469884"]},{"value":["2015-12-21","3.474204"]},{"value":["2015-12-22","3.478525"]},{"value":["2015-12-23","3.482500"]},{"value":["2015-12-24","3.485438"]},{"value":["2015-12-25","3.487857"]},{"value":["2015-12-28","3.489758"]},{"value":["2015-12-29","3.492005"]},{"value":["2015-12-30","3.492005"]},{"value":["2015-12-31","3.493214"]},{"value":["2016-01-04","3.491832"]},{"value":["2016-01-05","3.490449"]},{"value":["2016-01-06","3.489931"]},{"value":["2016-01-07","3.487857"]},{"value":["2016-01-08","3.486302"]},{"value":["2016-01-11","3.482327"]},{"value":["2016-01-12","3.478525"]},{"value":["2016-01-13","3.474550"]},{"value":["2016-01-14","3.468674"]},{"value":["2016-01-15","3.459687"]},{"value":["2016-01-18","3.450182"]},{"value":["2016-01-19","3.438776"]},{"value":["2016-01-20","3.427024"]},{"value":["2016-01-21","3.415445"]},{"value":["2016-01-22","3.405076"]},{"value":["2016-01-25","3.395053"]},{"value":["2016-01-26","3.381054"]},{"value":["2016-01-27","3.366537"]},{"value":["2016-01-28","3.350811"]},{"value":["2016-01-29","3.335948"]},{"value":["2016-02-01","3.320740"]},{"value":["2016-02-02","3.306569"]},{"value":["2016-02-03","3.291533"]},{"value":["2016-02-04","3.276152"]},{"value":["2016-02-05","3.260771"]},{"value":["2016-02-15","3.246600"]},{"value":["2016-02-16","3.231392"]},{"value":["2016-02-17","3.217912"]},{"value":["2016-02-18","3.201494"]},{"value":["2016-02-19","3.182138"]},{"value":["2016-02-22","3.166930"]},{"value":["2016-02-23","3.151722"]},{"value":["2016-02-24","3.137551"]},{"value":["2016-02-25","3.120441"]},{"value":["2016-02-26","3.104542"]},{"value":["2016-02-29","3.089506"]},{"value":["2016-03-01","3.073953"]},{"value":["2016-03-02","3.060473"]},{"value":["2016-03-03","3.047338"]},{"value":["2016-03-04","3.035759"]},{"value":["2016-03-07","3.023835"]},{"value":["2016-03-08","3.011392"]},{"value":["2016-03-09","2.999986"]},{"value":["2016-03-10","2.987370"]},{"value":["2016-03-11","2.975964"]},{"value":["2016-03-14","2.964558"]},{"value":["2016-03-15","2.953497"]},{"value":["2016-03-16","2.943128"]},{"value":["2016-03-17","2.932759"]},{"value":["2016-03-18","2.922735"]},{"value":["2016-03-21","2.915131"]},{"value":["2016-03-22","2.907181"]},{"value":["2016-03-23","2.898540"]},{"value":["2016-03-24","2.890936"]},{"value":["2016-03-25","2.883332"]},{"value":["2016-03-28","2.878147"]},{"value":["2016-03-29","2.873136"]},{"value":["2016-03-30","2.870025"]},{"value":["2016-03-31","2.866741"]},{"value":["2016-04-01","2.865532"]},{"value":["2016-04-05","2.865013"]},{"value":["2016-04-06","2.863112"]},{"value":["2016-04-07","2.861557"]},{"value":["2016-04-08","2.860347"]},{"value":["2016-04-11","2.859310"]},{"value":["2016-04-12","2.858446"]},{"value":["2016-04-13","2.861557"]},{"value":["2016-04-14","2.865013"]},{"value":["2016-04-15","2.869506"]},{"value":["2016-04-18","2.872444"]},{"value":["2016-04-19","2.876074"]},{"value":["2016-04-20","2.878839"]},{"value":["2016-04-21","2.881777"]},{"value":["2016-04-22","2.884542"]},{"value":["2016-04-25","2.887480"]},{"value":["2016-04-26","2.891455"]},{"value":["2016-04-27","2.893528"]},{"value":["2016-04-28","2.895775"]},{"value":["2016-04-29","2.897849"]},{"value":["2016-05-03","2.900441"]},{"value":["2016-05-04","2.901824"]},{"value":["2016-05-05","2.904070"]},{"value":["2016-05-06","2.905626"]},{"value":["2016-05-09","2.908391"]},{"value":["2016-05-10","2.910465"]},{"value":["2016-05-11","2.912193"]},{"value":["2016-05-12","2.913057"]},{"value":["2016-05-13","2.912884"]},{"value":["2016-05-16","2.912366"]},{"value":["2016-05-17","2.909601"]},{"value":["2016-05-18","2.907008"]},{"value":["2016-05-19","2.904070"]},{"value":["2016-05-20","2.900787"]},{"value":["2016-05-23","2.899231"]},{"value":["2016-05-24","2.896466"]},{"value":["2016-05-25","2.893528"]},{"value":["2016-05-26","2.890936"]},{"value":["2016-05-27","2.887480"]},{"value":["2016-05-30","2.884715"]},{"value":["2016-05-31","2.883505"]},{"value":["2016-06-01","2.881949"]},{"value":["2016-06-02","2.880740"]},{"value":["2016-06-03","2.879876"]},{"value":["2016-06-06","2.879703"]},{"value":["2016-06-07","2.879530"]},{"value":["2016-06-08","2.879703"]},{"value":["2016-06-13","2.879012"]},{"value":["2016-06-14","2.877629"]},{"value":["2016-06-15","2.876246"]},{"value":["2016-06-16","2.874345"]},{"value":["2016-06-17","2.872272"]},{"value":["2016-06-20","2.870543"]},{"value":["2016-06-21","2.869852"]},{"value":["2016-06-22","2.869852"]},{"value":["2016-06-23","2.869506"]},{"value":["2016-06-24","2.868433"]},{"value":["2016-06-27","2.867205"]},{"value":["2016-06-28","2.865795"]},{"value":["2016-06-29","2.864748"]},{"value":["2016-06-30","2.864585"]},{"value":["2016-07-01","2.864776"]},{"value":["2016-07-04","2.865859"]},{"value":["2016-07-05","2.867287"]},{"value":["2016-07-06","2.868543"]},{"value":["2016-07-07","2.869981"]},{"value":["2016-07-08","2.870699"]},{"value":["2016-07-11","2.871955"]},{"value":["2016-07-12","2.873775"]},{"value":["2016-07-13","2.876669"]},{"value":["2016-07-14","2.878853"]},{"value":["2016-07-15","2.881219"]},{"value":["2016-07-18","2.883767"]},{"value":["2016-07-19","2.886642"]},{"value":["2016-07-20","2.890017"]},{"value":["2016-07-21","2.893566"]},{"value":["2016-07-22","2.897114"]},{"value":["2016-07-25","2.901554"]},{"value":["2016-07-26","2.906540"]},{"value":["2016-07-27","2.911161"]},{"value":["2016-07-28","2.915774"]},{"value":["2016-07-29","2.920569"]},{"value":["2016-08-01","2.925718"]},{"value":["2016-08-02","2.930877"]},{"value":["2016-08-03","2.935326"]},{"value":["2016-08-04","2.939757"]},{"value":["2016-08-05","2.944542"]},{"value":["2016-08-08","2.949510"]},{"value":["2016-08-09","2.954668"]},{"value":["2016-08-10","2.959127"]},{"value":["2016-08-11","2.962939"]},{"value":["2016-08-12","2.968918"]},{"value":["2016-08-15","2.976161"]},{"value":["2016-08-16","2.981230"]},{"value":["2016-08-17","2.986116"]},{"value":["2016-08-18","2.990466"]},{"value":["2016-08-19","2.994825"]},{"value":["2016-08-22","3.000585"]},{"value":["2016-08-23","3.006181"]},{"value":["2016-08-24","3.011777"]},{"value":["2016-08-25","3.017719"]},{"value":["2016-08-26","3.023306"]},{"value":["2016-08-29","3.028356"]},{"value":["2016-08-30","3.033424"]},{"value":["2016-08-31","3.038511"]},{"value":["2016-09-01","3.043607"]},{"value":["2016-09-02","3.049249"]},{"value":["2016-09-05","3.054345"]},{"value":["2016-09-06","3.059623"]},{"value":["2016-09-07","3.064355"]},{"value":["2016-09-08","3.068723"]},{"value":["2016-09-09","3.072727"]},{"value":["2016-09-12","3.075093"]},{"value":["2016-09-13","3.077459"]},{"value":["2016-09-14","3.079097"]},{"value":["2016-09-19","3.080735"]},{"value":["2016-09-20","3.082737"]},{"value":["2016-09-21","3.085103"]},{"value":["2016-09-22","3.086741"]},{"value":["2016-09-23","3.087833"]},{"value":["2016-09-26","3.088743"]},{"value":["2016-09-27","3.089653"]},{"value":["2016-09-28","3.090017"]},{"value":["2016-09-29","3.090927"]},{"value":["2016-09-30","3.092201"]},{"value":["2016-10-10","3.094021"]},{"value":["2016-10-11","3.095841"]},{"value":["2016-10-12","3.096933"]},{"value":["2016-10-13","3.097297"]},{"value":["2016-10-14","3.098025"]},{"value":["2016-10-17","3.098571"]},{"value":["2016-10-18","3.099481"]},{"value":["2016-10-19","3.100027"]},{"value":["2016-10-20","3.100573"]},{"value":["2016-10-21","3.101665"]},{"value":["2016-10-24","3.103667"]},{"value":["2016-10-25","3.105670"]},{"value":["2016-10-26","3.106944"]},{"value":["2016-10-27","3.107490"]},{"value":["2016-10-28","3.108582"]},{"value":["2016-10-31","3.108400"]},{"value":["2016-11-01","3.106580"]},{"value":["2016-11-02","3.103303"]},{"value":["2016-11-03","3.102029"]},{"value":["2016-11-04","3.100755"]},{"value":["2016-11-07","3.099663"]},{"value":["2016-11-08","3.098753"]},{"value":["2016-11-09","3.097115"]},{"value":["2016-11-10","3.095659"]},{"value":["2016-11-11","3.094203"]},{"value":["2016-11-14","3.093111"]},{"value":["2016-11-15","3.092019"]},{"value":["2016-11-16","3.091291"]},{"value":["2016-11-17","3.090199"]},{"value":["2016-11-18","3.088561"]},{"value":["2016-11-21","3.087287"]},{"value":["2016-11-22","3.086559"]},{"value":["2016-11-23","3.086377"]},{"value":["2016-11-24","3.086195"]},{"value":["2016-11-25","3.086923"]},{"value":["2016-11-28","3.088561"]},{"value":["2016-11-29","3.090745"]},{"value":["2016-11-30","3.093111"]},{"value":["2016-12-01","3.095659"]},{"value":["2016-12-02","3.098753"]},{"value":["2016-12-05","3.101665"]},{"value":["2016-12-06","3.104577"]},{"value":["2016-12-07","3.106762"]},{"value":["2016-12-08","3.109128"]},{"value":["2016-12-09","3.112586"]},{"value":["2016-12-12","3.117500"]},{"value":["2016-12-13","3.121140"]},{"value":["2016-12-14","3.125326"]},{"value":["2016-12-15","3.127328"]},{"value":["2016-12-16","3.128966"]},{"value":["2016-12-19","3.130058"]},{"value":["2016-12-20","3.130422"]},{"value":["2016-12-21","3.131696"]},{"value":["2016-12-22","3.132970"]},{"value":["2016-12-23","3.133880"]},{"value":["2016-12-26","3.135336"]},{"value":["2016-12-27","3.135700"]},{"value":["2016-12-28","3.136428"]},{"value":["2016-12-29","3.136974"]},{"value":["2016-12-30","3.137702"]},{"value":["2017-01-03","3.138430"]},{"value":["2017-01-04","3.138976"]},{"value":["2017-01-05","3.139886"]},{"value":["2017-01-06","3.140796"]},{"value":["2017-01-09","3.141342"]},{"value":["2017-01-10","3.142070"]},{"value":["2017-01-11","3.142616"]},{"value":["2017-01-12","3.143526"]},{"value":["2017-01-13","3.144982"]},{"value":["2017-01-16","3.147348"]},{"value":["2017-01-17","3.149896"]},{"value":["2017-01-18","3.152262"]},{"value":["2017-01-19","3.155174"]},{"value":["2017-01-20","3.157722"]},{"value":["2017-01-23","3.160088"]},{"value":["2017-01-24","3.162636"]},{"value":["2017-01-25","3.165366"]},{"value":["2017-01-26","3.168824"]},{"value":["2017-02-03","3.171736"]},{"value":["2017-02-06","3.174830"]},{"value":["2017-02-07","3.177196"]},{"value":["2017-02-08","3.179380"]},{"value":["2017-02-09","3.181564"]},{"value":["2017-02-10","3.184294"]},{"value":["2017-02-13","3.186842"]},{"value":["2017-02-14","3.188298"]},{"value":["2017-02-15","3.190300"]},{"value":["2017-02-16","3.193576"]},{"value":["2017-02-17","3.195578"]},{"value":["2017-02-20","3.198854"]},{"value":["2017-02-21","3.202494"]},{"value":["2017-02-22","3.206498"]},{"value":["2017-02-23","3.210684"]},{"value":["2017-02-24","3.214324"]},{"value":["2017-02-27","3.216144"]},{"value":["2017-02-28","3.216872"]},{"value":["2017-03-01","3.218510"]},{"value":["2017-03-02","3.219966"]},{"value":["2017-03-03","3.223060"]},{"value":["2017-03-06","3.226154"]},{"value":["2017-03-07","3.229430"]},{"value":["2017-03-08","3.233434"]},{"value":["2017-03-09","3.236710"]},{"value":["2017-03-10","3.239986"]},{"value":["2017-03-13","3.244354"]},{"value":["2017-03-14","3.248358"]},{"value":["2017-03-15","3.252909"]},{"value":["2017-03-16","3.257641"]},{"value":["2017-03-17","3.261827"]},{"value":["2017-03-20","3.266013"]},{"value":["2017-03-21","3.269289"]},{"value":["2017-03-22","3.271655"]},{"value":["2017-03-23","3.274203"]},{"value":["2017-03-24","3.277661"]},{"value":["2017-03-27","3.281483"]},{"value":["2017-03-28","3.285305"]},{"value":["2017-03-29","3.289309"]},{"value":["2017-03-30","3.293495"]},{"value":["2017-03-31","3.297863"]},{"value":["2017-04-05","3.301139"]},{"value":["2017-04-06","3.304051"]},{"value":["2017-04-07","3.306417"]},{"value":["2017-04-10","3.308419"]},{"value":["2017-04-11","3.310239"]},{"value":["2017-04-12","3.312241"]},{"value":["2017-04-13","3.313879"]},{"value":["2017-04-14","3.315153"]},{"value":["2017-04-17","3.315517"]},{"value":["2017-04-18","3.315881"]},{"value":["2017-04-19","3.315517"]},{"value":["2017-04-20","3.314971"]},{"value":["2017-04-21","3.314789"]},{"value":["2017-04-24","3.314789"]},{"value":["2017-04-25","3.314243"]},{"value":["2017-04-26","3.313333"]},{"value":["2017-04-27","3.312787"]},{"value":["2017-04-28","3.311331"]},{"value":["2017-05-02","3.308965"]},{"value":["2017-05-03","3.307145"]},{"value":["2017-05-04","3.304233"]},{"value":["2017-05-05","3.301139"]},{"value":["2017-05-08","3.298409"]},{"value":["2017-05-09","3.295497"]},{"value":["2017-05-10","3.292949"]},{"value":["2017-05-11","3.291675"]},{"value":["2017-05-12","3.291493"]},{"value":["2017-05-15","3.291311"]},{"value":["2017-05-16","3.290583"]},{"value":["2017-05-17","3.290037"]},{"value":["2017-05-18","3.289673"]},{"value":["2017-05-19","3.289127"]},{"value":["2017-05-22","3.288945"]},{"value":["2017-05-23","3.289673"]},{"value":["2017-05-24","3.290219"]},{"value":["2017-05-25","3.290765"]},{"value":["2017-05-26","3.291493"]},{"value":["2017-05-31","3.292767"]},{"value":["2017-06-01","3.293859"]},{"value":["2017-06-02","3.295133"]},{"value":["2017-06-05","3.294405"]},{"value":["2017-06-06","3.294223"]},{"value":["2017-06-07","3.295497"]},{"value":["2017-06-08","3.296771"]},{"value":["2017-06-09","3.297317"]},{"value":["2017-06-12","3.297135"]},{"value":["2017-06-13","3.297135"]},{"value":["2017-06-14","3.296771"]},{"value":["2017-06-15","3.295861"]},{"value":["2017-06-16","3.294405"]},{"value":["2017-06-19","3.293131"]},{"value":["2017-06-20","3.292403"]},{"value":["2017-06-21","3.292039"]},{"value":["2017-06-22","3.292403"]},{"value":["2017-06-23","3.293313"]},{"value":["2017-06-26","3.294041"]},{"value":["2017-06-27","3.294587"]},{"value":["2017-06-28","3.295679"]},{"value":["2017-06-29","3.297135"]},{"value":["2017-06-30","3.299501"]},{"value":["2017-07-03","3.302413"]},{"value":["2017-07-04","3.305507"]},{"value":["2017-07-05","3.308055"]},{"value":["2017-07-06","3.309875"]},{"value":["2017-07-07","3.312059"]},{"value":["2017-07-10","3.314061"]},{"value":["2017-07-11","3.316427"]},{"value":["2017-07-12","3.319339"]},{"value":["2017-07-13","3.323889"]},{"value":["2017-07-14","3.329747"]},{"value":["2017-07-17","3.337701"]},{"value":["2017-07-18","3.344901"]},{"value":["2017-07-19","3.351555"]},{"value":["2017-07-20","3.358780"]},{"value":["2017-07-21","3.365633"]},{"value":["2017-07-24","3.372139"]},{"value":["2017-07-25","3.378688"]},{"value":["2017-07-26","3.386371"]},{"value":["2017-07-27","3.394029"]},{"value":["2017-07-28","3.402059"]},{"value":["2017-07-31","3.411033"]},{"value":["2017-08-01","3.420769"]},{"value":["2017-08-02","3.431855"]},{"value":["2017-08-03","3.442222"]},{"value":["2017-08-04","3.452382"]},{"value":["2017-08-07","3.461432"]},{"value":["2017-08-08","3.471626"]},{"value":["2017-08-09","3.479559"]},{"value":["2017-08-10","3.487310"]},{"value":["2017-08-11","3.494854"]},{"value":["2017-08-14","3.503273"]},{"value":["2017-08-15","3.513987"]},{"value":["2017-08-16","3.523965"]},{"value":["2017-08-17","3.533752"]},{"value":["2017-08-18","3.543340"]},{"value":["2017-08-21","3.553102"]},{"value":["2017-08-22","3.563054"]},{"value":["2017-08-23","3.574331"]},{"value":["2017-08-24","3.585972"]},{"value":["2017-08-25","3.600288"]},{"value":["2017-08-28","3.615168"]},{"value":["2017-08-29","3.631580"]},{"value":["2017-08-30","3.647040"]},{"value":["2017-08-31","3.661191"]},{"value":["2017-09-01","3.674407"]},{"value":["2017-09-04","3.687996"]},{"value":["2017-09-05","3.702728"]},{"value":["2017-09-06","3.717087"]},{"value":["2017-09-07","3.730502"]},{"value":["2017-09-08","3.743172"]},{"value":["2017-09-11","3.756024"]},{"value":["2017-09-12","3.769058"]},{"value":["2017-09-13","3.781711"]},{"value":["2017-09-14","3.793403"]},{"value":["2017-09-15","3.803588"]},{"value":["2017-09-18","3.813773"]},{"value":["2017-09-19","3.823793"]},{"value":["2017-09-20","3.834220"]},{"value":["2017-09-21","3.844126"]},{"value":["2017-09-22","3.853842"]},{"value":["2017-09-25","3.862224"]},{"value":["2017-09-26","3.870606"]},{"value":["2017-09-27","3.878035"]},{"value":["2017-09-28","3.884703"]},{"value":["2017-09-29","3.891180"]},{"value":["2017-10-09","3.897276"]},{"value":["2017-10-10","3.902801"]},{"value":["2017-10-11","3.907944"]},{"value":["2017-10-12","3.913469"]},{"value":["2017-10-13","3.918422"]},{"value":["2017-10-16","3.922803"]},{"value":["2017-10-17","3.926232"]},{"value":["2017-10-18","3.928518"]},{"value":["2017-10-19","3.930995"]},{"value":["2017-10-20","3.933662"]},{"value":["2017-10-23","3.935567"]},{"value":["2017-10-24","3.936519"]},{"value":["2017-10-25","3.938234"]},{"value":["2017-10-26","3.938615"]},{"value":["2017-10-27","3.940139"]},{"value":["2017-10-30","3.942806"]},{"value":["2017-10-31","3.940901"]},{"value":["2017-11-01","3.938615"]},{"value":["2017-11-02","3.936138"]},{"value":["2017-11-03","3.933852"]},{"value":["2017-11-06","3.931566"]},{"value":["2017-11-07","3.929852"]},{"value":["2017-11-08","3.927756"]},{"value":["2017-11-09","3.925089"]},{"value":["2017-11-10","3.918612"]},{"value":["2017-11-13","3.912135"]},{"value":["2017-11-14","3.903753"]},{"value":["2017-11-15","3.896133"]},{"value":["2017-11-16","3.888703"]},{"value":["2017-11-17","3.883369"]},{"value":["2017-11-20","3.878226"]},{"value":["2017-11-21","3.871558"]},{"value":["2017-11-22","3.867177"]},{"value":["2017-11-23","3.862986"]},{"value":["2017-11-24","3.859176"]},{"value":["2017-11-27","3.854413"]},{"value":["2017-11-28","3.849079"]},{"value":["2017-11-29","3.844317"]},{"value":["2017-11-30","3.840697"]},{"value":["2017-12-01","3.838030"]},{"value":["2017-12-04","3.835363"]},{"value":["2017-12-05","3.833458"]},{"value":["2017-12-06","3.830220"]},{"value":["2017-12-07","3.826028"]},{"value":["2017-12-08","3.820885"]},{"value":["2017-12-11","3.815170"]},{"value":["2017-12-12","3.809074"]},{"value":["2017-12-13","3.804311"]},{"value":["2017-12-14","3.799549"]},{"value":["2017-12-15","3.794596"]},{"value":["2017-12-18","3.789643"]},{"value":["2017-12-19","3.784880"]},{"value":["2017-12-20","3.779737"]},{"value":["2017-12-21","3.774784"]},{"value":["2017-12-22","3.770212"]},{"value":["2017-12-25","3.765830"]},{"value":["2017-12-26","3.762211"]},{"value":["2017-12-27","3.757448"]},{"value":["2017-12-28","3.752495"]},{"value":["2017-12-29","3.748685"]},{"value":["2018-01-02","3.746208"]},{"value":["2018-01-03","3.743922"]},{"value":["2018-01-04","3.742589"]},{"value":["2018-01-05","3.742208"]},{"value":["2018-01-08","3.741827"]},{"value":["2018-01-09","3.740112"]},{"value":["2018-01-10","3.741827"]},{"value":["2018-01-11","3.745256"]},{"value":["2018-01-12","3.749066"]},{"value":["2018-01-15","3.754781"]},{"value":["2018-01-16","3.761449"]},{"value":["2018-01-17","3.767735"]},{"value":["2018-01-18","3.775546"]},{"value":["2018-01-19","3.784690"]},{"value":["2018-01-22","3.794786"]},{"value":["2018-01-23","3.809836"]},{"value":["2018-01-24","3.825266"]},{"value":["2018-01-25","3.839745"]},{"value":["2018-01-26","3.855175"]},{"value":["2018-01-29","3.868129"]},{"value":["2018-01-30","3.878416"]},{"value":["2018-01-31","3.890227"]},{"value":["2018-02-01","3.901658"]},{"value":["2018-02-02","3.912707"]},{"value":["2018-02-05","3.927185"]},{"value":["2018-02-06","3.942615"]},{"value":["2018-02-07","3.955379"]},{"value":["2018-02-08","3.964713"]},{"value":["2018-02-09","3.971572"]},{"value":["2018-02-12","3.977096"]},{"value":["2018-02-13","3.983954"]},{"value":["2018-02-14","3.989669"]},{"value":["2018-02-22","3.997480"]},{"value":["2018-02-23","4.006052"]},{"value":["2018-02-26","4.015196"]},{"value":["2018-02-27","4.022626"]},{"value":["2018-02-28","4.029293"]},{"value":["2018-03-01","4.035389"]},{"value":["2018-03-02","4.040723"]},{"value":["2018-03-05","4.046439"]},{"value":["2018-03-06","4.052725"]},{"value":["2018-03-07","4.059012"]},{"value":["2018-03-08","4.064727"]},{"value":["2018-03-09","4.070061"]},{"value":["2018-03-12","4.075204"]},{"value":["2018-03-13","4.080348"]},{"value":["2018-03-14","4.084539"]},{"value":["2018-03-15","4.089301"]},{"value":["2018-03-16","4.093111"]},{"value":["2018-03-19","4.096159"]},{"value":["2018-03-20","4.099779"]},{"value":["2018-03-21","4.103017"]},{"value":["2018-03-22","4.106256"]},{"value":["2018-03-23","4.108923"]},{"value":["2018-03-26","4.108542"]},{"value":["2018-03-27","4.108732"]},{"value":["2018-03-28","4.107399"]},{"value":["2018-03-29","4.105494"]},{"value":["2018-03-30","4.102446"]},{"value":["2018-04-02","4.096731"]},{"value":["2018-04-03","4.090254"]},{"value":["2018-04-04","4.082824"]},{"value":["2018-04-09","4.073109"]},{"value":["2018-04-10","4.064155"]},{"value":["2018-04-11","4.055964"]},{"value":["2018-04-12","4.041295"]},{"value":["2018-04-13","4.025674"]},{"value":["2018-04-16","4.009291"]},{"value":["2018-04-17","3.992336"]},{"value":["2018-04-18","3.976715"]},{"value":["2018-04-19","3.963570"]},{"value":["2018-04-20","3.949092"]},{"value":["2018-04-23","3.932900"]},{"value":["2018-04-24","3.918803"]},{"value":["2018-04-25","3.900515"]},{"value":["2018-04-26","3.882036"]},{"value":["2018-04-27","3.867367"]},{"value":["2018-05-02","3.855747"]},{"value":["2018-05-03","3.846793"]},{"value":["2018-05-04","3.838983"]},{"value":["2018-05-07","3.830410"]},{"value":["2018-05-08","3.822599"]},{"value":["2018-05-09","3.813265"]},{"value":["2018-05-10","3.803549"]},{"value":["2018-05-11","3.794024"]},{"value":["2018-05-14","3.786023"]},{"value":["2018-05-15","3.779546"]},{"value":["2018-05-16","3.772498"]},{"value":["2018-05-17","3.766402"]},{"value":["2018-05-18","3.761449"]},{"value":["2018-05-21","3.756115"]},{"value":["2018-05-22","3.749637"]},{"value":["2018-05-23","3.742779"]},{"value":["2018-05-24","3.735731"]},{"value":["2018-05-25","3.729063"]},{"value":["2018-05-28","3.722015"]},{"value":["2018-05-29","3.715538"]},{"value":["2018-05-30","3.707727"]},{"value":["2018-05-31","3.700869"]},{"value":["2018-06-01","3.693821"]},{"value":["2018-06-04","3.686582"]},{"value":["2018-06-05","3.679342"]},{"value":["2018-06-06","3.671722"]},{"value":["2018-06-07","3.665245"]},{"value":["2018-06-08","3.660483"]},{"value":["2018-06-11","3.656101"]},{"value":["2018-06-12","3.651910"]},{"value":["2018-06-13","3.646957"]},{"value":["2018-06-14","3.643147"]},{"value":["2018-06-15","3.640671"]},{"value":["2018-06-19","3.638385"]},{"value":["2018-06-20","3.636099"]},{"value":["2018-06-21","3.634003"]},{"value":["2018-06-22","3.630384"]},{"value":["2018-06-25","3.625240"]},{"value":["2018-06-26","3.620287"]},{"value":["2018-06-27","3.615525"]},{"value":["2018-06-28","3.611524"]},{"value":["2018-06-29","3.608666"]},{"value":["2018-07-02","3.602761"]},{"value":["2018-07-03","3.597427"]},{"value":["2018-07-04","3.592664"]},{"value":["2018-07-05","3.588854"]},{"value":["2018-07-06","3.584282"]},{"value":["2018-07-09","3.583330"]},{"value":["2018-07-10","3.582187"]},{"value":["2018-07-11","3.580282"]},{"value":["2018-07-12","3.579901"]},{"value":["2018-07-13","3.578329"]},{"value":["2018-07-16","3.575939"]},{"value":["2018-07-17","3.572577"]},{"value":["2018-07-18","3.568053"]},{"value":["2018-07-19","3.563138"]},{"value":["2018-07-20","3.560224"]},{"value":["2018-07-23","3.557728"]},{"value":["2018-07-24","3.555433"]},{"value":["2018-07-25","3.553128"]},{"value":["2018-07-26","3.551185"]},{"value":["2018-07-27","3.549042"]},{"value":["2018-07-30","3.546746"]},{"value":["2018-07-31","3.544660"]},{"value":["2018-08-01","3.541946"]},{"value":["2018-08-02","3.538593"]},{"value":["2018-08-03","3.535812"]},{"value":["2018-08-06","3.533241"]},{"value":["2018-08-07","3.531669"]},{"value":["2018-08-08","3.529307"]},{"value":["2018-08-09","3.528879"]},{"value":["2018-08-10","3.527860"]},{"value":["2018-08-13","3.526231"]},{"value":["2018-08-14","3.524231"]},{"value":["2018-08-15","3.521031"]},{"value":["2018-08-16","3.518612"]},{"value":["2018-08-17","3.515603"]},{"value":["2018-08-20","3.514355"]},{"value":["2018-08-21","3.513508"]},{"value":["2018-08-22","3.512260"]},{"value":["2018-08-23","3.511203"]},{"value":["2018-08-24","3.510746"]},{"value":["2018-08-27","3.510308"]},{"value":["2018-08-28","3.509880"]},{"value":["2018-08-29","3.510023"]},{"value":["2018-08-30","3.509375"]},{"value":["2018-08-31","3.509128"]},{"value":["2018-09-03","3.509642"]},{"value":["2018-09-04","3.511709"]},{"value":["2018-09-05","3.513348"]},{"value":["2018-09-06","3.515529"]},{"value":["2018-09-07","3.516958"]},{"value":["2018-09-10","3.521254"]},{"value":["2018-09-11","3.524588"]},{"value":["2018-09-12","3.527141"]},{"value":["2018-09-13","3.529742"]},{"value":["2018-09-14","3.531961"]},{"value":["2018-09-17","3.530923"]},{"value":["2018-09-18","3.530866"]},{"value":["2018-09-19","3.531200"]},{"value":["2018-09-20","3.530400"]},{"value":["2018-09-21","3.532800"]},{"value":["2018-09-25","3.535800"]},{"value":["2018-09-26","3.539800"]},{"value":["2018-09-27","3.544200"]},{"value":["2018-09-28","3.549600"]},{"value":["2018-10-08","3.551000"]},{"value":["2018-10-09","3.551000"]},{"value":["2018-10-10","3.551000"]},{"value":["2018-10-11","3.549200"]},{"value":["2018-10-12","3.548800"]},{"value":["2018-10-15","3.547600"]},{"value":["2018-10-16","3.545800"]},{"value":["2018-10-17","3.544200"]},{"value":["2018-10-18","3.543400"]},{"value":["2018-10-19","3.545800"]},{"value":["2018-10-22","3.550000"]},{"value":["2018-10-23","3.552200"]},{"value":["2018-10-24","3.554600"]},{"value":["2018-10-25","3.558200"]},{"value":["2018-10-26","3.561200"]},{"value":["2018-10-29","3.564400"]},{"value":["2018-10-30","3.569200"]},{"value":["2018-10-31","3.573400"]},{"value":["2018-11-01","3.577800"]},{"value":["2018-11-02","3.582600"]},{"value":["2018-11-05","3.587200"]},{"value":["2018-11-06","3.590400"]},{"value":["2018-11-07","3.593200"]},{"value":["2018-11-08","3.596600"]},{"value":["2018-11-09","3.598400"]},{"value":["2018-11-12","3.600000"]},{"value":["2018-11-13","3.601400"]},{"value":["2018-11-14","3.601800"]},{"value":["2018-11-15","3.602600"]},{"value":["2018-11-16","3.604000"]},{"value":["2018-11-19","3.605800"]},{"value":["2018-11-20","3.606800"]},{"value":["2018-11-21","3.607400"]},{"value":["2018-11-22","3.609000"]},{"value":["2018-11-23","3.610800"]},{"value":["2018-11-26","3.612600"]},{"value":["2018-11-27","3.613800"]},{"value":["2018-11-28","3.615600"]},{"value":["2018-11-29","3.617600"]},{"value":["2018-11-30","3.619000"]},{"value":["2018-12-03","3.621200"]},{"value":["2018-12-04","3.624000"]},{"value":["2018-12-05","3.625800"]},{"value":["2018-12-06","3.626800"]},{"value":["2018-12-07","3.627400"]},{"value":["2018-12-10","3.625200"]},{"value":["2018-12-11","3.623600"]},{"value":["2018-12-12","3.621800"]},{"value":["2018-12-13","3.620200"]},{"value":["2018-12-14","3.617400"]},{"value":["2018-12-17","3.617200"]},{"value":["2018-12-18","3.617200"]},{"value":["2018-12-19","3.616800"]},{"value":["2018-12-20","3.618000"]},{"value":["2018-12-21","3.618400"]},{"value":["2018-12-24","3.620200"]},{"value":["2018-12-25","3.622200"]},{"value":["2018-12-26","3.624000"]},{"value":["2018-12-27","3.626200"]},{"value":["2018-12-28","3.626200"]}],"yAxisIndex":0,"xAxisIndex":0,"name":"sma_50","type":"line","coordinateSystem":"cartesian2d","symbol":"none","lineStyle":{"width":1}}],"title":[{"left":"30","text":"601988.SS 中国银行"}],"tooltip":{"trigger":"axis","axisPointer":{"type":"cross","show":true}},"dataZoom":[{"start":0,"end":100,"xAxisIndex":0}],"toolbox":{"right":"30","feature":{"restore":[],"dataZoom":[],"saveAsImage":[]}},"grid":[{"show":true,"left":"30","right":"30","top":"30"}]},"dispose":true},"evals":[],"jsHooks":[]}</script>

## Issues and Contributions

This package still on the developing stage. If you have any issue when
using this package, please update to the latest version from github. If
the issue still exists, report it at [github
page](https://github.com/ShichenXie/pedquant/issues). Contributions in
any forms to this project are welcome.
