% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/writeFam.R
\name{writeFam}
\alias{writeFam}
\alias{openFamilias}
\title{Export \code{ped} objects to .fam}
\usage{
writeFam(
  ...,
  famfile = "ped.fam",
  params = NULL,
  dbOnly = FALSE,
  openFam = FALSE,
  FamiliasPath = NULL,
  verbose = TRUE
)

openFamilias(famfile = NULL, FamiliasPath = NULL, verbose = TRUE)
}
\arguments{
\item{...}{One or several pedigrees. Each argument should be either a single
\code{ped} object or a list of such. If the pedigrees are unnamed, they are
assigned names "Ped 1", "Ped 2", etc.}

\item{famfile}{The name or path to the output file to be written. The
extension ".fam" is added if missing.}

\item{params}{A list of further parameters; see \code{\link[=readFam]{readFam()}} for valid
entries. See also Details for default values.}

\item{dbOnly}{A logical. If TRUE, no pedigree information is included; only
the frequency database.}

\item{openFam}{A logical. If TRUE, an attempt is made to open the produced
.fam file in an external Familias session. Only available on Windows
systems with a working Familias installation.}

\item{FamiliasPath}{The path to the Familias executable. If empty, the
following are tried in order: "Familias3.exe", "C:/Program Files
(x86)/Familias3/Familias3.exe".}

\item{verbose}{A logical, by default TRUE.}
}
\value{
The file name is returned invisibly.
}
\description{
This function produces a .fam file readable by the Familias software (Egeland
et al., 2000), containing all input pedigrees, their marker data and mutation
models. The option \code{openFam = TRUE} calls \code{openFamilias()} to open a fresh
Familias session with the produced file loaded.
}
\details{
The following parameters are applied by default, but may be adjusted with the
\code{params} argument:
\itemize{
\item \code{dbName = "unknown"}
\item \code{dbSize = 1000}
\item \code{dropout = 0}
\item \code{maf = 0}
\item \code{theta = 0}
}

The \code{params} argument should be a list similar to the \code{params} slot produced
by \code{\link[=readFam]{readFam()}} with \code{includeParams = TRUE}. Single entries are recycled if
needed. If \code{params} contains a vector \code{dropout} with dropout probabilities
for certain pedigree members, it is converted into corresponding
\code{dropoutConsider} and \code{dropoutValue} vectors (see Examples).
}
\examples{

# Create pedigree with 2 markers
x = nuclearPed() |>
  addMarker(geno = c("2/2", "1/3", "2/3"), alleles = 1:3,
            afreq = c(.3,.3,.4), name = "M1")

# Write to .fam
tmp = writeFam(x, famfile = tempfile())

# Read back in
y = readFam(tmp)

stopifnot(identical(x, y))


### With stepwise mutation model
x2 = setMutmod(x, model = "stepwise",
               rate = list(male = 0.001, female = 0.002),
               range = 0.1, rate2 = 0.0001)

# Write and read
y2 = x2 |>
  writeFam(famfile = tempfile()) |>
  readFam()

stopifnot(identical(x2, y2))


### Read/write including detailed parameters
params = list(theta = 0.1, dbName = "myDB", dropout = c("3" = 0.01))
fam = writeFam(x2, famfile = tempfile(), params = params)

dat = readFam(fam, includeParams = TRUE)

# Pedigree is now in the `main` slot
stopifnot(identical(x2, dat$main))

# The `dropout` parameter is converted to (and is equivalent to):
dat$params$dropoutConsider
dat$params$dropoutValue


### Read/write frequency database

# Write database as fam file
dbfam = writeFam(x2, famfile = tempfile(), dbOnly = TRUE)

# Read back in: The result is a list of marker attributes
a = readFam(dbfam)

# Attach to a pedigree and write to a new file
z = singleton(1) |> setMarkers(locusAttributes = a)
dbfam2 = writeFam(z, famfile = tempfile(), dbOnly = TRUE)

stopifnot(identical(readLines(dbfam), readLines(dbfam2)))

}
\references{
Egeland et al. (2000). \emph{Beyond traditional paternity and
identification cases. Selecting the most probable pedigree.} Forensic Sci
Int 110(1): 47-59.
}
\seealso{
\code{\link[=readFam]{readFam()}}.
}
