#' Import Avantes binary file
#'
#' Internal function used by \code{\link{getspec}} to parse Avantes binary files
#' generated by the Avasoft software.
#'
#' @param path Path of the ABS, TRM or ROH file
#'
#' @return A dataframe with 2 columns: first column contains the wavelengths and
#' second column contains the processed spectrum
#'
#' @keywords internal
#'
#' @author Hugo Gruson \email{hugo.gruson+R@@normalesup.org}
#'
#' @seealso \code{\link{getspec}}
#'
parse_avantes <- function(filename) {
  # Reads a single binary file saved by Avantes AvaSoft7USB2 software.

  # Translation of the matlab script from:
  # Copyright: (cc-by) Kotya Karapetyan, 2011.
  # kotya.karapetyan@gmail.com

  # Binary files structure provided by Avantes (http://www.avantes.com/)

  # Translation into R by Hugo Gruson

  if (!grepl("\\.(abs|trm|roh)$", filename, ignore.case = TRUE)) {
    stop(
      "Unsupported file format. Please provide a file with either ",
      "ABS, TRM or ROH ",
      "file extension."
    )
  }

  f <- file(filename, "rb")

  # Header
  versionID <- readBin(f, "numeric", n = 1, size = 4)
  serialnr <- intToUtf8(readBin(f, "numeric", 9, 4))
  userfriendlyname <- intToUtf8(readBin(f, "numeric", 64, 4))

  # Coefficients for the polynome controlling wavelength sampling
  WLIntercept <- readBin(f, "numeric", 1, 4)
  WLX1 <- readBin(f, "numeric", 1, 4)
  WLX2 <- readBin(f, "numeric", 1, 4)
  WLX3 <- readBin(f, "numeric", 1, 4)
  WLX4 <- readBin(f, "numeric", 1, 4)

  ipixfirst <- as.numeric(readBin(f, "numeric", 1, 4))
  ipixlast <- as.numeric(readBin(f, "numeric", 1, 4))

  measuremode <- readBin(f, "numeric", 1, 4)
  dummy1 <- readBin(f, "numeric", 1, 4)
  laserwavelength <- readBin(f, "numeric", 1, 4)
  laserdelay <- readBin(f, "numeric", 1, 4)
  laserwidth <- readBin(f, "numeric", 1, 4)
  strobercontrol <- readBin(f, "numeric", 1, 4)
  dummy2 <- readBin(f, "numeric", 2, 4)
  timestamp <- readBin(f, "numeric", 1, 4)
  dyndarkcorrection <- readBin(f, "numeric", 1, 4)

  smoothpix <- readBin(f, "numeric", 1, 4)
  smoothmodel <- readBin(f, "numeric", 1, 4)
  triggermode <- readBin(f, "numeric", 1, 4)
  triggersource <- readBin(f, "numeric", 1, 4)
  triggersourcetype <- readBin(f, "numeric", 1, 4)
  NTC1 <- readBin(f, "numeric", 1, 4) # onboard temp in degrees Celsius
  NTC2 <- readBin(f, "numeric", 1, 4) # NTC2 in Volt (not connected)
  Thermistor <- readBin(f, "numeric", 1, 4) # detector temp in degr Celsius (only TEC, NIR)
  dummy3 <- readBin(f, "numeric", 1, 4)

  # Data
  if (grepl("\\.(abs|trm)$", filename, ignore.case = TRUE)) {
    data <- readBin(f, "numeric", 3 * (ipixlast - ipixfirst + 1), 4)
    data <- setNames(
      data.frame(matrix(data, ncol = 3, byrow = TRUE)),
      c("scope", "reference", "dark")
    )
  } else { # scope mode
    data <- data.frame(
      "scope" = readBin(f, "numeric", ipixlast - ipixfirst + 1, 4),
      "reference" = NA,
      "dark" = NA
    )
  }

  # Footer
  inttime <- readBin(f, "numeric", 1, 4) # integration time [ms] during file-save
  average <- readBin(f, "numeric", 1, 4) # nr of average during file-save
  integrationdelay <- readBin(f, "numeric", 1, 4)

  close(f)

  len <- nrow(data)

  data$wl <- rep_len(WLIntercept, len) +
    WLX1 * seq(0, len - 1) +
    WLX2 * seq(0, len - 1)^2 +
    WLX3 * seq(0, len - 1)^3 +
    WLX4 * seq(0, len - 1)^4

  data$spec <- (data$scope - data$dark) / (data$reference - data$dark) * 100

  return(data[, c("wl", "spec")])
}
