% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/minCharChange.R
\name{minCharChange}
\alias{ancPropStateMat}
\alias{minCharChange}
\title{Estimating the Minimum Number of Character Transitions Using Maximum Parsimony}
\usage{
minCharChange(trait, tree, randomMax = 10000, maxParsimony = TRUE,
  orderedChar = FALSE, type = "MPR", cost = NULL, printMinResult = TRUE)

ancPropStateMat(trait, tree, orderedChar = FALSE, type = "MPR",
  cost = NULL)
}
\arguments{
\item{trait}{A vector of trait values for a discrete character, preferably named with taxon names
identical to the tip labels on the input tree.}

\item{tree}{A cladogram of type 'phylo'. Any branch lengths are ignored.}

\item{randomMax}{The maximum number of cladograms examined when searching a large number of solutions
consistent with the reconstructed ancestral states from \code{ancestral.pars} with the minimum number
of character state transitions. If the number of potential solutions is less than randomMax, then
solutions are exhaustively searched.}

\item{maxParsimony}{If maxParsimony is TRUE (the default) then only solutions with the smallest
number of total transitions examined will be returned. Note that since solutions are stochastically
'guessed' at, and the number of possible solutions may not be exhaustively searched, there may have
been solutions not examined with a lower number of transitions even if \code{maxParsimony = TRUE}. Regardless,
one may want to do \code{maxParsimony = FALSE} if one is interested in whether there are solutions with a
smaller number of gains or losses and thus wants to return all solutions.}

\item{orderedChar}{If TRUE (not the default), then the character will be reconstructed with a cost (step)
matrix of a linear, ordered character. This is not applicable if \code{type = "ACCTRAN"}, as cost matrices cannot
be used with ACCTRAN in \code{ancestral.pars}, and an error will be returned if \code{orderedChar = TRUE} but
a cost matrix is given, as the only reason to use orderedChar is to produce a cost matrix automatically.}

\item{type}{The parsimony algorithm applied by \code{ancestral.pars}, which can apply one of two:
"MPR" (the default) is a relatively fast algorithm developed by Hamazawa et al. (1995) and Narushima
and Hanazawa (1997), which relies on reconstructing the states at each internal node by re-rooting at
that node.  "ACCTRAN", the 'accelerated transitions' algorithm (Swofford and Maddison, 1987), favors
character reversal over independent gains when there is ambiguity. The "ACCTRAN" option in
ancestral.pars avoids repeated rerooting of the tree to search for a smaller set of maximum-parsimony
solutions that satisfy the ACCTRAN algorithm, but does so by assigning edge weights.
As of phangorn v1.99-12, both of these algorithms apply
the Sankoff parsimony algorithm, which allows multifurcations (polytomies).}

\item{cost}{A matrix of the cost (i.e. number of steps) necessary to change between states of the input
character trait. If NULL (the
default), the character is assumed to be unordered with equal cost to change from any state to another.
Cost matrices only impact the "MPR" algorithm; if a cost matrix is given but \code{type = "ACCTRAN"}, an error
is issued.}

\item{printMinResult}{If TRUE (the default), a summary of the results is printed to the terminal. The
information in this summary may be more detailed if the results of the analysis are simpler (i.e.
fewer unique solutions).}
}
\value{
A list is invisibly returned containing the following elements:

\describe{
 \item{\code{message}}{Describes the performance of \code{minCharChange} at searching for a minimum solution.}
\item{\code{sumTransitions}}{A vector recording the total number of necessary transitions (sum total of gains
and losses/reversal) for each solution; effectively the parsimony cost of each solution.}
\item{\code{minTransitions}}{A symmetrical matrix with number of rows and columns equal to the number of
character states, with values in each cell indicating the minimum number of transitions from one ancestral
state (i.e. the rows) to a descendant state (i.e. the columns), taken across the set of kept solutions
(dependent on which are kept as decided by argument \code{maxParsimony}).  Generally guaranteed not to
add up to the number of edges contained within the input tree, and thus may not represent any realistic
evolutionary scenario but does represent a conservative approach for asking 'what is the smallest possible
number of transitions from 0 to 1' or 'smallest possible number of transitions from 1 to 0', independently
of each other.}
\item{\code{solutionArray}}{A three-dimensional array, where for each solution, we have a matrix with edges
for rows and two columns indicating the ancestral and child nodes of that edge, with values indicating the
states inferred for those nodes in a particular solution.}
\item{\code{transitionArray}}{A labelled three-dimensional array where for each solution we have a symmetrical
matrix with number of rows and columns equal to the number of character states, with values in each cell
indicating the total number of transitions from one ancestral state (i.e. the rows) to a descendant state
(i.e. the columns).}
\item{\code{transitionSumChanges}}{Which is a three column matrix with a row for every solution, with the
values in the three columns measuring the number of edges (branches) inferred to respectively have gains,
no change or losses (i.e. reversals), as calculated relative to the order of character states.}
}
}
\description{
\code{minCharChange} is a function which takes a cladogram and a discrete trait and finds the
solutions of inferred character states for ancestral nodes that minimizes the number of
character state transitions (either gains or losses/reversals) for a given topology and a set of
discrete character data. \code{minCharChange} relies on \code{ancPropStateMat}, which is a wrapper
for \code{phangorn}'s function \code{ancestral.pars}.
}
\details{
The wrapper function \code{ancPropStateMat} simply automates the application of functions
\code{ancestral.pars} and \code{phyDat} from \code{phangorn}, along with several additional checks
and code to present the result as a matrix, rather than a specialized list.

Note that although the default \code{orderedChar} argument assumes that multistate characters are unordered,
the results of character change will always be reported as gains and losses relative to the numbering of the
states in the output \code{transitionSumChanges}, exactly as if they had been ordered. In the case
where the character is actually ordered, this may be
considered a conservative approach, as using a parsimony algorithm for unordered character states allows fewer
gains or losses to be counted on branches where multiple gains and losses are reported. If the character is
presumably unordered \emph{and multistate}, however, then the gains and losses division
is \emph{arbitrary nonsense} and should be combined to
to obtain the total number of character changes.
}
\examples{
# let's write a quick & dirty ancestral trait plotting function

quickAncPlotter<-function(tree,ancData,cex){
	ancCol<-(1:ncol(ancData))+1
	plot(tree,show.tip.label=FALSE,no.margin=TRUE,direction="upwards")
	tiplabels(pch=16,pie=ancData[(1:Ntip(tree)),],cex=cex,piecol=ancCol,
		col=0)
	nodelabels(pie=ancData[-(1:Ntip(tree)),],cex=cex,piecol=ancCol)
	}

# example with retiolitid graptolite data

data(retiolitinae)

#unordered, MPR
ancMPR<-ancPropStateMat(retioTree, trait=retioChar[,2], type="MPR")
#unordered, ACCTRAN
ancACCTRAN<-ancPropStateMat(retioTree, trait=retioChar[,2], type="ACCTRAN")

#let's compare MPR versus ACCTRAN results
layout(1:2)
quickAncPlotter(retioTree,ancMPR,cex=0.5)
text(x=4,y=5,"type='MPR'",cex=1.5)
quickAncPlotter(retioTree,ancACCTRAN,cex=0.5)
text(x=5,y=5,"type='ACCTRAN'",cex=1.5)

minCharChange(retioTree,trait=retioChar[,2],type="MPR")
minCharChange(retioTree,trait=retioChar[,2],type="ACCTRAN")

# with simulated data

set.seed(444)
tree<-rtree(50)
#simulate under a likelihood model
char<-rTraitDisc(tree,k=3,rate=0.7)
tree$edge.length<-NULL
tree<-ladderize(tree)

#unordered, MPR
ancMPR<-ancPropStateMat(tree, trait=char, type="MPR")
#unordered, ACCTRAN
ancACCTRAN<-ancPropStateMat(tree, trait=char, type="ACCTRAN")
#ordered, MPR
ancMPRord<-ancPropStateMat(tree, trait=char, orderedChar=TRUE, type="MPR")

#let's compare MPR versus ACCTRAN results
layout(1:2)
quickAncPlotter(tree,ancMPR,cex=0.3)
text(x=8,y=15,"type='MPR'",cex=1.5)
quickAncPlotter(tree,ancACCTRAN,cex=0.3)
text(x=9,y=15,"type='ACCTRAN'",cex=1.5)
#MPR has much more uncertainty in node estimates
	#but that doesn't mean ACCTRAN is preferable

#let's compare unordered versus ordered under MPR
layout(1:2)
quickAncPlotter(tree,ancMPR,cex=0.3)
text(x=8,y=15,"ordered char",cex=1.5)
quickAncPlotter(tree,ancMPRord,cex=0.3)
text(x=9,y=15,"ordered char'",cex=1.5)
layout(1)

\dontrun{
# what ancPropStateMat automates (with lots of checks):

require(phangorn)

char1<-matrix(char,,1)
rownames(char1)<-names(char)
#translate into something for phangorn to read
char1<-phyDat(char1,type="USER",levels=sort(unique(char1)))
x<-ancestral.pars(tree,char1,type="MPR")
y<-ancestral.pars(tree,char1,type="ACCTRAN")
}

#estimating minimum number of transitions with MPR
minCharChange(tree,trait=char,type="MPR")

#and now with ACCTRAN
minCharChange(tree,trait=char,type="ACCTRAN")
}
\author{
David W. Bapst
}
\references{
Hanazawa, M., H. Narushima, and N. Minaka. 1995. Generating most parsimonious reconstructions on
a tree: A generalization of the Farris-Swofford-Maddison method. Discrete Applied Mathematics
56(2-3):245-265.

Narushima, H., and M. Hanazawa. 1997. A more efficient algorithm for MPR problems in phylogeny.
Discrete Applied Mathematics 80(2-3):231-238.

Schliep, K. P. 2011. phangorn: phylogenetic analysis in R. \emph{Bioinformatics} 27(4):592-593.

Swofford, D. L., and W. P. Maddison. 1987. Reconstructing ancestral character states under
Wagner parsimony. Mathematical Biosciences 87(2):199-229.
}
\seealso{
The functions described here are effectively wrapers of \code{phangorn}'s function
\code{ancestral.pars}.
}

