% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/netcomparison.R
\name{netcomparison}
\alias{netcomparison}
\alias{print.netcomparison}
\title{Calculate comparison effects of two arbitrary complex interventions
in component network meta-analysis}
\usage{
netcomparison(
  x,
  treat1,
  treat2,
  fixed = x$fixed,
  random = x$random,
  level = x$level.ma,
  nchar.comps = x$nchar.comps
)

\method{print}{netcomparison}(
  x,
  fixed = x$fixed,
  random = x$random,
  backtransf = x$backtransf,
  nchar.comps = x$nchar.comps,
  digits = gs("digits"),
  digits.stat = gs("digits.stat"),
  digits.pval = gs("digits.pval"),
  scientific.pval = gs("scientific.pval"),
  zero.pval = gs("zero.pval"),
  JAMA.pval = gs("JAMA.pval"),
  big.mark = gs("big.mark"),
  legend = TRUE,
  ...
)
}
\arguments{
\item{x}{An object of class \code{netcomb} or \code{netcomparison}
(print function).}

\item{treat1}{A character vector defining the first complex
intervention(s).}

\item{treat2}{A character vector defining the second complex
intervention(s).}

\item{fixed}{A logical indicating whether results for fixed effects
/ common effects model should be conducted.}

\item{random}{A logical indicating whether results for random
effects model should be conducted.}

\item{level}{The level used to calculate confidence intervals for
combinations of components.}

\item{nchar.comps}{A numeric defining the minimum number of
characters used to create unique names for components (see
Details).}

\item{backtransf}{A logical indicating whether printed results
should be back transformed. If \code{backtransf=TRUE}, results
for \code{sm="OR"} are printed as odds ratios rather than log
odds ratios.}

\item{digits}{Minimal number of significant digits, see
\code{print.default}.}

\item{digits.stat}{Minimal number of significant digits for z-value
of test for overall effect, see \code{print.default}.}

\item{digits.pval}{Minimal number of significant digits for
p-values, see \code{print.default}.}

\item{scientific.pval}{A logical specifying whether p-values should
be printed in scientific notation, e.g., 1.2345e-01 instead of
0.12345.}

\item{zero.pval}{A logical specifying whether p-values should be
printed with a leading zero.}

\item{JAMA.pval}{A logical specifying whether p-values for test of
combination effect should be printed according to JAMA reporting
standards.}

\item{big.mark}{A character used as thousands separator.}

\item{legend}{A logical indicating whether a legend should be
printed.}

\item{\dots}{Additional arguments (ignored).}
}
\value{
A list is returned by the function \code{netcomparison} with the
following elements:
\item{comparison}{Comparison.}
\item{TE.fixed, TE.random}{A vector of comparison effects (fixed
  and random effects model).}
\item{seTE.fixed, seTE.random}{A vector with corresponding standard
  errors (fixed and random effects model).}
\item{lower.fixed, lower.random}{A vector with lower confidence
  limits for comparisons (fixed and random effects model).}
\item{upper.fixed, upper.random}{A vector with upper confidence
  limits for comparisons (fixed and random effects model).}
\item{statistic.fixed, statistic.random}{A vector with z-values for
  the overall effect of comparisons (fixed and random effects
  model).}
\item{pval.fixed, pval.random}{A vector with p-values for the
  overall effect of comparisons (fixed and random effects model).}
\item{trts}{Treatments included in comparisons.}
\item{comps}{Components included in comparisons.}
\item{treat1, treat2}{A defined above.}
\item{fixed, random}{A defined above.}
\item{level, nchar.comps, backtransf, x}{A defined above.}
\item{B.matrix}{B matrix.}
\item{C.matrix}{C matrix.}
}
\description{
Calculate comparison effects of two arbitrary complex interventions
(i.e., combinations of several components) in component network
meta-analysis.
}
\details{
R functions \code{\link{netcomb}} and \code{\link{discomb}}
calculate effects for individual components and complex
interventions present in the component network meta-analysis
(CNMA). This function can be used to calculate the effect for
comparisons of two arbitrary complex interventions defined by
arguments \code{treat1} and \code{treat2}.

All complex interventions occuring in the network are considered
for the first complex intervention if argument \code{treat1} is
missing. The reference group defined in the (C)NMA is used as
second complex intervention if argument \code{treat2} is
missing. The first complex intervention in the (C)NMA is used if
the reference group is not defined.

The following matrices are needed to calculate comparison effects
of arbitrary complex interventions, (Rücker et al., 2020, Section
3.2):
\itemize{
\item B matrix describing how comparisons are composed by complex
  intervetions,
\item C matrix describing how the complex interventions are
  composed by the components.
}
Internally, both matrices are constructed based on arguments
\code{x}, \code{treat1} and \code{treat2}.

By default, component names are not abbreviated in
printouts. However, in order to get more concise printouts,
argument \code{nchar.comps} can be used to define the minimum
number of characters for abbreviated component names (see
\code{\link{abbreviate}}, argument \code{minlength}). R function
\code{\link{treats}} is utilised internally to create abbreviated
component names.
}
\note{
R function \code{\link{netcomplex}} can be used to calculate the
effect for arbitrary complex interventions in a component network
meta-analysis.
}
\examples{
data(Linde2016)

# Only consider studies including Face-to-face PST (to reduce
# runtime of example)
#
face <- subset(Linde2016, id \%in\% c(16, 24, 49, 118))

# Conduct random effects network meta-analysis
#
net1 <- netmeta(lnOR, selnOR, treat1, treat2, id,
                data = face, ref = "placebo",
                sm = "OR", fixed = FALSE)

# Additive model for treatment components (with placebo as inactive
# treatment)
#
nc1 <- netcomb(net1, inactive = "placebo")

# Result for comparison Face-to-face PST vs TCA
netcomparison(nc1, "Face-to-face PST", "TCA", nchar.comps = 4)
netcomparison(nc1, "F", "T", nchar.comps = 4)

# Result for comparison Face-to-face PST vs TCA + Placebo
netcomparison(nc1, "Face-to-face PST", "TCA + Plac", nchar.comps = 4)

\dontrun{
# Artificial example
t1 <- rep("A", 3)
t2 <- c("B+C", "A+C", "C+D")
TE <- c(0, 1, 0)
seTE <- rep(1, 3)
# Conduct (C)NMA
net2 <- netmeta(TE, seTE, t1, t2, random = FALSE)
nc2 <- netcomb(net2)

# Result for comparison A vs B + D
netcomparison(nc2, "A", "B + D")
# Same results
netcomparison(nc2, "A", "B+D")
netcomparison(nc2, "A", "D+B")
netcomparison(nc2, "a", "d+b")

# Generated B matrix
netcomparison(nc2, "A", "B + D")$C.matrix
# Generated B matrix
netcomparison(nc2, "A", "B + D")$B.matrix
}

}
\references{
Rücker G, Petropoulou M, Schwarzer G (2020):
Network meta-analysis of multicomponent interventions.
\emph{Biometrical Journal},
\bold{62}, 808--21
}
\seealso{
\code{\link{netcomb}}, \code{\link{discomb}},
  \code{\link{netcomplex}}
}
\author{
Guido Schwarzer \email{sc@imbi.uni-freiburg.de}
}
