% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_omega.R
\name{plot_omega}
\alias{plot_omega}
\title{Plot weights for propensity score estimation in the navigated weighting}
\usage{
plot_omega(object, relative = TRUE)
}
\arguments{
\item{object}{an object of class “nawt”, usually, a result of a call to \code{\link{nawt}}.
Note that it cannot be used when the \code{object} is a result of a call to
\code{nawt} where \code{method = "both"} and \code{twostep = FALSE}.}

\item{relative}{a logical value indicating whether or not relative weights 
standardized to have mean one are shown.}
}
\value{
No retrun value, called for side effects.
}
\description{
Plots weight of each observation in the score condition \eqn{\omega(\pi)} for
  propensity score estimation and estimated propensity score distribution 
  in the navigated weighting.
}
\details{
The x-axis shows estimated propensity scores, and the y-axis shows weight of 
each observation in propensity score estimation. When \code{estimand = "ATE"},
the navigated weighting estimates two propensity scores for each observation;
one for estimating the average of the potential outcomes with treatment and  
the other for estimating the average of the potential outcomes without 
treatment. Therefore, there are two weighting functions for estimating two 
sets of propensity scores and two propensity score distributions. Points 
rising to the right and a solid curve represent the weighting functions and 
distribution of propensity scores for estimating the average of the potential
outcomes without treatment whereas points rising to the left and a dashed 
curve represent the weighting functions and distribution of propensity scores
for estimating the average of the potential outcomes with treatment.

Position of the legend is determined internally.
}
\examples{
# Simulation from Kang and Shafer (2007) and Imai and Ratkovic (2014)
tau <- 10
set.seed(12345)
n <- 1000
X <- matrix(rnorm(n * 4, mean = 0, sd = 1), nrow = n, ncol = 4)
prop <- 1 / (1 + exp(X[, 1] - 0.5 * X[, 2] + 0.25 * X[, 3] + 0.1 * X[, 4]))
treat <- rbinom(n, 1, prop)
y <- 210 + 27.4 * X[, 1] + 13.7 * X[, 2] + 13.7 * X[, 3] + 13.7 * X[, 4] + 
     tau * treat + rnorm(n)

# Data frame and formulas for propensity score estimation
df <- data.frame(X, treat, y)
colnames(df) <- c("x1", "x2", "x3", "x4", "treat", "y")
formula_c <- as.formula(treat ~ x1 + x2 + x3 + x4)

# Power weighting function with alpha = 2
# ATT estimation
fitatt <- nawt(formula = formula_c, outcome = "y", estimand = "ATT", 
               method = "score", data = df, alpha = 2)
plot_omega(fitatt)

# ATE estimation
fitate <- nawt(formula = formula_c, outcome = "y", estimand = "ATE", 
               method = "score", data = df, alpha = 2)
plot_omega(fitate)

# Use method = "both"
# Two-step estimation
fitateb2s <- nawt(formula = formula_c, outcome = "y", estimand = "ATE", 
                  method = "both", data = df, alpha = 2, twostep = TRUE)
plot_omega(fitateb2s)

# Continuously-updating GMM estimation
\dontrun{
fitatebco <- nawt(formula = formula_c, outcome = "y", estimand = "ATE", 
                  method = "both", data = df, alpha = 2, twostep = FALSE)
plot_omega(fitatebco) # error}
}
\author{
Hiroto Katsumata
}
