#' Gradient score function
#'
#' Compute the peaks-over-threshold gradient score function for the Brown--Resnick model.
#'
#' The function computes the gradient score based on the representation developed by Wadsworth et al. (2013).
#' Margins must have been standardized. The weighting function must differentiable and verify some properties
#' for consistency, see Fondeville and Davison (2016) for more details.
#'
#'
#' @param obs List of vectors exceeding an R-threshold, see Fondeville and Davison (2016) for more details.
#' @param loc Matrix of coordinates as given by \code{expand.grid()}.
#' @param vario Semi-variogram function taking a vector of coordinates as input.
#' @param weigthFun Function of weights.
#' @param dWeigthFun Partial derivative function of \code{weigthFun}.
#' @param ... Parameters for \code{weigthFun} and \code{dWeigthFun}.
#' @param nCores Number of cores used for the computation
#' @param cl Cluster instance as created by \code{makeCluster} of the \code{parallel} package.
#' @return Evaluation of the gradient score function for the set of observations \code{obs} and semi-variogram \code{vario}.
#' @examples
#' #Define variogram function
#' vario <- function(h){
#'    1 / 2 * norm(h,type = "2")^1.5
#' }
#'
#' #Define locations
#' loc <- expand.grid(1:4, 1:4)
#'
#' #Simulate data
#' obs <- simulPareto(1000, loc, vario)
#'
#' #Evaluate risk functional
#' sums <- sapply(obs, sum)
#'
#' #Define weighting function
#' weigthFun <- function(x, u){
#'x * (1 - exp(-(sum(x) / u - 1)))
#'}
#'
#' #Define partial derivative of weighting function
#' dWeigthFun <- function(x, u){
#'(1 - exp(-(sum(x) / u - 1))) + (x / u) * exp( - (sum(x) / u - 1))
#'}
#'
#' #Select exceedances
#' threshold <- quantile(sums, 0.9)
#' exceedances <- obs[sums > threshold]
#'
#' #Evaluate gradient score function
#' scoreEstimation(exceedances, loc, vario, weigthFun, dWeigthFun, u = threshold)
#' @export
#' @references Fondeville, R. de and Davison A. (2016). High-dimensional Peaks-over-threshold Inference for Brown-Resnick Processes. Submitted.

scoreEstimation <- function(obs, loc, vario, weigthFun, dWeigthFun, nCores = 1, cl = NULL,  ... ){

  if(class(obs) != "list" || length(obs) < 1 || class(obs[[1]]) != "numeric"){
    stop('obs must be a list of vectors.')
  }
  if(class(loc) != "data.frame") {
    stop('loc must be the data frame of coordinates as generated by expand.grid().')
  }
  if(class(weigthFun) != "function") {
    stop('weigthFun must be a function.')
  }
  if(class(dWeigthFun) != "function") {
    stop('weigthFun must be a function.')
  }
  if(!is.numeric(nCores) || nCores < 1) {
    stop('nCores must a positive number of cores to use for parallel computing.')
  }
  if(nCores > 1 && length(grep("cluster",class(cl))) == 0) {
    stop('For parallel computation, cl must an cluster created by makeCluster of the package parallel.')
  }

  n <- length(obs)
  dim <- nrow(loc)

  if(dim != length(obs[[1]])){
    stop('The size of the vectors of observations does not match grid size.')
  }

  gamma <- tryCatch({
    dists <- lapply(1:ncol(loc), function(i) {
      outer(loc[,i],loc[,i], "-")
      })

    computeVarMat <- sapply(1:length(dists[[1]]), function(i){
      h <- rep(0,ncol(loc))
      for(j in 1:ncol(loc)){
        h[j] = dists[[j]][i]
      }
      vario(h)
    })
    matrix(computeVarMat, dim, dim)
  }, warning = function(war) {
    war
  }, error = function(err) {
    stop('The semi-variogram provided is not valide for the provided locations.')
  })
  gammaOrigin <- apply(loc, 1, vario)

  SigmaS <- (outer(gammaOrigin,gammaOrigin, "+") - gamma)
  sigmaInv <- MASS::ginv(SigmaS)
  sigma <- diag(SigmaS)
  q <- rowSums(sigmaInv)

  A <- sigmaInv - q %*% t(q)/sum(q)
  zeroDiagA <- A
  diag(zeroDiagA) <- 0
  mtp <- 2*q / (sum(q)) + 2 + sigmaInv %*% sigma - (q %*% t(q) %*% sigma)/(sum(q))

  computeScores = function(i){
    gradient <- - 1 / 2 * ((A + t(A)) %*% log(obs[[i]])) *  (1 / obs[[i]]) - 1/2* (1/obs[[i]]) * mtp
    laplacian <- - 1 / 2 * diag(A + t(A)) * (1/ obs[[i]]^2) + 1 / 2 * ((A + t(A)) %*% log(obs[[i]])) * (1/obs[[i]])^2  + 1/2* (1/obs[[i]])^2 * mtp

    weights <- weigthFun(obs[[i]], ...)
    dWeights  <- dWeigthFun(obs[[i]], ...)

    sum(2 * (weights * dWeights) * gradient + weights^2 * laplacian + 1 / 2 * weights^2 * gradient^2)
  }

  if(nCores > 1){
    scores <- parallel::parLapply(cl, 1:n, computeScores)
  } else {
    scores <- lapply(1:n, computeScores)
  }

  sum(unlist(scores))/n
}
