% Generated by roxygen2 (4.0.2): do not edit by hand
\name{cluster.boot}
\alias{cluster.boot}
\title{Bootstrapped multi-way standard error clustering}
\usage{
cluster.boot(model, cluster, parallel = FALSE, use_white = NULL,
  force_posdef = FALSE, R = 300, boot_type = "xy",
  wild_type = "rademacher", debug = FALSE)
}
\arguments{
\item{model}{The estimated model, usually an \code{lm} or \code{glm} class object}

\item{cluster}{A vector, \code{matrix}, or \code{data.frame} of cluster variables,
where each column is a separate variable.  If the vector \code{1:nrow(data)}
is used, the function effectively produces a regular
heteroskedasticity-robust matrix.}

\item{parallel}{Scalar or list.  If a list, use the list as a list
of connected processing cores/clusters.  Scalar values of \code{TRUE}
and \code{"snow"} (which are equivalent) ask \code{boot} to handle parallelization, as does
\code{"multicore"}.  See the \code{parallel} and \code{boot} package.}

\item{use_white}{Logical or \code{NULL}.  See description below.}

\item{force_posdef}{Logical.  Force the eigenvalues of the variance-covariance
matrix to be positive.}

\item{R}{\code{Integer}. The number of bootstrap replicates; passed directly to \code{boot}.}

\item{boot_type}{\code{"xy"}, \code{"residual"}, or \code{"wild"}.  See details.}

\item{wild_type}{\code{"rademacher"}, \code{"mammen"}, or \code{"norm"}.  See details.}

\item{debug}{Logical.  Print internal values useful for debugging to
the console.}
}
\value{
a \eqn{k} x \eqn{k} variance-covariance matrix of type \code{matrix}
}
\description{
Return a bootstrapped multi-way cluster-robust variance-covariance matrix
}
\details{
This function implements cluster bootstrapping (also known as the block bootstrap)
for variance-covariance matrices, following Cameron, Gelbach, & Miller (CGM) (2008).
Usage is generally similar to the \code{cluster.vcov} function in this package, but this
function does not support degrees of freedome corrections or leverage adjustments.

In the terminology that CGM (2008) use, this function implements
\emph{pairs, residual, or wild cluster bootstrap-se}.

A pairs (or xy) cluster bootstrap can be obtained by setting \code{boot_type = "xy"},
which resamples the entire regression data set (both X and y).
Setting \code{boot_type = "residual"} will obtain a residual cluster
bootstrap, which resamples only the residuals (in this case, we resample the blocks/clusters
rather than the individual observations' residuals).  To get a wild cluster bootstrap set
\code{boot_type = "wild"}, which does not resample anything, but instead reforms the
dependent variable by multiplying the residual by a randomly drawn value and adding the
result to the fitted value.  The default method is the pairs/xy bootstrap.

There are three built-in distributions to draw multipliers from for wild bootstraps:
the Rademacher (\code{wild_type = "rademacher"}, the default), which draws from [-1, 1],
each with P = 0.5, Mammen's suggested distribution (\code{wild_type = "mammen"}, see
Mammen, 1993), and the standard normal/Gaussian distribution (\code{wild_type = "norm"}).
The default is the Rademacher distribution, following CGM (2008).  Alternatively, you can
set the function to draw multipliers from by assigning
\code{wild_type} to a function that takes no arguments and returns a single real value.

Multi-way clustering is handled as described by Petersen (2009) and generalized
according to Cameron, Gelbach, & Miller (2011).  This means that cluster.boot
estimates a set of variance-covariance matrices \emph{for the variables} separately
and then sums them (subtracting some matrices and adding others).
The method described by CGM (2011) estimates a set of variance-covariance matrices
\emph{for the residuals} (sometimes referred to as the meat of the sandwich estimator)
and sums them appropriately.  Whether you sum the meat matrices and then compute
the model's variance-covariance matrix or you compute a series of model matrices
and sum those is mathematically irrelevant, but may lead to (very) minor numerical
differences.

Unlike the \code{cluster.vcov} function, this function does not depend upon the \code{estfun}
function from the \bold{sandwich} package, although it does make use of the \code{vcovHC}
function for computing White HC0 variance-covariance matrices.

Parallelization (if used) is handled by the \bold{boot} package.  Be sure to set
\code{options(boot.ncpus = N)} where \code{N} is the number of CPU cores you want
the \code{boot} function to use.
}
\examples{
\dontrun{
library(lmtest)
data(petersen)
m1 <- lm(y ~ x, data = petersen)

# Cluster by firm
boot_firm <- cluster.boot(m1, petersen$firmid)
coeftest(m1, boot_firm)

# Cluster by year
boot_year <- cluster.boot(m1, petersen$year)
coeftest(m1, boot_year)

# Double cluster by firm and year
boot_both <- cluster.boot(m1, cbind(petersen$firmid, petersen$year))
coeftest(m1, boot_both)

# Cluster by firm with wild bootstrap and custom wild distribution
boot_firm2 <- cluster.boot(m1, petersen$firmid, boot_type = "wild",
                           wild_type = function() sample(c(-1, 1), 1))
coeftest(m1, boot_firm)

# Go multicore using the parallel package
require(parallel)
cl <- makeCluster(4)
options(boot.ncpus = 4)
boot_both <- cluster.boot(m1, cbind(petersen$firmid, petersen$year), parallel = cl)
stopCluster(cl)
coeftest(m1, boot_both)

# Go multicore using the parallel package, let boot handle the parallelization
require(parallel)
options(boot.ncpus = 8)
boot_both <- cluster.boot(m1, cbind(petersen$firmid, petersen$year), parallel = TRUE)
coeftest(m1, boot_both)
}
}
\author{
Nathaniel Graham \email{npgraham1@gmail.com}
}
\references{
Cameron, A. C., Gelbach, J. B., & Miller, D. L. (2008). Bootstrap-based improvements
for inference with clustered errors. The Review of Economics and Statistics, 90(3), 414-427.

Cameron, A. C., Gelbach, J. B., & Miller, D. L. (2011). Robust inference with multiway
clustering. Journal of Business & Economic Statistics, 29(2).

Mammen, E. (1993). Bootstrap and wild bootstrap for high dimensional linear models. The
Annals of Statistics, 255-285.

Petersen, M. A. (2009). Estimating standard errors in finance panel data sets: Comparing
approaches. Review of financial studies, 22(1), 435-480.
}
\keyword{block}
\keyword{boot}
\keyword{bootstrap}
\keyword{clustering}
\keyword{errors}
\keyword{multi-way}
\keyword{robust}
\keyword{standard}

