% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/df_stats.R
\name{df_stats}
\alias{df_stats}
\title{Calculate statistics for "response" variables}
\usage{
df_stats(
  formula,
  data,
  ...,
  drop = TRUE,
  fargs = list(),
  sep = "_",
  format = c("wide", "long"),
  groups = NULL,
  long_names = FALSE,
  nice_names = FALSE,
  na.action = "na.warn"
)
}
\arguments{
\item{formula}{A formula indicating which variables are to be used.
Semantics are approximately as in \code{\link[=lm]{lm()}} since \code{\link[stats:model.frame]{stats::model.frame()}}
is used to turn the formula into a data frame.  But first conditions and \code{groups}
are re-expressed into a form that \code{\link[stats:model.frame]{stats::model.frame()}} can interpret.
Multiple response variables can be separated by \code{+} on the left hand side of
the formula.  A one-sided formula \code{~ rhs | cond} is treated as \code{rhs ~ 1 | cond}.
See details.}

\item{data}{A data frame or list containing the variables.}

\item{...}{Functions used to compute the statistics.  If this is empty,
a default set of summary statistics is used.  Functions used must accept
a vector of values and return either a (possibly named) single value,
a (possibly named) vector of values, or a data frame with one row.
Functions can be specified with character strings, names, or expressions
that look like function calls with the first argument missing.  The latter
option provides a convenient way to specify additional arguments.  See the
examples.
Note: If these arguments are named, those names will be used in the data
frame returned (see details).  Such names may not be among the names of the named
arguments of \code{df_stats}().

If a function is specified using \code{::}, be sure to include the trailing
parens, even if there are no additional arguments required.}

\item{drop}{A logical indicating whether combinations of the grouping
variables that do not occur in \code{data} should be dropped from the
result.}

\item{fargs}{Arguments passed to the functions in \code{...}.}

\item{sep}{A character string to separate components of names.  Set to \code{""} if
you don't want separation.}

\item{format}{One of \code{"long"} or \code{"wide"} indicating the desired shape of the
returned data frame.}

\item{groups}{An expression or formula to be evaluated in \code{data} and defining (additional) groups.
This isn't necessary, since these can be placed into the formula, but it is provided
for similarity to other functions from the \pkg{mosaic} package.}

\item{long_names}{A logical indicating whether the default names should include the name
of the variable being summarized as well as the summarizing function name in the default
case when names are not derived from the names of the returned object or
an argument name.}

\item{nice_names}{A logical indicating whether \code{\link[=make.names]{make.names()}} should be
used to force names of the returned data frame to by syntactically valid.}

\item{na.action}{A function (or character string naming a function) that determines how NAs are treated.
Options include \code{"na.warn"} which removes missing data and emits a warning,
\code{"na.pass"} which includes all of the data,
\code{"na.omit"} or \code{"na.exclude"} which silently discard missing data,
and \code{"na.fail"} which fails if there is missing data.
See \code{link[stats]{na.pass}()} and \code{\link[=na.warn]{na.warn()}} for details.
The default is \code{"na.warn"} unless no function are specified in \code{...}, in which case
\code{"na.pass"} is used since the default function reports the number of missing values.}
}
\value{
A data frame. Names of columns in the resulting data frame consist of three
parts separated by \code{sep}.
The first part is the argument name, if it exists, else the function.
The second part is the name of the variable being summarised if \code{long_names == TRUE} and
the first part is the function name, else ""
The third part is the names of the object returned by the summarizing function, if they
exist, else a sequence of consecutive integers or "" if there is only one component
returned by the summarizing function.
See the examples.
}
\description{
Creates a data frame of statistics calculated on one or more response variables,
possibly for each group formed by combinations of additional variables.
The resulting data frame has one column
for each of the statistics requested as well as columns for any grouping variables and a
column identifying the response variable for which the statistics was calculated.
}
\details{
Use a one-sided formula to compute summary statistics for the right hand side
expression over the entire data.
Use a two-sided formula to compute summary statistics for the left hand (response)
expression(s) for each combination of levels of the expressions occurring on the
right hand side.
This is most useful when the left hand side is quantitative and each expression
on the right hand side has relatively few unique values.  A function like
\code{\link[mosaic:ntiles]{mosaic::ntiles()}} is often useful to create a few groups of roughly equal size
determined by ranges of a quantitative variable.  See the examples.

Note that unlike \code{dplyr::\link[dplyr]{summarise}()}, \code{df_stats()} ignores
any grouping defined in \code{data} if \code{data} is a grouped \code{tibble}.
}
\section{Cautions Regarding Formulas}{


The use of \code{|} to define groups is tricky because (a) \code{\link[stats:model.frame]{stats::model.frame()}}
doesn't handle this sort of thing and (b) \code{|} is also used for logical or.  The
current algorithm for handling this will turn the first  occurrence of \code{|} into an attempt
to condition, so logical or cannot be used before conditioning in the formula.
If you have need of logical or, we suggest creating a new variable that contains the
results of evaluating the expression.

Similarly, addition (\code{+}) is used to separate grouping variables, not for
arithmetic.
}

\examples{
df_stats( ~ hp, data = mtcars)
# There are several ways to specify functions
df_stats( ~ hp, data = mtcars, mean, trimmed_mean = mean(trim = 0.1), "median",
  range, Q = quantile(c(0.25, 0.75)))
# When using ::, be sure to include parens, even if there are no additional arguments.
df_stats( ~ hp, data = mtcars, mean = base::mean(), trimmed_mean = base::mean(trim = 0.1))

# force names to by syntactically valid
df_stats( ~ hp, data = mtcars, Q = quantile(c(0.25, 0.75)), nice_names = TRUE)
# longer names
df_stats( ~ hp, data = mtcars, mean, trimmed_mean = mean(trim = 0.1), "median", range,
  long_names = TRUE)
# wide vs long format
df_stats( hp ~ cyl, data = mtcars, mean, median, range)
df_stats( hp + wt + mpg ~ cyl, data = mtcars, mean, median, range)
df_stats( hp ~ cyl, data = mtcars, mean, median, range, format = "long")
# More than one grouping variable -- 4 ways.
df_stats( hp ~ cyl + gear, data = mtcars, mean, median, range)
df_stats( hp ~ cyl | gear, data = mtcars, mean, median, range)
df_stats( hp ~ cyl, groups = ~gear, data = mtcars, mean, median, range)
df_stats( hp ~ cyl, groups = gear, data = mtcars, mean, median, range)

# because the result is a data frame, df_stats() is also useful for creating plots
if(require(ggformula)) {
  gf_violin(hp ~ cyl, data = mtcars, group = ~ cyl) |>
  gf_point(mean ~ cyl, data = df_stats(hp ~ cyl, data = mtcars, mean),
    color = ~ "mean") |>
  gf_point(median ~ cyl, data = df_stats(hp ~ cyl, data = mtcars, median),
    color = ~"median") |>
  gf_labs(color = "")
}

# magrittr style piping is also supported
if (require(ggformula)) {
  mtcars |>
    df_stats(hp ~ cyl, mean, median, range)
  mtcars |>
    df_stats(hp ~ cyl + gear, mean, median, range) |>
    gf_point(mean ~ cyl, color = ~ factor(gear)) |>
    gf_line(mean ~ cyl, color = ~ factor(gear))
}

# can be used with a categorical response, too
if (require(mosaic)) {
  df_stats(sex ~ substance, data = HELPrct, table, prop_female = prop)
}
if (require(mosaic)) {
  df_stats(sex ~ substance, data = HELPrct, table, props)
}
}
