% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multilevel.invar.R
\name{multilevel.invar}
\alias{multilevel.invar}
\title{Cross-Level Measurement Invariance Evaluation}
\usage{
multilevel.invar(..., data = NULL, cluster, model = NULL, rescov = NULL,
                 invar = c("config", "metric", "scalar"), fix.resid = NULL,
                 ident = c("marker", "var", "effect"),
                 estimator = c("ML", "MLR"), optim.method = c("nlminb", "em"),
                 missing = c("listwise", "fiml"),
                 print = c("all", "summary", "coverage", "descript", "fit",
                           "est", "modind", "resid"),
                 print.fit = c("all", "standard", "scaled", "robust"),
                 mod.minval = 6.63, resid.minval = 0.1, digits = 3, p.digits = 3,
                 as.na = NULL, write = NULL, append = TRUE, check = TRUE,
                 output = TRUE)
}
\arguments{
\item{...}{a matrix or data frame. If \code{model} is \code{NULL},
multilevel confirmatory factor analysis based on a
measurement model with one factor at the Within and Between
level comprising all variables in the matrix or data frame
is conducted to evaluate cross-level measurement invariance.
Note that the cluster variable specified in \code{cluster}
is excluded from \code{x} when specifying the argument
\code{cluster} using the variable name of the cluster
variable. If \code{model} is specified, the matrix or data
frame needs to contain all variables used in the \code{model}
argument. Alternatively, an expression indicating the
variable names in \code{data} e.g.,
\code{multilevel.invar(x1, x2, x3, data = dat)}. Note that
the operators \code{.}, \code{+}, \code{-}, \code{~},
\code{:}, \code{::}, and \code{!} can also be used to
select variables, see 'Details' in the \code{\link{df.subset}}
function.}

\item{data}{a data frame when specifying one or more variables in the
argument \code{...}. Note that the argument is \code{NULL}
when specifying a matrix or data frame for the argument \code{...}.}

\item{cluster}{either a character string indicating the variable name of
the cluster variable in \code{...} or \code{data}, or a
vector representing the nested grouping structure (i.e.,
group or cluster variable).}

\item{model}{a character vector specifying the same factor structure
with one factor at the Within and Between Level, or a list
of character vectors for specifying the same measurement
model with more than one factor at the Within and Between
Level, e.g.,\code{model = c("x1", "x2", "x3", "x4")} for
specifying a measurement model with one factor labeled
\code{wf} at the Within level and a measurement model with
one factor labeled \code{bf} at the Between level each
comprising four indicators, or \code{model = list(factor1 = c("x1", "x2", "x3", "x4"),
factor2 = c("x5", "x6", "x7", "x8"))} for specifying a
measurement model with two latent factors labeled \code{wfactor1}
and \code{wfactor2} at the Within level and a measurement
model with two latent factors labeled \code{bfactor1} and
\code{bfactor2} at the Between level each comprising four
indicators. Note that the name of each list element is used
to label factors, where prefixes \code{w} and \code{b} are
added the labels to distinguish factor labels at the Within
and Between level, i.e., all list elements need to be named,
otherwise factors are labeled with \code{"wf1", "wf2", "wf3"}
for labels at the Within level and \code{"bf1", "bf2", "bf3"}
for labels at the Between level and so on.}

\item{rescov}{a character vector or a list of character vectors for specifying
residual covariances at the Within level, e.g. \code{rescov = c("x1", "x2")}
for specifying a residual covariance between indicators \code{x1}
and \code{x2} at the Within level or \code{rescov = list(c("x1", "x2"), c("x3", "x4"))}
for specifying residual covariances between indicators \code{x1}
and \code{x2}, and indicators \code{x3} and \code{x4} at
the Within level. Note that residual covariances at the
Between level can only be specified by using the arguments
\code{model.w}, \code{model.b}, and \code{model.b}.}

\item{invar}{a character string indicating the level of measurement invariance
to be evaluated, i.e., \code{config} to evaluate configural
measurement invariance (i.e., same factor structure across
levels), \code{metric} (default) to evaluate configural and
metric measurement invariance (i.e., equal factor loadings
across level), and \code{scalar} to evaluate configural,
metric and scalar measurement invariance (i.e., all residual
variances at the Between level equal zero).}

\item{fix.resid}{a character vector for specifying residual variances to be
fixed at 0 at the Between level for the configural and metric
invariance model, e.g., \code{fix.resid = c("x1", "x3")}
to fix residual variances of indicators \code{x1} and \code{x2}
at the Between level at 0. Note that it is also possible
to specify \code{fix.resid = "all"} which fixes all residual
variances at the Between level at 0 in line with the strong
factorial measurement invariance assumption across cluster.}

\item{ident}{a character string indicating the method used for identifying
and scaling latent variables, i.e., \code{"marker"} for the
marker variable method fixing the first factor loading of
each latent variable to 1, \code{"var"} for the fixed variance
method fixing the variance of each latent variable to 1,
or \code{"effect"} for the effects-coding method using equality
constraints so that the average of the factor loading for
each latent variable equals 1.}

\item{estimator}{a character string indicating the estimator to be used:
\code{"ML"} for maximum likelihood with conventional standard
errors and \code{"MLR"} (default) for maximum likelihood
with Huber-White robust standard errors and a scaled test
statistic that is asymptotically equal to the Yuan-Bentler
test statistic. Note that by default, full information maximum
likelihood (FIML) method is used to deal with missing data
when using \code{"ML"} (\code{missing = "fiml"}), whereas
incomplete cases are removed listwise (i.e., \code{missing = "listwise"})
when using \code{"MLR"}.}

\item{optim.method}{a character string indicating the optimizer, i.e., \code{"nlminb"}
(default) for the unconstrained and bounds-constrained
quasi-Newton method optimizer and \code{"em"} for the
Expectation Maximization (EM) algorithm.}

\item{missing}{a character string indicating how to deal with missing data,
i.e., \code{"listwise"} (default) for listwise deletion or
\code{"fiml"} for full information maximum likelihood (FIML)
method. Note that FIML method is only available when
\code{estimator = "ML"}, that it takes longer to estimate
the model  using FIML, and that FIML is prone to convergence
issues which might be resolved by switching to listwise deletion.}

\item{print}{a character string or character vector indicating which
results to show on the console, i.e. \code{"all"} for all
results, \code{"summary"} for a summary of the specification
of the estimation method and missing data handling in lavaan,
\code{"coverage"} for the variance-covariance coverage of
the data, \code{"descript"} for descriptive statistics,
\code{"fit"} for model fit and  model comparison, \code{"est"}
for parameter estimates, and \code{"modind"} for modification
indices. By default, a summary of the specification and model fit
and model comparison are printed.}

\item{print.fit}{a character string or character vector indicating which
version of the CFI, TLI, and RMSEA to show on the console,
i.e., \code{"all"} for all versions of the CFI, TLI, and
RMSEA, \code{"standard"} (default when \code{estimator = "ML"})
for fit indices without any non-normality correction,
\code{"scaled"} for population-corrected robust fit indices
with ad hoc non-normality correction, and \code{robust}
(default when \code{estimator = "MLR"}) for sample-corrected
robust fit indices based on formula provided by Li and Bentler
(2006) and Brosseau-Liard and Savalei (2014).}

\item{mod.minval}{numeric value to filter modification indices and only
show modifications with a modification index value equal
or higher than this minimum value. By default, modification
indices equal or higher 6.63 are printed. Note that a
modification index value of 6.63 is equivalent to a
significance level of \eqn{\alpha = .01}.}

\item{resid.minval}{numeric value indicating the minimum absolute residual
correlation coefficients and standardized means to
highlight in boldface. By default, absolute residual
correlation coefficients and standardized means equal
or higher 0.1 are highlighted. Note that highlighting
can be disabled by setting the minimum value to 1.}

\item{digits}{an integer value indicating the number of decimal places
to be used for displaying results. Note that information
criteria and chi-square test statistic is printed with
\code{digits} minus 1 decimal places.}

\item{p.digits}{an integer value indicating the number of decimal places
to be used for displaying the \emph{p}-value.}

\item{as.na}{a numeric vector indicating user-defined missing values,
i.e. these values are converted to \code{NA} before conducting
the analysis. Note that \code{as.na()} function is only
applied to \code{x} but not to \code{cluster}.}

\item{write}{a character string naming a file for writing the output into
either a text file with file extension \code{".txt"} (e.g.,
\code{"Output.txt"}) or Excel file with file extension
\code{".xlsx"}  (e.g., \code{"Output.xlsx"}). If the file
name does not contain any file extension, an Excel file will
be written.}

\item{append}{logical: if \code{TRUE} (default), output will be appended
to an existing text file with extension \code{.txt} specified
in \code{write}, if \code{FALSE} existing text file will be
overwritten.}

\item{check}{logical: if \code{TRUE} (default), argument specification, convergence
and model identification is checked.}

\item{output}{logical: if \code{TRUE} (default), output is shown.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:

\item{\code{call}}{function call}
\item{\code{type}}{type of analysis}
\item{\code{data}}{matrix or data frame specified in \code{x}}
\item{\code{args}}{specification of function arguments}
\item{\code{model}}{list with specified model for the configural, metric, and
                    scalar invariance model}
\item{\code{model.fit}}{list with fitted lavaan object of the configural, metric,
                        and scalar invariance model}
\item{\code{check}}{list with the results of the convergence and model identification
                    check for the configural, metric, and scalar invariance model}
\item{\code{result}}{list with result tables, i.e., \code{summary} for the
                     summary of the specification of the estimation method and
                     missing data handling in lavaan, \code{coverage} for the
                     variance-covariance coverage of the data, \code{descript}
                     for descriptive statistics, \code{fit} for a list with
                     model fit based on standard, scaled, and robust fit indices,
                     \code{est} for a list with parameter estimates for the
                     configural, metric, and scalar invariance model, and
                     \code{modind} for the list with modification indices for
                     the configural, metric, and scalar invariance model}
}
\description{
This function is a wrapper function for evaluating configural, metric, and
scalar cross-level measurement invariance using multilevel confirmatory factor
analysis with continuous indicators by calling the \code{cfa} function in the
R package \pkg{lavaan}.
}
\note{
The function uses the functions \code{lavTestLRT} provided in the R package
\pkg{lavaan} by Yves Rosseel (2012).
}
\examples{
\dontrun{
# Load data set "Demo.twolevel" in the lavaan package
data("Demo.twolevel", package = "lavaan")

#----------------------------------------------------------------------------
# Cluster variable specification

# Example 1a: Cluster variable 'cluster' in 'x'
multilevel.invar(Demo.twolevel[,c("y1", "y2", "y3", "y4", "cluster")], cluster = "cluster")

# Example 1b: Cluster variable 'cluster' not in 'x'
multilevel.invar(Demo.twolevel[,c("y1", "y2", "y3", "y4")], cluster = Demo.twolevel$cluster)

# Example 1c: Alternative specification using the 'data' argument
multilevel.invar(y1:y4, data = Demo.twolevel, cluster = "cluster")

#----------------------------------------------------------------------------
# Model specification using 'x' for a one-factor model

#..........
# Level of measurement invariance

# Example 2a: Configural invariance
multilevel.invar(Demo.twolevel[,c("y1", "y2", "y3", "y4")],
                 cluster = Demo.twolevel$cluster, invar = "config")

# Example 2b: Metric invariance
multilevel.invar(Demo.twolevel[,c("y1", "y2", "y3", "y4")],
                 cluster = Demo.twolevel$cluster, invar = "metric")

# Example 2c: Scalar invariance
multilevel.invar(Demo.twolevel[,c("y1", "y2", "y3", "y4")],
                 cluster = Demo.twolevel$cluster, invar = "scalar")

#..........
# Residual covariance at the Within level and residual variance at the Between level

# Example 3a: Residual covariance between "y3" and "y4" at the Within level
multilevel.invar(Demo.twolevel[,c("y1", "y2", "y3", "y4")],
                 cluster = Demo.twolevel$cluster, rescov = c("y3", "y4"))

# Example 3b: Residual variances of 'y1' at the Between level fixed at 0
multilevel.invar(Demo.twolevel[,c("y1", "y2", "y3", "y4")],
                 cluster = Demo.twolevel$cluster, fix.resid = "y1")

#..........
# Example 4: Print all results
multilevel.invar(Demo.twolevel[,c("y1", "y2", "y3", "y4")],
                 cluster = Demo.twolevel$cluster, print = "all")

#..........
# Example 5: lavaan model and summary of the estimated model
mod <- multilevel.invar(Demo.twolevel[,c("y1", "y2", "y3", "y4")],
                        cluster = Demo.twolevel$cluster, output = FALSE)

# lavaan syntax of the metric invariance model
mod$model$metric

# Fitted lavaan object of the metric invariance model
lavaan::summary(mod$model.fit$metric, standardized = TRUE, fit.measures = TRUE)

#----------------------------------------------------------------------------
# Model specification using 'model' for one or multiple factor model

# Example 6a: One-factor model
multilevel.invar(Demo.twolevel, cluster = "cluster", model = c("y1", "y2", "y3", "y4"))

# Example 6b:  Two-factor model
multilevel.invar(Demo.twolevel, cluster = "cluster",
                 model = list(c("y1", "y2", "y3"), c("y4", "y5", "y6")))

#----------------------------------------------------------------------------
# Write results

# Example 7a: Write results into an Excel file
multilevel.invar(Demo.twolevel[,c("y1", "y2", "y3", "y4")],
                 cluster = Demo.twolevel$cluster, print = "all",
                 write = "Multilevel_Invariance.txt")

# Example 7b:  Write results into an Excel file
multilevel.invar(Demo.twolevel[,c("y1", "y2", "y3", "y4")],
                 cluster = Demo.twolevel$cluster, print = "all",
                 write = "Multilevel_Invariance.xlsx")

# Assign results into an object and write results into an Excel file
mod <- multilevel.invar(Demo.twolevel[,c("y1", "y2", "y3", "y4")],
                        cluster = Demo.twolevel$cluster, print = "all",
                        output = FALSE)

# Write results into an Excel file
write.result(mod, "Multilevel_Invariance.xlsx")
}
}
\references{
Rosseel, Y. (2012). lavaan: An R Package for Structural Equation Modeling.
\emph{Journal of Statistical Software, 48}, 1-36. https://doi.org/10.18637/jss.v048.i02
}
\seealso{
\code{\link{multilevel.cfa}}, \code{\link{multilevel.fit}}, \code{\link{multilevel.omega}},
\code{\link{multilevel.cor}}, \code{\link{multilevel.descript}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
