# WARNING - Generated by {fusen} from dev/flat_teaching.Rmd: do not edit by hand

#' Export List with Advanced Directory Management
#'
#' @description
#' The `export_list` function exports a list of `data.frame`, `data.table`, or compatible data structures 
#' with sophisticated directory handling, flexible naming, and multiple file format support.
#'
#' @param split_dt A `list` of `data.frame`, `data.table`, or compatible data structures 
#'   to be exported.
#' @param export_path Base directory path for file export. Defaults to a temporary directory 
#'   created by `tempdir()`.
#' @param file_type File export format, either `"txt"` (tab-separated) or `"csv"`. 
#'   Defaults to `"txt"`.
#'
#' @details
#' Comprehensive List Export Features:
#' \itemize{
#'   \item Advanced nested directory structure support based on list element names
#'   \item Intelligent handling of unnamed list elements
#'   \item Automatic conversion to `data.table` for consistent export
#'   \item Hierarchical directory creation with nested path names
#'   \item Multi-format file export with intelligent separator selection
#'   \item Robust error handling and input validation
#' }
#'
#' File Export Capabilities:
#' \itemize{
#'   \item Supports `"txt"` (tab-separated) and `"csv"` formats
#'   \item Intelligent file naming based on list element names
#'   \item Handles complex nested directory structures
#'   \item Efficient file writing using `data.table::fwrite()`
#' }
#'
#' @return 
#' An `integer` representing the total number of files exported successfully.
#'
#' @note
#' Key Capabilities:
#' \itemize{
#'   \item Flexible list naming and directory management
#'   \item Comprehensive support for `data.frame` and `data.table` inputs
#'   \item Intelligent default naming for unnamed elements
#'   \item High-performance file writing mechanism
#' }
#'
#' @importFrom data.table fwrite as.data.table
#' @importFrom utils head tail
#' @export
#' @examples
#' # Example: Export split data to files
#'
#' # Step 1: Create split data structure
#' dt_split <- w2l_split(
#'   data = iris,              # Input iris dataset
#'   cols2l = 1:2,             # Columns to be split
#'   by = "Species"            # Grouping variable
#' )
#'
#' # Step 2: Export split data to files
#' export_list(
#'   split_dt = dt_split       # Input list of data.tables
#' )
#' # Returns the number of files created
#' # Files are saved in tempdir() with .txt extension
#'
#' # Check exported files
#' list.files(
#'   path = tempdir(),         # Default export directory
#'   pattern = "txt",          # File type pattern to search
#'   recursive = TRUE          # Search in subdirectories
#' )
#'
#' # Clean up exported files
#' files <- list.files(
#'   path = tempdir(),         # Default export directory
#'   pattern = "txt",          # File type pattern to search
#'   recursive = TRUE,         # Search in subdirectories
#'   full.names = TRUE         # Return full file paths
#' )
#' file.remove(files)          # Remove all exported files

export_list <- function(split_dt, export_path = tempdir(), file_type = "txt") {
  # Input validation
  if (!is.list(split_dt)) {
    stop("split_dt must be a list of data.tables/data.frames")
  }

  file_type <- match.arg(file_type, c("txt", "csv"))

  # Define separator mapping for file types
  sep_map <- c(txt = "\t", csv = ",")

  # Create base export directory if it doesn't exist
  dir.create(export_path, recursive = TRUE, showWarnings = FALSE)

  # Initialize counter
  count <- 0L

  # Process each element in the list
  exported_files <- vapply(seq_along(split_dt), function(i) {
    current_data <- split_dt[[i]]
    current_name <- names(split_dt)[i]

    current_name <- if (is.null(current_name) || current_name == "") {
      paste0("split_", i)
    } else {
      current_name
    }

    # Handle path components
    if (grepl("/", current_name)) {
      path_components <- strsplit(current_name, "/")[[1]]
      file_name <- tail(path_components, 1)
      sub_dirs <- head(path_components, -1)

      full_path <- file.path(export_path, paste(sub_dirs, collapse = "/"))
      dir.create(full_path, recursive = TRUE, showWarnings = FALSE)
    } else {
      file_name <- current_name
      full_path <- export_path
    }

    file_path <- file.path(full_path, paste0(file_name, ".", file_type))

    if (!data.table::is.data.table(current_data)) {
      current_data <- data.table::as.data.table(current_data)
    }

    data.table::fwrite(current_data,
                       file = file_path,
                       sep = sep_map[file_type],
                       quote = TRUE)

    # Increment counter
    count <<- count + 1L

    file_path
  }, character(1))

  names(exported_files) <- names(split_dt)

  # Return count
  return(count)

  invisible(exported_files)
}
