% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mdgc.R
\name{mdgc}
\alias{mdgc}
\title{Perform Model Estimation and Imputation}
\usage{
mdgc(
  dat,
  lr = 0.001,
  maxit = 25L,
  batch_size = NULL,
  rel_eps = 0.001,
  method = c("svrg", "adam", "aug_Lagran"),
  seed = 1L,
  epsilon = 1e-08,
  beta_1 = 0.9,
  beta_2 = 0.999,
  n_threads = 1L,
  do_reorder = TRUE,
  abs_eps = -1,
  maxpts = 10000L,
  minvls = 100L,
  verbose = FALSE,
  irel_eps = rel_eps,
  imaxit = maxpts,
  iabs_eps = abs_eps,
  iminvls = 1000L,
  start_val = NULL,
  decay = 0.98,
  conv_crit = 1e-05,
  use_aprx = FALSE
)
}
\arguments{
\item{dat}{\code{\link{data.frame}} with continuous, multinomial, ordinal, and binary
variables.}

\item{lr}{learning rate.}

\item{maxit}{maximum number of iteration.}

\item{batch_size}{number of observations in each batch.}

\item{rel_eps}{relative error for each marginal likelihood factor.}

\item{method}{estimation method to use. Can be \code{"svrg"}, \code{"adam"},
or \code{"aug_Lagran"}.}

\item{seed}{fixed seed to use. Use \code{NULL} if the seed should not be
fixed.}

\item{epsilon}{ADAM parameters.}

\item{beta_1}{ADAM parameters.}

\item{beta_2}{ADAM parameters.}

\item{n_threads}{number of threads to use.}

\item{do_reorder}{logical for whether to use a heuristic variable
reordering. \code{TRUE} is likely the best option.}

\item{abs_eps}{absolute convergence threshold for each marginal likelihood
factor.}

\item{maxpts}{maximum number of samples to draw for each marginal likelihood
term.}

\item{minvls}{minimum number of samples.}

\item{verbose}{logical for whether to print output during the estimation.}

\item{irel_eps}{relative error for each term in the imputation.}

\item{imaxit}{maximum number of samples to draw in the imputation.}

\item{iabs_eps}{absolute convergence threshold for each term in the imputation.}

\item{iminvls}{minimum number of samples in the imputation.}

\item{start_val}{starting value for the covariance matrix. Use
\code{NULL} if unspecified.}

\item{decay}{the learning rate used by SVRG is given by \code{lr * decay^iteration_number}.}

\item{conv_crit}{relative convergence threshold.}

\item{use_aprx}{logical for whether to use an approximation of
\code{\link{pnorm}} and \code{\link{qnorm}}. This may yield a
noticeable reduction in the computation time.}
}
\value{
A list with the following entries:

\item{ximp}{\code{\link{data.frame}} with the observed and imputed values.}
\item{imputed}{output from \code{\link{mdgc_impute}}.}
\item{vcov}{the estimated covariance matrix.}
\item{mea}{the estimated non-zero mean terms.}

Additional elements may be present depending on the chosen \code{method}.
See \code{\link{mdgc_fit}}.
}
\description{
A convenience function to perform model estimation and imputation in one
call. The learning rate is likely model specific and should be altered.
See \code{\link{mdgc_fit}}.

See the README at \url{https://github.com/boennecd/mdgc} for examples.
}
\details{
It is important that the input for \code{data} has the appropriate types and
classes. See \code{\link{get_mdgc}}.
}
\examples{
\donttest{
if(require(catdata)){
  data(retinopathy)

  # prepare data and save true data set
  retinopathy$RET <- as.ordered(retinopathy$RET)
  retinopathy$SM <- as.logical(retinopathy$SM)

  # randomly mask data
  set.seed(28325145)
  truth <- retinopathy
  for(i in seq_along(retinopathy))
    retinopathy[[i]][runif(NROW(retinopathy)) < .3] <- NA

  cat("\nMasked data:\n")
  print(head(retinopathy, 10))
  cat("\n")

  # impute data
  impu <- mdgc(retinopathy, lr = 1e-3, maxit = 25L, batch_size = 25L,
               rel_eps = 1e-3, maxpts = 5000L, verbose = TRUE,
               n_threads = 1L, method = "svrg")

  # show correlation matrix
  cat("\nEstimated correlation matrix\n")
  print(impu$vcov)

  # compare imputed and true values
  cat("\nObserved;\n")
  print(head(retinopathy, 10))
  cat("\nImputed values:\n")
  print(head(impu$ximp, 10))
  cat("\nTruth:\n")
  print(head(truth, 10))

  # using augmented Lagrangian method
  cat("\n")
  impu_aug <- mdgc(retinopathy, maxit = 25L, rel_eps = 1e-3,
                   maxpts = 5000L, verbose = TRUE,
                   n_threads = 1L, method = "aug_Lagran")

  # compare the log-likelihood estimate
  obj <- get_mdgc_log_ml(retinopathy)
  cat(sprintf(
    "Maximum log likelihood with SVRG vs. augmented Lagrangian:\n  \%.2f vs. \%.2f\n",
    mdgc_log_ml(obj, vcov = impu    $vcov, mea = impu    $mea, rel_eps = 1e-3),
    mdgc_log_ml(obj, vcov = impu_aug$vcov, mea = impu_aug$mea, rel_eps = 1e-3)))

  # show correlation matrix
  cat("\nEstimated correlation matrix (augmented Lagrangian)\n")
  print(impu_aug$vcov)

  cat("\nImputed values (augmented Lagrangian):\n")
  print(head(impu_aug$ximp, 10))
}
}

}
\references{
Kingma, D.P., & Ba, J. (2015). \emph{Adam: A Method for Stochastic Optimization}. abs/1412.6980.

Johnson, R., & Zhang, T. (2013). \emph{Accelerating stochastic gradient descent using predictive variance reduction}. In Advances in neural information processing systems.
}
\seealso{
\code{\link{get_mdgc}}, \code{\link{mdgc_start_value}},
\code{\link{get_mdgc_log_ml}}, \code{\link{mdgc_fit}},
\code{\link{mdgc_impute}}
}
