% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/est_marg_liks.R
\name{est_marg_liks}
\alias{est_marg_liks}
\title{Estimate the marginal likelihoods for one or more inference models}
\usage{
est_marg_liks(
  fasta_filename,
  inference_models = list(beautier::create_inference_model(mcmc =
    beautier::create_ns_mcmc())),
  beast2_optionses = rep(list(beastier::create_mcbette_beast2_options()), times =
    length(inference_models)),
  verbose = FALSE,
  os = rappdirs::app_dir()$os
)
}
\arguments{
\item{fasta_filename}{name of the FASTA file}

\item{inference_models}{a list of one or more inference models,
as can be created by \link[beautier]{create_inference_model}}

\item{beast2_optionses}{list of one or more \code{beast2_options}
structures,
as can be created by \link[beastier]{create_mcbette_beast2_options}.
Use of reduplicated plural to achieve difference with
\code{beast2_options}}

\item{verbose}{if TRUE show debug output}

\item{os}{name of the operating system,
must be \code{unix} (Linux, Mac) or \code{win} (Windows)}
}
\value{
a \link{data.frame} showing the estimated marginal likelihoods
(and its estimated error) per combination of models. Columns are:
\itemize{
  \item \code{site_model_name}: name of the site model
  \item \code{clock_model_name}: name of the clock model
  \item \code{tree_prior_name}: name of the tree prior
  \item \code{marg_log_lik}: estimated marginal (natural) log likelihood
  \item \code{marg_log_lik_sd}: estimated error of \code{marg_log_lik}
  \item \code{weight}: relative model weight, a value from 1.0 (all
    evidence is in favor of this model combination) to 0.0 (no
    evidence in favor of this model combination)
  \item \code{ess}: effective sample size of the marginal likelihood
    estimation
}
}
\description{
Estimate the marginal likelihoods (aka evidence)
for one or more inference models, based on a single alignment.
Also, the marginal likelihoods are compared, resulting in a
relative weight for each model, where a relative weight of a model
close to \code{1.0} means that that model is way likelier than
the others.
}
\details{
In the process, multiple (temporary) files are created (where
\code{[x]} denotes the index in a list)

\itemize{
  \item \code{beast2_optionses[x]$input_filename}
    path to the the BEAST2 XML input file
  \item \code{beast2_optionses[x]$output_state_filename}
    path to the BEAST2 XML state file
  \item \code{inference_models[x]$mcmc$tracelog$filename}
    path to the BEAST2 trace file with parameter estimates
  \item \code{inference_models[x]$mcmc$treelog$filename}
    path to the BEAST2 \code{trees} file with the posterior trees
  \item \code{inference_models[x]$mcmc$screenlog$filename}
    path to the BEAST2 screen output file
}

These file can be deleted manually by \link[babette]{bbt_delete_temp_files},
else these will be deleted automatically by the operating system.
}
\examples{
if (can_run_mcbette()) {

  # Use an example FASTA file
  fasta_filename <- system.file("extdata", "simple.fas", package = "mcbette")

  # Create two inference models
  inference_model_1 <- beautier::create_ns_inference_model(
    site_model = beautier::create_jc69_site_model()
  )
  inference_model_2 <- beautier::create_ns_inference_model(
    site_model = beautier::create_hky_site_model()
  )

  # Shorten the run, by doing a short (dirty, unreliable) MCMC
  inference_model_1$mcmc <- beautier::create_test_ns_mcmc()
  inference_model_2$mcmc <- beautier::create_test_ns_mcmc()

  # Combine the inference models
  inference_models <- list(inference_model_1, inference_model_2)

  # Create the BEAST2 options, that will write the output
  # to different (temporary) filanems
  beast2_options_1 <- beastier::create_mcbette_beast2_options()
  beast2_options_2 <- beastier::create_mcbette_beast2_options()

  # Combine the two BEAST2 options sets,
  # use reduplicated plural
  beast2_optionses <- list(beast2_options_1, beast2_options_2)

  # Compare the models
  marg_liks <- est_marg_liks(
    fasta_filename,
    inference_models = inference_models,
    beast2_optionses = beast2_optionses
  )

  # Interpret the results
  interpret_marg_lik_estimates(marg_liks)

  beastier::remove_beaustier_folders()
  beastier::check_empty_beaustier_folders()
}
}
\seealso{
\itemize{
  \item \link{can_run_mcbette}: see if 'mcbette' can run
  \item \link{est_marg_liks}: estimate multiple marginal likelihood of a
    single inference mode
}
}
\author{
Richèl J.C. Bilderbeek
}
