% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cliBIOME.R
\name{cliBIOMEPoints}
\alias{cliBIOMEPoints}
\title{Vegetation Classifier Using the BIOME Model}
\usage{
cliBIOMEPoints(
  temp,
  prec,
  bsdf,
  lat,
  elv,
  year = 2000,
  MSMC = 150,
  aprchTEMP = c("hip", "tsi", "const"),
  aprchPREC = c("tsi", "hip", "const"),
  aprchBSDF = c("hip", "const"),
  dvTEMP = rep(0.7, 12),
  dvPREC = rep(0.7, 12),
  verbose = FALSE
)
}
\arguments{
\item{temp}{'numeric' R object with one-year time series of monthly mean air temperature (in °C)}

\item{prec}{'numeric' R object with one-year time series of monthly precipitation sum (in mm)}

\item{bsdf}{'numeric' R object with one-year time series of monthly mean relative sunshine duration (dimensionless)}

\item{lat}{'numeric' vector with the latitude coordinates (in decimal degrees)}

\item{elv}{'numeric' vector with the elevation values (in meters above sea level)}

\item{year}{'numeric' vector with values of the year (using astronomical year numbering)}

\item{MSMC}{'numeric' vector with values of the maximum soil moisture capacity (aka 'bucket size') (in mm)}

\item{aprchTEMP}{'character' vector of length 1 that indicates the scheme used to generate daily values of the
daily mean air temperature for a specific year. Valid values are as follows: \cr
(a) \code{'hip'} -
this scheme applies the mean-preserving 'harmonic' interpolation method of Epstein (1991) to the values of
monthly mean air temperature in order to generate daily values; \cr
(b) \code{'tsi'} -
this scheme uses an iterative interpolation technique (Lüdeke et al. 1994) to time series of the monthly mean
air temperature, in order to generate a synthetic time series of the selected meteorological variable at a
temporal resolution that is higher than the daily scale; finally, this synthetic time series is upscaled to a
daily resolution; \cr
(c) \code{'const'} -
this scheme is assumed that values of the daily mean air temperature are constant within each month.}

\item{aprchPREC}{'character' vector of length 1 that indicates the scheme to generate daily values of the
daily precipitation sum. Valid values are as follows: \cr
(a) \code{'tsi'} -
this scheme uses an iterative interpolation technique (Lüdeke et al. 1994) to time series of the monthly mean
precipitation intensity, in order to generate a synthetic time series of the selected meteorological variable
at a temporal resolution that is higher than the daily scale; finally, this synthetic time series is upscaled
to a daily resolution; \cr
(b) \code{'hip'} -
this scheme applies the mean-preserving 'harmonic' interpolation method of Epstein (1991) to the values of
monthly mean precipitation intensity in order to generate daily values; \cr
(c) \code{'const'} -
this scheme is assumed that values of the daily precipitation sum are constant within each month (the monthly
precipitation sum is divided equally across each day of the month).}

\item{aprchBSDF}{'character' vector of length 1 that indicates the scheme used to generate daily values of the
daily fractional sunshine duration for a specific year. Valid values are as follows: \cr
(a) \code{'hip'} -
this scheme applies the mean-preserving 'harmonic' interpolation method of Epstein (1991) to the values of
monthly mean relative sunshine duration in order to generate daily values; \cr
(b) \code{'const'} -
this scheme is assumed that values of the daily relative sunshine duration are constant within each month.}

\item{dvTEMP}{'numeric' vector of length 12 with monthly values of the damping variable for the air temperature
data.}

\item{dvPREC}{'numeric' vector of length 12 with monthly values of the damping variable for the precipitation data.}

\item{verbose}{'logical' scalar that indicates whether or not values of the bioclimatic indices used should be
added to the output.}
}
\value{
Depending on the setting, a data frame with one or more columns where the biome types are stored in the
    last (character) column, while the additional columns contain the values of bioclimatic indices used. The
    abbreviations of biome types can be found in the data frame \code{\link[macroBiome]{vegClsNumCodes}}. If
    \code{verbose = FALSE}, the return object is a one-column data frame with the biome types.
}
\description{
Calculates the values of bioclimatic indices used in the BIOME model developed by Prentice et al.
    (1992), and designates the biome type using these values, for a given geographical location (latitude and
    elevation) and year/epoch, by using the monthly time series of temperature, precipitation and relative
    sunshine duration.
}
\details{
To classify vegetation, the BIOME model developed by Prentice et al. (1992) uses the values of the
    following 5 bioclimatic indices:

    \itemize{
      \item{\code{tc}: Mean Temperature of the Coldest Month (in °C)}
      \item{\code{tw}: Mean Temperature of the Warmest Month (in °C)}
      \item{\code{gdd5}: Growing Degree-Days on 5°C base (in °C day)}
      \item{\code{gdd0}: Growing Degree-Days on 0°C base (in °C day)}
      \item{\code{ptc}: Priestley–Taylor Coefficient (dimensionless)}
    }

    For details about calculating bioclimatic indices, see the function
    \code{\link[macroBiome]{cliBioCliIdxPoints}}. The Priestley–Taylor coefficient (\code{'ptc'}, dimensionless)
    is exceptional because its computation requires a simulation of evapotranspiration at daily time step via the
    implementation of the SPLASH algorithm (Davis et al. 2017) (see
    \code{\link[macroBiome]{dlyEngWtrFluxPoints}}). The application of the SPLASH algorithm requires, among other
    things, one-year time series of the climate variables at daily scale, which are generated from average monthly
    values using the function \code{\link[macroBiome]{dlyWeaGenPoints}}. \cr
    The designation of the biome type is implemented as a two-step procedure. First, the presence of each plant
    functional type (PFT) is estimated under the given climatic conditions. Subsequently, the biome type is
    designated by combining PFTs occurring at the maximal dominance level with each other (see Table 5 in Prentice
    et al. (1992)). Each PFT is described by constraints of bioclimatic variables associated with their climatic
    tolerances and requirements (see Table 1 in Prentice et al. (1992)). In the initial version of the BIOME
    model, a total of 17 biome types are distinguished (see \code{\link[macroBiome]{vegClsNumCodes}}).
}
\note{
As with any function with a point mode, a set of basic input data is defined here. In this case, they are as
    follows: \code{'temp'} (one-year time series of monthly mean air temperature), \code{'prec'} (one-year time
    series of monthly precipitation sum), and \code{'bsdf'} (one-year time series of monthly mean relative
    sunshine duration). The objects \code{'temp'}, \code{'prec'} and \code{'bsdf'} must be either vectors of
    length 12 or 12-column matrices. The first dimensions of these matrices have to be the same length. The
    function automatically converts vectors into single-row matrices during the error handling, and then uses these
    matrices. The first dimensions of these matrices determines the number of rows in the result matrix. In the
    case of arguments that do not affect the course of the calculation procedure or the structure of the return
    object, scalar values (i.e., 'numeric' vector of length 1) may also be allowed. In this case, they are as
    follows: \code{'lat'} (latitude coordinates in decimal degrees), \code{'elv'} (elevation in meters above sea
    level), \code{'year'} (year using astronomical year numbering), and \code{'MSMC'} ('bucket size' in mm). These
    scalars are converted to vectors by the function during the error handling, and these vectors are applied in
    the further calculations. If these data are stored in vectors of length at least 2, their length must be the
    same size of first dimension of the matrices containing the basic data.
}
\examples{
\donttest{
# Loading mandatory data for the Example 'Points'
data(inp_exPoints)

# Designate the biome type (using the related biolcimatic indices), with default settings,
# at a grid cell near Szeged, Hungary (46.3N, 20.2E) (for the normal period 1981-2010)
with(inp_exPoints, {
year <- trunc(mean(seq(1981, 2010)))
BIOME <- cliBIOMEPoints(colMeans(temp), colMeans(prec), colMeans(bsdf), lat, elv,
    year = year, verbose = TRUE)
numCode <- which(sapply(vegClsNumCodes$Code.BIOME, identical, BIOME[, "vegCls"]))
cbind(BIOME[,-c(6)], vegClsNumCodes[numCode, c("Name.BIOME", "Code.BIOME")])
})
}

}
\references{
\cite{Davis TW, Prentice IC, Stocker BD, Thomas RT, Whitley RJ, Wang H, Evans BJ, Gallego-Sala AV, Sykes MT,
    Cramer W (2017) Simple process-led algorithms for simulating habitats (SPLASH v.1.0): robust indices of
    radiation, evapotranspiration and plant-available moisture. Geosci Model Dev 10(2):689–708.
    \doi{10.5194/gmd-10-689-2017}}

\cite{Epstein ES (1991) On Obtaining Daily Climatological Values from Monthly Means. J Clim 4(3):365–368.
    \doi{10.1175/1520-0442(1991)004<0365:OODCVF>2.0.CO;2}}

\cite{Lüdeke MKB, Badeck FW, Otto RD, Häger C, Dönges S, Kindermann J, Würth G, Lang T, Jäkel U, Klaudius A,
    Ramge P, Habermehl S, Kohlmaier GH (1994) The Frankfurt Biosphere Model: A global process-oriented model of
    seasonal and long-term CO2 exchange between terrestrial ecosystems and the atmosphere. I. Model description
    and illustrative results for cold deciduous and boreal forests. Clim Res 4(2):143-166. \doi{10.3354/cr004143}}

\cite{Prentice IC, Cramer W, Harrison SP, Leemans R, Monserud RA, Solomon AM (1992) A Global Biome Model Based on
    Plant Physiology and Dominance, Soil Properties and Climate. J Biogeogr 19(2):117–134. \doi{10.2307/2845499}}
}
