% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lpbwdensity.R
\name{lpbwdensity}
\alias{lpbwdensity}
\title{Data-driven Bandwidth Selection for Local Polynomial Density Estimators}
\usage{
lpbwdensity(data, grid = NULL, p = NULL, v = NULL,
  kernel = c("triangular", "uniform", "epanechnikov"),
  bwselect = c("mse-dpi", "imse-dpi", "mse-rot", "imse-rot"),
  massPoints = TRUE, stdVar = TRUE, regularize = TRUE,
  nLocalMin = NULL, nUniqueMin = NULL, Cweights = NULL,
  Pweights = NULL)
}
\arguments{
\item{data}{Numeric vector or one dimensional matrix/data frame, the raw data.}

\item{grid}{Numeric, specifies the grid of evaluation points. When set to default, grid points
will be chosen as 0.05-0.95 percentiles of the data, with a step size of 0.05.}

\item{p}{Nonnegative integer, specifies the order of the local polynomial used to construct point
estimates. (Default is \code{2}.)}

\item{v}{Nonnegative integer, specifies the derivative of the distribution function to be estimated. \code{0} for
the distribution function, \code{1} (default) for the density funtion, etc.}

\item{kernel}{String, specifies the kernel function, should be one of \code{"triangular"}, \code{"uniform"} or
\code{"epanechnikov"}.}

\item{bwselect}{String, specifies the method for data-driven bandwidth selection. This option will be
ignored if \code{bw} is provided. Can be (1) \code{"mse-dpi"} (default, mean squared error-optimal
bandwidth selected for each grid point); or (2) \code{"imse-dpi"} (integrated MSE-optimal bandwidth,
common for all grid points); (3) \code{"mse-rot"} (rule-of-thumb bandwidth with Gaussian
reference model); and (4) \code{"imse-rot"} (integrated rule-of-thumb bandwidth with Gaussian
reference model).}

\item{massPoints}{\code{TRUE} (default) or \code{FALSE}, specifies whether point estimates and standard errors
should be adjusted if there are mass points in the data.}

\item{stdVar}{\code{TRUE} (default) or \code{FALSE}, specifies whether the data should be standardized for
bandwidth selection.}

\item{regularize}{\code{TRUE} (default) or \code{FALSE}, specifies whether the bandwidth should be
regularized. When set to \code{TRUE}, the bandwidth is chosen such that the local region includes
at least \code{nLocalMin} observations and at least \code{nUniqueMin} unique observations.}

\item{nLocalMin}{Nonnegative integer, specifies the minimum number of observations in each local neighborhood. This option
will be ignored if \code{regularize=FALSE}. Default is \code{20+p+1}.}

\item{nUniqueMin}{Nonnegative integer, specifies the minimum number of unique observations in each local neighborhood. This option
will be ignored if \code{regularize=FALSE}. Default is \code{20+p+1}.}

\item{Cweights}{Numeric vector, specifies the weights used
for counterfactual distribution construction. Should have the same length as the data.
This option will be ignored if \code{bwselect} is \code{"mse-rot"} or \code{"imse-rot"}.}

\item{Pweights}{Numeric vector, specifies the weights used
in sampling. Should have the same length as the data.
This option will be ignored if \code{bwselect} is \code{"mse-rot"} or \code{"imse-rot"}.}
}
\value{
\item{BW}{A matrix containing (1) \code{grid} (grid point), (2) \code{bw} (bandwidth),
  (3) \code{nh} (number of observations in each local neighborhood), and
  (4) \code{nhu} (number of unique observations in each local neighborhood).}
\item{opt}{A list containing options passed to the function.}
}
\description{
\code{\link{lpbwdensity}} implements the bandwidth selection methods for local
  polynomial based density (and derivatives) estimation proposed and studied
  in Cattaneo, Jansson and Ma (2020a) and Cattaneo, Jansson and Ma (2020b).
  See Cattaneo, Jansson and Ma (2020c) for more implementation details and illustrations.

  Companion command: \code{\link{lpdensity}} for estimation and robust bias-corrected inference.

  Related \code{Stata} and \code{R} packages useful for nonparametric estimation and inference are
  available at \url{https://sites.google.com/site/nppackages/}.
}
\examples{
# Generate a random sample
set.seed(42); X <- rnorm(2000)

# Construct bandwidth
bw1 <- lpbwdensity(X)
summary(bw1)

# Display bandwidths for a subset of grid points
summary(bw1, grid=bw1$BW[4:10, "grid"])
summary(bw1, gridIndex=4:10)

}
\references{
Cattaneo, M. D., M. Jansson, and X. Ma. 2020a. \href{https://sites.google.com/site/nppackages/lpdensity/Cattaneo-Jansson-Ma_2020_JASA.pdf}{Simple Local Polynomial Density Estimators}. \emph{Journal of the American Statistical Association}, forthcoming.

  Cattaneo, M. D., M. Jansson, and X. Ma. 2020b. \href{https://sites.google.com/site/nppackages/lpdensity/Cattaneo-Jansson-Ma_2020_JoE.pdf}{Local Regression Distribution Estimators}. Working paper.

  Cattaneo, M. D., M. Jansson, and X. Ma. 2020c. \href{https://sites.google.com/site/nppackages/lpdensity/Cattaneo-Jansson-Ma_2020_JSS.pdf}{lpdensity: Local Polynomial Density Estimation and Inference}. Working paper.
}
\seealso{
Supported methods: \code{\link{coef.lpbwdensity}}, \code{\link{print.lpbwdensity}}, \code{\link{summary.lpbwdensity}}.
}
\author{
Matias D. Cattaneo, Princeton University. \email{cattaneo@princeton.edu}.

Michael Jansson, University of California Berkeley. \email{mjansson@econ.berkeley.edu}.

Xinwei Ma (maintainer), University of California San Diego. \email{x1ma@ucsd.edu}.
}
