
#' Plot gene tracks
#' 
#' Plot gene annotation tracks from `ensembldb` data using ggplot2.
#' 
#' @details This function is called by [locus_ggplot()]. It can be used to plot
#'   the gene annotation tracks on their own.
#' @param loc Object of class 'locus' generated by [locus()].
#' @param filter_gene_name Vector of gene names to display.
#' @param filter_gene_biotype Vector of gene biotypes to be filtered. Use
#' [ensembldb::listGenebiotypes()] to display possible biotypes. For example, 
#' `ensembldb::listGenebiotypes(EnsDb.Hsapiens.v75)`
#' @param cex.axis Specifies font size for axis numbering.
#' @param cex.lab Specifies font size for axis titles.
#' @param cex.text Font size for gene text.
#' @param showExons Logical whether to show exons or simply show whole gene as a
#'   rectangle. If `showExons = FALSE` colours are specified by `exon_border`
#'   for rectangle border and `gene_col` for the fill colour.
#' @param maxrows Specifies maximum number of rows to display in gene 
#' annotation panel.
#' @param xticks Logical whether x axis ticks and numbers are plotted.
#' @param xlab Title for x axis. Defaults to chromosome `seqname` specified 
#' in `locus`.
#' @param border Logical whether a bounding box is plotted.
#' @param gene_col Colour for gene lines.
#' @param exon_col Fill colour for exons.
#' @param exon_border Border line colour outlining exons (or genes if
#'   `showExons` is `FALSE`). Set to `NA` for no border.
#' @param text_pos Character value of either 'top' or 'left' specifying 
#' placement of gene name labels.
#' @return A ggplot2 object.
#' @examples
#' if(require(EnsDb.Hsapiens.v75)) {
#' data(SLE_gwas_sub)
#' loc <- locus(SLE_gwas_sub, gene = 'IRF5', flank = c(7e4, 2e5), LD = "r2",
#'              ens_db = "EnsDb.Hsapiens.v75")
#' grid::grid.newpage()
#' gg_genetracks(loc)
#' }
#' @importFrom ggplot2 xlab
#' @importFrom grid viewport linesGrob rectGrob textGrob xaxisGrob gList gTree
#'   gpar
#' @importFrom gggrid grid_panel
#' @export

gg_genetracks <- function(loc,
                          filter_gene_name = NULL,
                          filter_gene_biotype = NULL,
                          border = FALSE,
                          cex.axis = 1,
                          cex.lab = 1,
                          cex.text = 0.7,
                          gene_col = ifelse(showExons, 'blue4', 'skyblue'),
                          exon_col = 'blue4',
                          exon_border = 'blue4',
                          showExons = TRUE,
                          maxrows = NULL,
                          text_pos = 'top',
                          xticks = TRUE,
                          xlab = NULL) {
  if (!inherits(loc, "locus")) stop("Object of class 'locus' required")
  g <- genetracks_grob(loc,
                       filter_gene_name,
                       filter_gene_biotype,
                       border,
                       cex.text,
                       gene_col,
                       exon_col,
                       exon_border,
                       showExons,
                       maxrows,
                       text_pos)
  if (is.null(xlab) & xticks) xlab <- paste("Chromosome", loc$seqname, "(Mb)")
  
  g2 <- ggplot(data.frame(x = NA),
               aes(xmin = loc$xrange[1] / 1e6, xmax = loc$xrange[2] / 1e6)) +
    (if (!is.null(g)) gggrid::grid_panel(g)) +
    xlab(xlab) +
    theme_classic() +
    theme(axis.text = element_text(colour = "black", size = 10 * cex.axis),
          axis.title = element_text(size = 10 * cex.lab),
          axis.line.y = element_blank())
  if (!xticks) {
    g2 <- g2 +
      theme(axis.line.x = element_blank(),
            axis.ticks.x = element_blank(),
            axis.text.x = element_blank())
  }
  g2
}

