#' Generic plot function for b3lmeta object in jarbes.
#'
#' @param x The object generated by the b3lmeta function.
#'
#' @param x.lim   Numeric vector of length 2 specifying the x-axis limits.
#' @param y.lim   Numeric vector of length 2 specifying the y-axis limits.
#' @param x.lab Text with the label of the x-axis.
#' @param y.lab Text with the label of the y-axis.
#' @param title.plot.1 Text for the posterior means by design.
#' @param title.plot.2 Text for the posterior pooled mean.
#'
#'
#' @param ... \dots
#'
#' @import ggplot2
#' @importFrom tidyr pivot_longer
#'
#' @export

plot.b3lmeta = function(x,
                       x.lim = c(-3, 3),
                       y.lim = c(0, 2.7),
                       x.lab = "Treatment Effect: log(OR)",
                       y.lab = "Posterior",
                       title.plot.1 = "Mean Design Components",
                       title.plot.2 = "Three Levels Bayesian Meta-Analysis",
                       ...) {
  object = x

  Design=Type=NULL

  # Group means...

  # Posterior means by design component .................................

  # Names of the designs
  levels.design =  levels(object$data$design)

  mu = object$BUGSoutput$sims.list$mu
  mu = data.frame(mu)

  Ndesign = object$Ndesign
  mu_names = levels.design
  names(mu) = mu_names
  mu.groups = pivot_longer(data=mu, cols=mu_names)

  names(mu.groups) = c("Design", "mu")

  plot1 = ggplot(mu.groups, aes(x = mu, group = Design,
                         fill = Design, lty = Design))+
            geom_density(alpha=.2, lwd = 1) +
            scale_x_continuous(limits = x.lim ) +
            scale_y_continuous(limits = y.lim )+
            xlab(x.lab)+
            ylab(y.lab)+
            geom_vline(aes(xintercept=0), lty = 2)+
            ggtitle(title.plot.1)+
            theme_bw()

  # Posterior: pooled mean and predictive ..........................

  mu.0.wide = data.frame(mu.0 = object$BUGSoutput$sims.list$mu.0,
                       mu.0.new = object$BUGSoutput$sims.list$mu.0.new)

  mu.0.names = c("Pooled", "Predictive")
  names(mu.0.wide) = mu.0.names
  mu.0.long = pivot_longer(data = mu.0.wide, mu.0.names)
  names(mu.0.long) = c("Type", "mu")

  plot2 = ggplot(mu.0.long, aes(x = mu, group = Type,
                                fill = Type, lty = Type))+
          geom_density(alpha=.2, lwd = 1) +
          scale_x_continuous(limits = x.lim ) +
          scale_y_continuous(limits = y.lim )+
          xlab(x.lab)+
          ylab(y.lab)+
          geom_vline(aes(xintercept=0), lty = 2)+
          ggtitle(title.plot.2)+
          theme_bw()


  return(suppressWarnings(
    grid.arrange(plot1, plot2, ncol = 2, nrow = 1)
    ))
}



