% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bilog.R
\name{est_bilog}
\alias{est_bilog}
\title{Run BILOG-MG in batch mode}
\usage{
est_bilog(
  x = NULL,
  model = "3PL",
  target_dir = getwd(),
  analysis_name = "bilog_calibration",
  items = NULL,
  id_var = NULL,
  group_var = NULL,
  logistic = TRUE,
  num_of_alternatives = NULL,
  criterion = 0.01,
  num_of_quadrature = 81,
  max_em_cycles = 100,
  newton = 20,
  reference_group = NULL,
  scoring_options = c("METHOD=1", "NOPRINT"),
  calib_options = c("NORMAL"),
  overwrite = FALSE,
  show_output_on_console = TRUE,
  bilog_exe_folder = file.path("C:/Program Files/BILOGMG")
)
}
\arguments{
\item{x}{Either a \code{data.frame} or \code{matrix} object. When the data
is not necessary, i.e. user only wants to read the BILOG-MG output from
the \code{target_dir}, then this can be set to \code{NULL}.}

\item{model}{The model of the items. The value is one of the
  following:
  \describe{
    \item{\code{"1PL"}}{One-parameter logistic model.}
    \item{\code{"2PL"}}{Two-parameter logistic model.}
    \item{\code{"3PL"}}{Three-parameter logistic model.}
    \item{\code{"CTT"}}{Return only Classical Test theory statistics such as
      p-values, point-biserial and  biserial correlations.}
  }

  The default value is \code{"3PL"}.}

\item{target_dir}{The directory/folder where the BILOG-MG analysis and data
files will be saved. The default value is the current working directory,
i.e. \code{get_wd()}.}

\item{analysis_name}{A short file name that will be used for the data files
created for the analysis.}

\item{items}{A vector of column names or numbers of the \code{x} that
represents the responses. If, in the syntax file, no entry for item
names are desired, then, simply write \code{items = "none"}.}

\item{id_var}{The column name or number that contains individual subject IDs.
If none is provided (i.e. \code{id_var = NULL}), the program will check
whether the data provided has row names.}

\item{group_var}{The column name or number that contains group membership
information if multi-group calibration is desired. Ideally, it grouping
variable is represented by single digit integers. If other type of data
provided, an integer value will automatically assigned to the variables.
The default value is \code{NULL}, where no multi-group analysis will be
performed.}

\item{logistic}{A logical value. If \code{TRUE}, \code{LOGISTIC} keyword will
be added to the BILOG-MG command file which means the calibration will
assume the natural metric of the logistic response function in all
calculations. If \code{FALSE}, the logit is multiplied by D = 1.7 to obtain
the metric of the normal-ogive model. The default value is \code{TRUE}.}

\item{num_of_alternatives}{An integer specifying the maximum number of
  response alternatives in the raw data. \code{1/num_of_alternatives} is
  used by the analysis as automatic starting value for estimating the
  pseudo-guessing parameters.

  The default value is \code{NULL}. In this case, for 3PL, 5 will be used
  and for 1PL and 2PL, 1000 will be used.

  This value will be represented in BILOG-MG control file as:
  \code{NALT = num_of_alternatives}.}

\item{criterion}{Convergence criterion for EM and Newton iterations. The
default value is 0.01.}

\item{num_of_quadrature}{The number of quadrature points in MML estimation.
The default value is 81. This value will be represented in BILOG-MG control
file as: \code{NQPT = num_of_quadrature}. The BILOG-MG default value is
20 if there are more than one group, 10 otherwise.}

\item{max_em_cycles}{An integer (0, 1, ...) representing the maximum number
of EM cycles. This value will be represented in BILOG-MG control file as:
\code{CYCLES = max_em_cycles}.
The default value is 10.}

\item{newton}{An integer (0, 1, ...) representing the number of Gauss-Newton
iterations following EM cycles. This value will be represented in BILOG-MG
control file as: \code{NEWTON = newton}.}

\item{reference_group}{Represent which group's ability distribution will be
  set to mean = 0 and standard deviation = 1. For example, if the value is 1,
  then the group whose code is 1 will have ability distribution with mean 0
  and standard deviation 1. When groups are assumed to coming from a single
  population, set this value to 0.

  The default value is \code{NULL}.

  This value will be represented in BILOG-MG control file as:
  \code{REFERENCE = reference_group}.}

\item{scoring_options}{A string vector of keywords/options that will be added
  to the \code{SCORE} section in BILOG-MG syntax. Set the value of
  \code{scoring_options} to \code{NULL} if scoring of individual examinees is
  not necessary.

  The default value is \code{c("METHOD=1", "NOPRINT")} where scale scores
  will be estimated using Maximum Likelihood estimation and the scoring
  process will not be printed to the R console (if
  \code{show_output_on_console = TRUE}).

  The main option to be added to this vector is \code{"METHOD=n"}.
  Following options are available:

  \describe{
    \item{"METHOD=1"}{Maximum Likelihood (ML)}
    \item{"METHOD=2"}{Expected a Posteriori (EAP)}
    \item{"METHOD=3"}{Maximum a Posteriori (MAP)}
  }

  In addition to \code{"METHOD=n"} keyword, following keywords can be added:

  \code{"NOPRINT"}: Suppresses the display of the scores on the R console.

  \code{"FIT"}: likelihood ratio chi-square goodness-of-fit statistic for
    each response pattern will be computed.

  \code{"NQPT=(list)"}, \code{"IDIST=n"}, \code{"PMN=(list)"},
  \code{"PSD=(list)"}, \code{"RSCTYPE=n"}, \code{"LOCATION=(list)"},
  \code{"SCALE=(list)"}, \code{"INFO=n"}, \code{"BIWEIGHT"},
  \code{"YCOMMON"}, \code{"POP"}, \code{"MOMENTS"},
  \code{"FILE"}, \code{"READF"}, \code{"REFERENCE=n"}, \code{"NFORMS=n"}

  See BILOG-MG manual for more details about these keywords/options.}

\item{calib_options}{A string vector of keywords/options that will be added
  to the \code{CALIB} section in BILOG-MG syntax in addition to the keywords
  \code{NQPT}, \code{CYCLES}, \code{NEWTON}, \code{CRIT}, \code{REFERENCE}.

  The default value is \code{c("NORMAL")}.

  When \code{"NORMAL"} is included in \code{calib_options}, the prior
  distributions of ability in the population is assumed to have normal
  distribution.

  When \code{"COMMON"} is included in \code{calib_options}, a common value
  for the lower asymptote for all items in the 3PL model will be estimated.

  Following keywords/options can be added to \code{calib_options}:

  \code{"PRINT=n"}, \code{"IDIST=n"}, \code{"PLOT=n"}, \code{"DIAGNOSIS=n"},
  \code{"REFERENCE=n"}, \code{"SELECT=(list)"}, \code{"RIDGE=(a,b,c)"},
  \code{"ACCEL=n"}, \code{"NSD=n"}, \code{"COMMON"}, \code{"EMPIRICAL"},
  \code{"NORMAL"}, \code{"FIXED"}, \code{"TPRIOR"}, \code{"SPRIOR"},
  \code{"GPRIOR"}, \code{"NOTPRIOR"}, \code{"NOSPRIOR"}, \code{"NOGPRIOR"},
  \code{"READPRIOR"}, \code{"NOFLOAT"}, \code{"FLOAT"}, \code{"NOADJUST"},
  \code{"GROUP-PLOT"}, \code{"RASCH"}, \code{"NFULL"}, \code{"CHI=(a,b)"}.

  See BILOG-MG manual for more details about these keywords/options.

  NOTE: Do not add any of the following keywords to \code{calib_options}
  since they will already be included:

  \code{NQPT}, \code{CYCLES}, \code{NEWTON}, \code{CRIT}, \code{REFERENCE}}

\item{overwrite}{If \code{TRUE} and there are already a BILOG-MG analysis
files in the target path with the same name, these file will be
overwritten.}

\item{show_output_on_console}{logical (not NA), indicates whether to capture
the output of the command and show it on the R console. The default value
is \code{TRUE}.}

\item{bilog_exe_folder}{The location of the \code{"blm1.exe"},
\code{"blm2.exe"} and \code{"blm3.exe"} files. The default location is
\code{file.path("C:/Program Files/BILOGMG")}.}
}
\value{
A list of following objects:
  \describe{
    \item{"ip"}{An \code{\link{Itempool-class}} object holding the item
      parameters. This element will not be created when
      \code{model = "CTT"}.}
    \item{"score"}{A data frame object that holds the number of item
      examinee has attempted (\code{tried}), the number of item examinee got
      right (\code{right}), the estimated scores of examinees
      (\code{ability}), the standard errors of ability estimates (\code{se}),
      and the probability of the response string (\code{prob}). This element
      will not be created when \code{model = "CTT"}.}
    \item{"ctt"}{The Classical Test Theory (CTT) stats such as p-value,
      biserial, point-biserial estimated by BILOG-MG. If there are groups,
      then the CTT statistics for groups can be found in
      \code{ctt$group$GROUP-NAME}. Overall statistics for the whole group is
      at \code{ctt$overall}.
      }
    \item{"failed_items"}{A data frame consist of items that cannot be
      estimated.}
    \item{"syntax"}{The syntax file.}
    \item{"converged"}{A logical value indicating whether a model has been
      converged or not. If the value is \code{TRUE}, model has been
      converged. This element will not be created when \code{model = "CTT"}.}
    \item{"neg_2_log_likelihood"}{-2 Log Likelihood value. This value is
      \code{NULL}, when model does not converge. This element will not be
      created when \code{model = "CTT"}.}
    \item{"input"}{A list object that stores the arguments that are passed
      to the function.}
  }
}
\description{
\code{est_bilog} runs BILOG-MG in batch mode or reads BILOG-MG
  output generated by BILOG-MG program. In the first case, this function
  requires BILOG-MG already installed on your computer under
  \code{bilog_exe_folder} directory.

  In the latter case, where appropriate BILOG-MG files are present (i.e.
  \code{"<analysis_name>.PAR"}, \code{"<analysis_name>.PH1"},
  \code{"<analysis_name>.PH2"} and \code{"<analysis_name>.PH3"} files exist)
  and \code{overwrite = FALSE}, there is no need for BILOG-MG program. This
  function can read BILOG-MG output without BILOG-MG program.
}
\examples{
\dontrun{
### Example 1 ###
# Create responses to be used in BILOG-MG estimation
true_ip <- generate_ip(n = 30, model = "2PL")
resp <- sim_resp(true_ip, rnorm(4000))

# The following line will run BILOG-MG, estimate 2PL model and put the
# analysis results under the target directory:
bilog_calib <- est_bilog(x = resp, model = "2PL",
                         target_dir = "C:/Temp/Analysis", overwrite = TRUE)
# Check whether the calibration converged
bilog_calib$converged

# Get the estimated item pool
bilog_calib$ip

# See the BILOG-MG syntax
cat(bilog_calib$syntax)

# See the classical test theory statistics estimated by BILOG-MG:
bilog_calib$ctt

# Get -2LogLikelihood for the model (mainly for model comparison purposes):
bilog_calib$neg_2_log_likelihood


### Example 2 ###
# Get Expected-a-posteriori theta scores:
result <- est_bilog(x = resp, model = "2PL",
                    scoring_options = c("METHOD=2", "NOPRINT"),
                    target_dir = "C:/Temp/Analysis", overwrite = TRUE)


### Example 3 ###
# Multi-group calibration
ip <- generate_ip(n = 35, model = "3PL", D = 1.7)
n_upper <- sample(1200:3000, 1)
n_lower <- sample(1900:2800, 1)
theta_upper <- rnorm(n_upper, 1.5, .25)
theta_lower <- rnorm(n_lower)
resp <- sim_resp(ip = ip, theta = c(theta_lower, theta_upper))
dt <- data.frame(level = c(rep("Lower", n_lower), rep("Upper", n_upper)), resp)

mg_calib <- est_bilog(x = dt, model = "3PL",
                      group_var = "level",
                      reference_group = "Lower",
                      items = 2:ncol(dt), # Exclude the 'group' column
                      num_of_alternatives = 5,
                      # Use MAP ability estimation.
                      # "FIT": calculate GOF for response patterns
                      scoring_options = c("METHOD=3", "NOPRINT", "FIT"),
                      target_dir = "C:/Temp/Analysis", overwrite = TRUE,
                      show_output_on_console = FALSE)
# Estimated item pool
mg_calib$ip
# Print group means
mg_calib$group_info
# Check Convergence
mg_calib$converged
# Print estimated scores of first five examinees
head(mg_calib$score)


### Example 4 ###
# When user wants to read BILOG-MG output saved in the directory "Analysis/"
# with file names "my_analysis", use the following syntax:
# (The following code does not require an installed BILOG-MG program.)
result <- est_bilog(target_dir = file.path("Analysis/"), model = "3PL",
                    analysis_name = "my_analysis", overwrite = FALSE)

}
}
\author{
Emre Gonulates
}
