% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sampling.R
\name{sample.lgcp}
\alias{sample.lgcp}
\title{Sample from an inhomogeneous Poisson process}
\usage{
sample.lgcp(mesh, loglambda, strategy = "rectangle", R = 6371,
  samplers = NULL)
}
\arguments{
\item{mesh}{An \link[INLA]{inla.mesh} object}

\item{loglambda}{A vector of log intensities at the mesh vertices (for higher order basis functions, e.g. 
for \code{inla.mesh.1d} meshes, \code{loglambda} should be given as \code{mesh$m} basis 
function weights rather than the values at the \code{mesh$n} vertices)}

\item{strategy}{Only relevant for 2D meshes. Use "rectangle" for flat 2D meshes and "spherical" or 
"sliced-spherical" for spherical meshes.}

\item{R}{Numerical value only applicable to spherical meshes. This sets the radius of the sphere 
approximated by the mesh. The default is 6371, which is approximately the radius of the earth 
in kilometers.}

\item{samplers}{A SpatialPolygonsDataFrame. Simulated points that fall outside these polygons are discarded.}
}
\value{
A \code{data.frame} (1D case) or SpatialPoints (2D case) object of point locations.
}
\description{
This function provides point samples from one- and two-dimensional inhomogeneous Poisson processes. The
log intensity has to be provided via its values at the nodes of an \code{inla.mesh.1d} or 
\code{inla.mesh} object. In between mesh nodes the log intensity is assumed to be linear.
}
\details{
For 2D processes on a sphere the \code{R} parameter can be used to adjust to sphere's radius implied by 
the mesh. If the intensity is very high the standard \code{strategy} "spherical" can cause memory issues. 
Using the "sliced-spherical" strategy can help in this case.
}
\examples{
\donttest{
library(INLA)
vertices = seq(0, 3, by = 0.1)
mesh = inla.mesh.1d(vertices)
loglambda = 5-0.5*vertices
pts = sample.lgcp(mesh, loglambda)
pts$y = 0
plot(vertices, exp(loglambda), type = "l", ylim = c(0,150))
points(pts, pch = "|" )
}

\donttest{ 
data("gorillas")
pts = sample.lgcp(gorillas$mesh, rep(1.5, gorillas$mesh$n))
ggplot() + gg(gorillas$mesh) + gg(pts)
}
}
\author{
Daniel Simpson <\email{dp.simpson@gmail.com}> (base algorithm), 
Fabian E. Bachl <\email{bachlfab@gmail.com}> (inclusion in inlabru, sliced spherical sampling)
}
