% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hexify.R
\name{hexify}
\alias{hexify}
\title{Assign hexagonal DGGS cell IDs to geographic points}
\usage{
hexify(
  data,
  grid = NULL,
  lon = "lon",
  lat = "lat",
  area_km2 = NULL,
  diagonal = NULL,
  resolution = NULL,
  aperture = 3,
  resround = "nearest"
)
}
\arguments{
\item{data}{A data.frame or sf object containing coordinates}

\item{grid}{A HexGridInfo object from \code{hex_grid()}. If provided, overrides
area_km2, resolution, and aperture parameters.}

\item{lon}{Column name for longitude (ignored if data is sf)}

\item{lat}{Column name for latitude (ignored if data is sf)}

\item{area_km2}{Target cell area in km^2 (mutually exclusive with diagonal).}

\item{diagonal}{Target cell diagonal (long diagonal) in km}

\item{resolution}{Grid resolution (0-30). Alternative to area_km2.}

\item{aperture}{Grid aperture: 3, 4, 7, or "4/3" for mixed (default 3)}

\item{resround}{How to round resolution: "nearest", "up", or "down"}
}
\value{
A HexData object containing:
\itemize{
\item \code{data}: The original input data (unchanged)
\item \code{grid}: The HexGridInfo specification
\item \code{cell_id}: Numeric vector of cell IDs for each row
\item \code{cell_center}: Matrix of cell center coordinates (lon, lat)
}

Use \code{as.data.frame(result)} to extract the original data.
Use \code{cells(result)} to get unique cell IDs.
Use \code{result@cell_id} to get all cell IDs.
Use \code{result@cell_center} to get cell center coordinates.
}
\description{
Takes a data.frame or sf object with geographic coordinates and returns
a HexData object that stores the original data plus cell assignments.
The original data is preserved unchanged; cell IDs and centers are stored
in separate slots.
}
\details{
For sf objects, coordinates are automatically extracted and transformed to
'WGS84' (EPSG:4326) if needed. The geometry column is preserved.

Either \code{area_km2} (or \code{area}), \code{diagonal}, or \code{resolution}
must be provided unless a \code{grid} object is supplied.

The HexData return type (default) stores the grid specification so downstream
functions like \code{plot()}, \code{hexify_cell_to_sf()}, etc. don't need
grid parameters repeated.
}
\section{Grid Specification}{

You can create a grid specification once and reuse it:
\preformatted{
grid <- hex_grid(area_km2 = 1000)
result1 <- hexify(df1, grid = grid)
result2 <- hexify(df2, grid = grid)
}
}

\examples{
# Simple data.frame
df <- data.frame(
  site = c("Vienna", "Paris", "Madrid"),
  lon = c(16.37, 2.35, -3.70),
  lat = c(48.21, 48.86, 40.42)
)

# New recommended workflow: use grid object
grid <- hex_grid(area_km2 = 1000)
result <- hexify(df, grid = grid, lon = "lon", lat = "lat")
print(result)  # Shows grid info
plot(result)   # Plot with default styling

# Direct area specification (grid created internally)
result <- hexify(df, lon = "lon", lat = "lat", area_km2 = 1000)

# Extract plain data.frame
df_result <- as.data.frame(result)

# With sf object (any CRS)
library(sf)
pts <- st_as_sf(df, coords = c("lon", "lat"), crs = 4326)
result_sf <- hexify(pts, area_km2 = 1000)

# Different apertures
result_ap4 <- hexify(df, lon = "lon", lat = "lat", area_km2 = 1000, aperture = 4)

# Mixed aperture (ISEA43H)
result_mixed <- hexify(df, lon = "lon", lat = "lat", area_km2 = 1000, aperture = "4/3")
}
\seealso{
\code{\link{hex_grid}} for grid specification,
\code{\link{HexData-class}} for return object details,
\code{\link{as_sf}} for converting to sf

Other hexify main: 
\code{\link{hexify_grid}()}
}
\concept{hexify main}
