/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Fmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5Iprivate.h"  

#include "H5VLnative_private.h" 

#ifndef H5_NO_DEPRECATED_SYMBOLS

herr_t
H5Fget_info1(hid_t obj_id, H5F_info1_t *finfo )
{
    H5VL_object_t                   *vol_obj = NULL;
    H5VL_optional_args_t             vol_cb_args;   
    H5VL_native_file_optional_args_t file_opt_args; 
    H5I_type_t                       type;
    H5F_info2_t                      finfo2;              
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!finfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no info struct");

    
    type = H5I_get_type(obj_id);
    if (H5I_FILE != type && H5I_GROUP != type && H5I_DATATYPE != type && H5I_DATASET != type &&
        H5I_ATTR != type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");

    
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid object identifier");

    
    file_opt_args.get_info.type  = type;
    file_opt_args.get_info.finfo = &finfo2;
    vol_cb_args.op_type          = H5VL_NATIVE_FILE_GET_INFO;
    vol_cb_args.args             = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to retrieve file info");

    
    finfo->super_ext_size = finfo2.super.super_ext_size;
    finfo->sohm.hdr_size  = finfo2.sohm.hdr_size;
    finfo->sohm.msgs_info = finfo2.sohm.msgs_info;

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5Fis_hdf5(const char *name)
{
    H5VL_file_specific_args_t vol_cb_args;           
    bool                      is_accessible = false; 
    htri_t                    ret_value;             

    FUNC_ENTER_API((-1))

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, (-1), "no file name specified");

    
    vol_cb_args.op_type                       = H5VL_FILE_IS_ACCESSIBLE;
    vol_cb_args.args.is_accessible.filename   = name;
    vol_cb_args.args.is_accessible.fapl_id    = H5P_FILE_ACCESS_DEFAULT;
    vol_cb_args.args.is_accessible.accessible = &is_accessible;

    
    if (H5VL_file_specific(NULL, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_NOTHDF5, (-1), "unable to determine if file is accessible as HDF5");

    
    ret_value = (htri_t)is_accessible;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fset_latest_format(hid_t file_id, bool latest_format)
{
    H5VL_object_t                   *vol_obj;                       
    H5VL_optional_args_t             vol_cb_args;                   
    H5VL_native_file_optional_args_t file_opt_args;                 
    H5F_libver_t                     low       = H5F_LIBVER_LATEST; 
    herr_t                           ret_value = SUCCEED;           

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "not a file ID");

    
    if (H5CX_set_loc(file_id) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    if (!latest_format)
        low = H5F_LIBVER_EARLIEST;

    
    file_opt_args.set_libver_bounds.low  = low;
    file_opt_args.set_libver_bounds.high = H5F_LIBVER_LATEST;
    vol_cb_args.op_type                  = H5VL_NATIVE_FILE_SET_LIBVER_BOUNDS;
    vol_cb_args.args                     = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set library version bounds");

done:
    FUNC_LEAVE_API(ret_value)
} 
#endif 
