/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Fmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FDprivate.h" 
#include "H5FLprivate.h" 

typedef struct H5F_sfile_node_t {
    H5F_shared_t            *shared; 
    struct H5F_sfile_node_t *next;   
} H5F_sfile_node_t;

H5FL_DEFINE_STATIC(H5F_sfile_node_t);

static H5F_sfile_node_t *H5F_sfile_head_s = NULL;

void
H5F_sfile_assert_num(unsigned H5_ATTR_NDEBUG_UNUSED n)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
#ifndef NDEBUG
    if (n == 0) {
        assert(H5F_sfile_head_s == NULL);
    }
    else {
        unsigned          count; 
        H5F_sfile_node_t *curr;  

        
        curr  = H5F_sfile_head_s;
        count = 0;
        while (curr) {
            
            count++;

            
            curr = curr->next;
        }

        assert(count == n);
    }
#endif

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5F__sfile_add(H5F_shared_t *shared)
{
    H5F_sfile_node_t *new_shared;          
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(shared);

    
    if (NULL == (new_shared = H5FL_CALLOC(H5F_sfile_node_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    new_shared->shared = shared;

    
    new_shared->next = H5F_sfile_head_s;
    H5F_sfile_head_s = new_shared;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5F_shared_t *
H5F__sfile_search(H5FD_t *lf)
{
    H5F_sfile_node_t *curr;             
    H5F_shared_t     *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(lf);

    
    curr = H5F_sfile_head_s;
    while (curr) {
        
        if (0 == H5FD_cmp(curr->shared->lf, lf))
            HGOTO_DONE(curr->shared);

        
        curr = curr->next;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__sfile_remove(H5F_shared_t *shared)
{
    H5F_sfile_node_t *curr;                
    H5F_sfile_node_t *last;                
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(shared);

    
    last = NULL;
    curr = H5F_sfile_head_s;
    while (curr && curr->shared != shared) {
        
        last = curr;
        curr = curr->next;
    } 

    
    if (curr == NULL)
        HGOTO_ERROR(H5E_FILE, H5E_NOTFOUND, FAIL, "can't find shared file info");

    
    if (last != NULL)
        
        last->next = curr->next;
    else
        
        H5F_sfile_head_s = curr->next;

    
    
    curr = H5FL_FREE(H5F_sfile_node_t, curr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
