% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glms.R
\name{BoxTidwell.lm}
\alias{BoxTidwell.lm}
\title{Box-Tidwell transformations in Normal Linear Models}
\usage{
\method{BoxTidwell}{lm}(
  object,
  transf,
  epsilon = 1e-04,
  maxiter = 30,
  trace = FALSE,
  digits = getOption("digits") - 2,
  ...
)
}
\arguments{
\item{object}{an object of the class \emph{lm}.}

\item{transf}{an one-sided formula giving the quantitative predictors that are candidates for transformation.}

\item{epsilon}{an (optional) numerical value. If the maximum relative change in coefficients is less than
\emph{epsilon}, then convergence is declared. By default, \emph{epsilon} is set to be 0.0001.}

\item{maxiter}{an (optional) positive integer value indicating the maximum number of iterations. By default,
\emph{maxiter} is set to be 30.}

\item{trace}{an (optional) logical indicating if should the record of iterations be printed. By default,
\emph{trace} is set to be FALSE.}

\item{digits}{an (optional) integer value indicating the number of decimal places to be used.}

\item{...}{further arguments passed to or from other methods.}
}
\value{
a list list with components including
\tabular{ll}{
\code{marginal} \tab a matrix with estimates and standard errors of the estimated powers, as well as the statistic
                      and the p-value of the Wald test to assess the hypothesis \eqn{H_0:\tau=1} versus \eqn{H_1:\tau\neq 1},\cr
\tab \cr
\code{omnibus} \tab a matrix with the statistic and the p-value of the Wald test for null hypothesis that all powers
                     are 1,\cr
}
}
\description{
Computes the Box-Tidwell power transformations of the predictors in a normal linear model.
}
\examples{
###### Example 1: Hill races in Scotland
data(races)
fit1 <- lm(rtime ~ distance + cclimb, data=races)
AIC(fit1)
BoxTidwell(fit1, transf= ~ distance + cclimb)
fit1 <- update(fit1,formula=rtime ~ distance + I(cclimb^2))
AIC(fit1)

###### Example 2: Gasoline yield
fit2 <- lm(mpg ~ hp + wt + am, data=mtcars)
AIC(fit2)
BoxTidwell(fit2, transf= ~ hp + wt)
fit2 <- update(fit2,formula=mpg ~ log(hp) + log(wt) + am)
AIC(fit2)

###### Example 3: New York Air Quality Measurements
fit3 <- lm(log(Ozone) ~ Solar.R + Wind + Temp, data=airquality)
AIC(fit3)
BoxTidwell(fit3, transf= ~ Solar.R + Wind + Temp)
fit3 <- update(fit3,formula=log(Ozone) ~ log(Solar.R) + Wind + Temp)
AIC(fit3)

###### Example 4: Heat capacity of hydrobromic acid
data(heatcap,package="GLMsData")
fit4 <- lm(Cp ~ Temp, data=heatcap)
AIC(fit4)
BoxTidwell(fit4, transf= ~ Temp)
fit4 <- update(fit4,formula=Cp ~ I(Temp^5))
AIC(fit4)

###### Example 5: Age and Eye Lens Weight of Rabbits in Australia
data(rabbits)
fit5 <- lm(log(wlens) ~ age, data=rabbits)
AIC(fit5)
BoxTidwell(fit5, transf= ~ age)
fit5 <- update(fit5,formula=log(wlens) ~ I(age^(-1/3)))
AIC(fit5)

}
\references{
Box G.E.P., Tidwell P.W. (1962) Transformation of the independent variables. \emph{Technometrics} 4, 531-550.

Fox J. (2016) \emph{Applied Regression Analysis and Generalized Linear Models}, Third Edition. Sage.
}
\seealso{
\link{BoxTidwell.glm}
}
