#' Draw Lines on a ggplot Object from Line Data
#' @description
#' This function overlays lines or arrows to a ggplot object based on line data. It supports straight lines, curved lines, gradient color transitions, and one-way or two-way arrows. The data can come from a CSV file generated by the ggsem Shiny app or custom input.
#' @param p A ggplot object to which the lines will be added.
#' @param lines_data A data frame containing line information. The expected columns include:
#' \itemize{
#'   \item \code{x_start}, \code{y_start}: Starting coordinates of the line.
#'   \item \code{x_end}, \code{y_end}: Ending coordinates of the line.
#'   \item \code{ctrl_x}, \code{ctrl_y}: Control points for curved lines (optional).
#'   \item \code{type}: Type of line (\code{"Straight Line"}, \code{"Curved Line"}, \code{"Straight Arrow"}, or \code{"Curved Arrow"}).
#'   \item \code{color}: Start color of the line (hexadecimal color code).
#'   \item \code{end_color}: End color of the line for gradients (hexadecimal color code).
#'   \item \code{color_type}: \code{"Gradient"} for gradient lines, or \code{"Single"} for solid-colored lines.
#'   \item \code{gradient_position}: Position of the gradient transition along the line (numeric, 0 to 1).
#'   \item \code{width}: Width of the line (numeric).
#'   \item \code{alpha}: Transparency of the line (numeric, 0 to 1).
#'   \item \code{arrow}: Logical, whether an arrowhead is used.
#'   \item \code{arrow_type}: Type of arrow (\code{"open"}, \code{"closed"}, etc.).
#'   \item \code{arrow_size}: Size of the arrowhead.
#'   \item \code{two_way}: Logical, whether the line has two arrowheads (bidirectional).
#'   \item \code{line_style}: Line style (\code{"solid"}, \code{"dashed"}, or \code{"dotted"}).
#' }
#' @param zoom_level Numeric. Adjusts the size of line widths and arrowheads relative to the plot. Default is \code{1}.
#' @param n Integer. Number of points for interpolation in gradient or curved lines. Default is \code{100}.
#' @return
#' A ggplot object with the specified lines or arrows added.
#' @export
#'
#' @examples
#' library(ggplot2)
#'
#' lines_df <- data.frame(
#' x_start = 2, y_start = -2, x_end = 10, y_end = -2, ctrl_x = NA, ctrl_y = NA,
#' type = 'Straight Line', color = '#000000', end_color = '#cc3d3d', color_type = 'Gradient',
#' gradient_position = 0.35, width = 1.5, alpha = 1, arrow = FALSE,
#' arrow_type = NA, arrow_size = NA, two_way = FALSE, lavaan = FALSE,
#' network = FALSE, line_style = 'solid', locked = FALSE
#' )
#'
#' p <- ggplot()
#'
#' draw_lines(p, lines_data = lines_df, zoom_level = 1.2, n = 400)
#'
draw_lines <- function(p, lines_data, zoom_level = 1, n = n) {
  if (nrow(lines_data) > 0) {
    lines_data$color <- sapply(lines_data$color, valid_hex)
    lines_data$end_color <- sapply(lines_data$end_color, valid_hex)
    lines_data$line_style <- sapply(lines_data$line_style, valid_line_style)
    lines_data$alpha <- sapply(lines_data$alpha, valid_alpha)
    lines_data$gradient_position <- sapply(lines_data$gradient_position, valid_gradient_position)
    lines_data$type <- sapply(lines_data$type, valid_type)

    for (i in 1:nrow(lines_data)) {
      line_type <- lines_data$type[i]
      start_color <- lines_data$color[i]
      end_color <- if (length(lines_data$color_type[i]) > 0 && lines_data$color_type[i] == "Gradient") {
        lines_data$end_color[i]
      } else {
        start_color
      }
      gradient_position <- if (!is.null(lines_data$gradient_position[i]) && length(lines_data$gradient_position[i]) > 0) {
        lines_data$gradient_position[i]
      } else {
        1
      }

      if (!is.null(line_type) && length(line_type) > 0) {
        adjusted_line_width <- lines_data$width[i] / zoom_level
        adjusted_arrow_size <- if (!is.na(lines_data$arrow_size[i])) lines_data$arrow_size[i] / zoom_level else NA


        if (line_type == "Straight Line" || line_type == "Straight Arrow" || line_type == "Auto-generated") {
          if (!is.null(lines_data$x_start[i]) && !is.null(lines_data$x_end[i])) {
            if (lines_data$color_type[i] == "Gradient") {
              straight_points <- interpolate_points(
                x_start = lines_data$x_start[i], y_start = lines_data$y_start[i],
                x_end = lines_data$x_end[i], y_end = lines_data$y_end[i],
                n = n
              )

              n_points <- nrow(straight_points)
              split_index <- round(gradient_position * n_points)

              color_interpolator <- colorRampPalette(c(start_color, end_color))
              intermediate_color <- color_interpolator(3)[2]

              gradient_colors_start <- colorRampPalette(c(start_color, intermediate_color))(split_index)
              gradient_colors_end <- colorRampPalette(c(intermediate_color, end_color))(n_points - split_index + 1)


              # Draw the line segment by segment with interpolated colors
              for (j in 1:(split_index - 1)) {
                p <- p + annotate("segment",
                                  x = straight_points$x[j], y = straight_points$y[j],
                                  xend = straight_points$x[j + 1], yend = straight_points$y[j + 1],
                                  color = gradient_colors_start[j],
                                  size = adjusted_line_width, alpha = lines_data$alpha[i]
                )
              }

              # Draw the second segment with the end color gradient
              for (j in split_index:(n_points - 1)) {
                p <- p + annotate("segment",
                                  x = straight_points$x[j], y = straight_points$y[j],
                                  xend = straight_points$x[j + 1], yend = straight_points$y[j + 1],
                                  color = gradient_colors_end[j - split_index + 1],
                                  size = adjusted_line_width, alpha = lines_data$alpha[i]
                )
              }
            } else {
              # For single-color straight lines, use annotate("segment")
              p <- p + annotate("segment",
                                x = lines_data$x_start[i], y = lines_data$y_start[i],
                                xend = lines_data$x_end[i], yend = lines_data$y_end[i],
                                color = start_color,
                                size = adjusted_line_width, alpha = lines_data$alpha[i],
                                linetype = lines_data$line_style[i]
              )
            }

            # Add arrowhead if necessary
            arrow_type <- lines_data$arrow_type[i]
            if (!is.null(arrow_type) && !is.na(adjusted_arrow_size)) {
              offset_factor <- 0.01

              dx <- lines_data$x_end[i] - lines_data$x_start[i]
              dy <- lines_data$y_end[i] - lines_data$y_start[i]
              norm <- sqrt(dx^2 + dy^2)


              x_adjust_start <- lines_data$x_start[i] + offset_factor * dx / norm
              y_adjust_start <- lines_data$y_start[i] + offset_factor * dy / norm

              x_adjust_end <- lines_data$x_end[i] - offset_factor * dx / norm
              y_adjust_end <- lines_data$y_end[i] - offset_factor * dy / norm

              if (lines_data$two_way[i]) {
                # Two-way arrow logic
                p <- p + annotate("segment",
                                  x = x_adjust_start, y = y_adjust_start,
                                  xend = lines_data$x_start[i], yend = lines_data$y_start[i],
                                  size = adjusted_line_width, alpha = lines_data$alpha[i],
                                  arrow = arrow(type = arrow_type, length = unit(adjusted_arrow_size, "inches")),
                                  color = start_color
                ) +
                  annotate("segment",
                           x = x_adjust_end, y = y_adjust_end,
                           xend = lines_data$x_end[i], yend = lines_data$y_end[i],
                           size = adjusted_line_width, alpha = lines_data$alpha[i],
                           arrow = arrow(type = arrow_type, length = unit(adjusted_arrow_size, "inches")),
                           color = end_color
                  )
              } else {
                # One-way arrow logic
                p <- p + annotate("segment",
                                  x = x_adjust_end, y = y_adjust_end,
                                  xend = lines_data$x_end[i], yend = lines_data$y_end[i],
                                  size = adjusted_line_width, alpha = lines_data$alpha[i],
                                  arrow = arrow(type = arrow_type, length = unit(adjusted_arrow_size, "inches")),
                                  color = end_color
                ) # Use solid end color for arrowhead
              }
            }
          }
        }

        # For curved lines and arrows
        if (line_type == "Curved Line" || line_type == "Curved Arrow") {
          if (!is.null(lines_data$ctrl_x[i]) && !is.null(lines_data$ctrl_y[i])) {
            # Use create_bezier_curve for curved lines
            bezier_points <- create_bezier_curve(
              x_start = lines_data$x_start[i], y_start = lines_data$y_start[i],
              x_end = lines_data$x_end[i], y_end = lines_data$y_end[i],
              ctrl_x = lines_data$ctrl_x[i], ctrl_y = lines_data$ctrl_y[i],
              n_points = n
            )

            if (lines_data$color_type[i] == "Gradient") {
              # Interpolate gradient colors along the Bezier curve
              n_points <- nrow(bezier_points)
              split_index <- round(gradient_position * n_points)
              color_interpolator <- colorRampPalette(c(start_color, end_color))
              intermediate_color <- color_interpolator(3)[2]

              gradient_colors_start <- colorRampPalette(c(start_color, intermediate_color))(split_index)
              gradient_colors_end <- colorRampPalette(c(intermediate_color, end_color))(n_points - split_index + 1)

              p <- p + annotate("path",
                                x = bezier_points$x,
                                y = bezier_points$y,
                                color = start_color,
                                size = adjusted_line_width,
                                alpha = lines_data$alpha[i],
                                linetype = lines_data$line_style[i]
              )

              for (j in 1:(split_index - 1)) {
                p <- p + annotate("path",
                                  x = bezier_points$x[j:(j + 1)],
                                  y = bezier_points$y[j:(j + 1)],
                                  color = gradient_colors_start[j],
                                  size = adjusted_line_width, alpha = lines_data$alpha[i]
                )
              }

              for (j in split_index:(n_points - 1)) {
                p <- p + annotate("path",
                                  x = bezier_points$x[j:(j + 1)],
                                  y = bezier_points$y[j:(j + 1)],
                                  color = gradient_colors_end[j - split_index + 1],
                                  size = adjusted_line_width, alpha = lines_data$alpha[i]
                )
              }
            } else {
              p <- p + annotate("path",
                                x = bezier_points$x,
                                y = bezier_points$y,
                                color = start_color,
                                size = adjusted_line_width, alpha = lines_data$alpha[i],
                                linetype = lines_data$line_style[i]
              )
            }

            # Add arrowhead for curved lines if necessary
            arrow_type <- lines_data$arrow_type[i]
            if (line_type == "Curved Arrow" && !is.null(arrow_type) && !is.na(adjusted_arrow_size)) {
              if (lines_data$two_way[i]) {
                dx_start <- bezier_points$x[2] - bezier_points$x[1]
                dy_start <- bezier_points$y[2] - bezier_points$y[1]

                dx_end <- bezier_points$x[nrow(bezier_points)] - bezier_points$x[nrow(bezier_points) - 1]
                dy_end <- bezier_points$y[nrow(bezier_points)] - bezier_points$y[nrow(bezier_points) - 1]

                norm_start <- sqrt(dx_start^2 + dy_start^2)
                norm_end <- sqrt(dx_end^2 + dy_end^2)

                p <- p + annotate("segment",
                                  x = bezier_points$x[1], y = bezier_points$y[1],
                                  xend = bezier_points$x[1] - dx_start / norm_start * 1e-5,
                                  yend = bezier_points$y[1] - dy_start / norm_start * 1e-5,
                                  size = adjusted_line_width,
                                  arrow = arrow(type = arrow_type, length = unit(adjusted_arrow_size, "inches")),
                                  color = start_color
                ) +
                  annotate("segment",
                           x = bezier_points$x[nrow(bezier_points)], y = bezier_points$y[nrow(bezier_points)],
                           xend = bezier_points$x[nrow(bezier_points)] + dx_end / norm_end * 1e-5,
                           yend = bezier_points$y[nrow(bezier_points)] + dy_end / norm_end * 1e-5,
                           size = adjusted_line_width,
                           arrow = arrow(type = arrow_type, length = unit(adjusted_arrow_size, "inches")),
                           color = end_color
                  )
              } else {
                dx_end <- bezier_points$x[nrow(bezier_points)] - bezier_points$x[nrow(bezier_points) - 1]
                dy_end <- bezier_points$y[nrow(bezier_points)] - bezier_points$y[nrow(bezier_points) - 1]
                norm_end <- sqrt(dx_end^2 + dy_end^2)

                p <- p + annotate("segment",
                                  x = bezier_points$x[nrow(bezier_points)], y = bezier_points$y[nrow(bezier_points)],
                                  xend = bezier_points$x[nrow(bezier_points)] + dx_end / norm_end * 1e-5,
                                  yend = bezier_points$y[nrow(bezier_points)] + dy_end / norm_end * 1e-5,
                                  size = adjusted_line_width,
                                  arrow = arrow(type = arrow_type, length = unit(adjusted_arrow_size, "inches")),
                                  color = end_color
                )
              }
            }
          }
        }
      }
    }
  }
  return(p)
}
